#!/usr/bin/env ruby
#
# Samizdat publish message form
# 
#   Copyright (c) 2002-2003 Dmitry Borodaenko <angdraug@debian.org>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU General Public License version 2 or later.
#

require 'ftools'
require 'samizdat'

session = Session.new

session.out() do |t|
    session.id or raise AuthError, 'You should be logged in to post messages'

    title, content, format, parent =
        session.params %w[title content format parent]

    format = nil unless config['format'].values.flatten.include? format
    inline = (format.nil? or config['format']['inline'].include? format)

    file = session['file']
    file = file[0] if file.class == Array   # CGI#[] changed in Ruby 1.8

    if file and file.size > 0 then   # content uploaded from file
        content = nil
        format.nil? and raise UserError,
            "It is not possible to upload a file without specifying format"
        # todo: fine-grained size limits
        file.size > config['limit']['content'] and raise UserError,
"Uploaded file is larger than #{config['limit']['content']} bytes limit"

        location = Resource.new( session
            ).content_location('upload', format, session.login)
        upload = session.filename(location)
        upload.untaint   # todo: explore possible security issues!

        if (file.kind_of? StringIO or file.kind_of? Tempfile) and
        not session.has_key? 'confirm' then   # new upload
            # todo: autodetect format
            file.content_type.strip != format and raise UserError,
"Detected content-type #{file.content_type.strip} when #{format} was expected"
            if inline then
                content = file.read
                file = nil
            else
                config['site']['content'].nil? and raise UserError,
                    "Multimedia upload is disabled on this site"
                File.makedirs(File.dirname(upload)) unless
                    File.exists?(File.dirname(upload))
                if file.kind_of? Tempfile then   # copy large files directly
                    file = File.syscopy(file.path, upload)
                else   # StringIO
                    File.open(upload, 'w') {|f| f.write(file.read) }
                    file = true
                end
            end

        elsif file.kind_of? String and session.has_key? 'confirm' then
            inline and raise UserError, 'Unexpected inline upload confirm'
            file = nil unless File.exists?(upload)
        else
            raise UserError, 'Unexpected upload state'
        end
    else
        raise UserError, "You should upload #{format} content from file" if
            not inline
        file = nil
    end   # at this point, file is true and content is nil if upload is ready

    if title and (content or file) then
        if session.has_key? 'confirm' then   # write message into the database
            # todo: detect duplicates
            id = nil   # scope fix
            db.transaction do |db|
                # todo: translate into RDF merge
                db.do 'INSERT INTO Message (creator, title, parent, thread,
                    format, content) VALUES (?, ?, ?, ?, ?, ?)', session.id,
                    title, parent, 0, format, content
                id, = db.select_one "SELECT currval('Resource_id_seq')"
                if parent then
                    db.do "UPDATE Message SET thread = 
                        (SELECT thread FROM Message WHERE id = ?)
                        WHERE id = ?", parent, id
                else   # new thread
                    db.do "UPDATE Message SET thread = ? WHERE id = ?", id, id
                end
                if file then
                    # todo: inject file into p2p net
                    File.rename(upload, session.filename(
                        Resource.new( session
                        ).content_location(id, format, session.login)
                    ).untaint)
                end
            end   # transaction

            # Pingback client
            if format == 'text/uri-list' then
                # attempt Pingback on the first URI
                catch :fail do
                    begin
                        require 'uri'
                        require 'net/http'
                        require 'xmlrpc/client'
                    rescue LoadError
                        throw :fail
                    end
                    content =~ URI::ABS_URI or raise UserError,
                        "text/uri-list should contain at least one absolute URI"
                    uri, scheme = $&, $1
                    throw :fail unless scheme =~ /^http/
                    response = Net::HTTP.get_response(URI.parse(uri.untaint))
                    throw :fail unless response.kind_of? Net::HTTPSuccess
                    pingback = response['x-pingback']
                    if pingback.nil? then
                        throw :fail unless response.body =~
                            %r{<link rel="pingback" href="([^"]+)" ?/?>}
                        pingback = $1
                        pingback.gsub!(/&amp;/, '&')
                        pingback.gsub!(/&lt;/, '<')
                        pingback.gsub!(/&gt;/, '>')
                        pingback.gsub!(/&quot;/, '"')
                    end
                    throw :fail unless pingback =~ URI::ABS_URI
                    u = URI.parse(pingback.untaint)
                    server = XMLRPC::Client.new(u.host, u.path, u.port)
                    server.call2("pingback.ping", session.base + id.to_s, uri)
                    # discard the result
                    # todo: notify user that ping was registered
                end
            end
            # redirect to published message
            session.out({'status' => 'REDIRECT', 'location' => id.to_s})

        else   # preview message
            content = location if file   # preview upload.ext
            t.page( 'Message Preview',
                t.message(nil, Time::now, session.id, session.full_name,
                    nil, title, format, content, parent) +
                t.form( 'message.rb',
                    [:submit, 'confirm', 'Confirm'],
                    [:hidden, 'parent', parent],
                    [:hidden, 'title', title],
                    [:hidden, 'content', file ? nil : content],
                    [:hidden, 'file', file],
                    [:hidden, 'format', format]
                ) + "<p>Press 'Back' button to change the message.</p>"
            )
        end

    else   # edit message
        t.page('New Message', t.form('message.rb',
            [:label, 'title', 'Title'], [:text, 'title', title],
            [:label, 'content', 'Content'], [:textarea, 'content', content],
            [:label, 'file', 'Upload content from file'], [:file, 'file', file],
            [:label, 'format', 'Format'], [:select, 'format', ['Default'] +
                config['format']['inline'].to_a +
                config['format']['image'].to_a +
                config['format']['other'].to_a, format],
            [:label], [:submit, 'preview', 'Preview'],
            [:hidden, 'parent', parent]))
    end
end
