#!/usr/bin/env ruby

require 'singleton'
require 'time'
require 'dbi'
require 'yaml'
require 'graffiti'

class BsbmConfig
  include Singleton

  def initialize
    @config = File.open('samizdat-bsbm/bsbm.yaml') {|f| YAML.load(f) }
  end

  attr_reader :config
end

def config
  BsbmConfig.instance.config
end

class Store
  include Singleton

  def initialize
    @db = DBI.connect(
      config['db']['dsn'],
      config['db']['user'],
      config['db']['password'])
    @db['AutoCommit'] = false
    @store = Graffiti::Store.new(@db, config)
  end

  attr_reader :db, :store
end

def db
  Store.instance.db
end

def store
  Store.instance.store
end


class RandomParameterGenerator
  include Singleton

  class RandomId
    def initialize(table)
      @min, = db.select_one("SELECT min(id) FROM #{table}")
      @max, = db.select_one("SELECT max(id) FROM #{table}")
    end

    def generate
      @min + rand(@max - @min)
    end
  end

  class RandomFromPool
    def generate
      @pool[ rand(@pool.size) ]
    end
  end

  class RandomCountry < RandomFromPool
    def initialize
      @pool = db.select_all("SELECT DISTINCT country FROM person")
      @pool.collect! {|c,| c }
    end
  end

  class RandomWord < RandomFromPool
    def initialize
      @pool = File.open('titlewords.txt') {|f| f.readlines }
      @pool.collect! {|w| w.chomp }
    end
  end

  class RandomNumber
    def generate
      rand(499) + 1
    end
  end

  class CurrentDate
    def generate
      @@current_date ||= Time.parse('2008-01-01')
    end
  end

  def initialize
    @generators = {
      'ProductPropertyNumericValue' => RandomNumber.new,
      'ProductFeatureURI' => RandomId.new('productfeature'),
      'ProductTypeURI' => RandomId.new('producttype'),
      'CurrentDate' => CurrentDate.new,
      'Dictionary1' => RandomWord.new,
      'ProductURI' => RandomId.new('product'),
      'ReviewURI' => RandomId.new('review'),
      'CountryURI' => RandomCountry.new,
      'OfferURI' => RandomId.new('offer')
    }
  end

  def generate(type)
    @generators.has_key?(type) or raise "Unknown parameter type: " + type.inspect
    @generators[type].generate
  end
end


class TimeLog
  include Singleton

  def initialize
    reset
  end

  def reset
    @log = {}
    @total_count = {}
    @total_time = {}
  end

  def record(key, time = nil)
    @log[key] ||= []
    if time
      @log[key].push(time)
    elsif block_given?
      start = Time.now
      yield
      @log[key].push(Time.now - start)
    end
  end

  def flush
    @log.each do |key, times|
      @total_count[key] ||= 0
      @total_count[key] += times.size
      @total_time[key] ||= 0.0
      @total_time[key] += times.inject(:+)
    end
    @log = {}
  end

  def print_summary
    flush
    qmph = nil
    total = nil
    queries = []
    @total_count.keys.sort.each do |key|
      count = @total_count[key]
      time = @total_time[key]
      case key
      when 'querymix'
        qmph = sprintf("Query Mixes per Hour: %d", count * 3600 / time)
      when 'total'
        total = sprintf("Total Run Time: %02d:%02d:%05.2f", (time / 3600).to_i,
                        (time % 3600 / 60).to_i, (time % 60).to_f)
      when /query(\d+)/
        queries[$1.to_i] = sprintf("Query %2s: %7.2f per second", $1, count / time)
      end
    end
    puts queries.compact.join("\n"), qmph, total
  end
end


class Query
  def initialize(id)
    @id = id
    @squish = File.open("samizdat-bsbm/query#{@id}.txt").read.untaint

    @params = {}
    desc = File.open("queries/query#{@id}desc.txt").read.untaint
    desc.scan(/([^\s=]+)=([^\s=]+)/) do |name, type|
      if 'QueryType' == name
        @query_type = type.downcase.to_sym
      else
        @params[name.to_sym] = type
      end
    end

    sparql = File.open("queries/query#{@id}.txt").read.untaint
    @limit = /LIMIT\s+(\d+)/.match(sparql).to_a[1]
    @offset = /OFFSET\s+(\d+)/.match(sparql).to_a[1]
  end

  def run
    params = {}
    @params.each {|name, type| params[name] = RandomParameterGenerator.instance.generate(type) }
    TimeLog.instance.record("query#{@id}") do
      case @query_type
      when :select
        store.select_all(@squish, @limit, @offset, params)
      when :construct, :describe
        store.select_one(@squish, params)
      end
    end
  end
end


class QueryMix
  def initialize
    @queries = (1..12).collect {|i| Query.new(i) }
    @mix = File.open('querymix.txt') {|f| f.read }.scan(/\d+/).collect {|s| s.to_i - 1 }
  end

  def run
    TimeLog.instance.record('querymix') do
      @mix.each {|i| @queries[i].run }
    end
  end
end


class Drive
  def initialize
    init_log('Store') { Store.instance }
    init_log('RandomParameterGenerator') { RandomParameterGenerator.instance }
    init_log('QueryMix') { @mix = QueryMix.new }

    @qms_per_period = 50
    @timelimit = nil
    @runs = config['bsbm']['runs']
    @warmups = config['bsbm']['warmups']
  end

  def run
    period_count = 1
    runs_count = 0
    start = Time.now

    while runs_count < @warmups + @runs and (@timelimit.nil? or Time.now - start < @timelimit)
      @mix.run

      runs_count += 1
      if runs_count == @warmups
        TimeLog.instance.reset
        puts "Warmup completed."

      else
        if period_count >= @qms_per_period
          period_count = 1
          TimeLog.instance.flush
          srand(Time.now.to_i)
        else
          period_count += 1
        end
      end
    end

    TimeLog.instance.record('total', Time.now - start)
  end

  private

  def init_log(object)
    print "Initializing #{object}... "
    yield
    puts "done."
    $stdout.flush
  end
end

Drive.new.run
TimeLog.instance.print_summary
