/* libruliwrap - wrapper for getaddrinfo using libruli
 * Copyright (C) 2004 Göran Weinholt <goran@weinholt.se>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of version 2 of the GNU General Public License as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307,  USA.
 */

/*
  $Id: ruli_getaddrinfo.c,v 1.3 2004/05/31 05:57:41 evertonm Exp $
*/


#include <assert.h>
#include <stdlib.h>
#include <string.h>
#include <netinet/in.h>
#include <sys/types.h>
#include <sys/socket.h>

#include <ruli_getaddrinfo.h>
#include <ruli_sync.h>
#include <ruli_txt.h>

#include <netdb.h>
#include <stdio.h>
#include <dlfcn.h>


static int (*real_getaddrinfo)(const char *node, const char *service,
			       const struct addrinfo *hints,
			       struct addrinfo **res) = NULL;

int ruli_getaddrinfo(const char *node, const char *service,
		     const struct addrinfo *hints, struct addrinfo **_res)
{
  ruli_sync_t *query;
  const ruli_list_t *srv_list;
  struct addrinfo *res = NULL, *res0 = NULL, *res1 = NULL;
  char *full_service;
  int srv_code, srv_list_size, i;

  if (!real_getaddrinfo)
    real_getaddrinfo = (int (*)(const char *,
				const char *,
				const struct addrinfo *,
				struct addrinfo **))
      dlsym(RTLD_NEXT, "getaddrinfo");

  if (!node || (node && !*node) ||
      !service || (service && !*service) || !hints || !_res)
    return real_getaddrinfo(node, service, hints, _res);

  /* Can't SRV records for numeric hosts. */
  if (hints->ai_flags & AI_NUMERICHOST)
    return real_getaddrinfo(node, service, hints, _res);
	
  /* No numeric services either. */
  if (atoi(service) > 0 || *service == '0')
    return real_getaddrinfo(node, service, hints, _res);
	
  /* No AF_INET6 support in ruli yet. */
  if (hints->ai_family != AF_UNSPEC && hints->ai_family != AF_INET)
    return real_getaddrinfo(node, service, hints, _res);
  if (hints->ai_socktype == SOCK_STREAM)
    asprintf(&full_service, "_%s._tcp", service);
  else if (hints->ai_socktype == SOCK_DGRAM)
    asprintf(&full_service, "_%s._udp", service);
  else
    return real_getaddrinfo(node, service, hints, _res);
	
  query = ruli_sync_query(full_service, node, -1, RULI_RES_OPT_SEARCH);
  free(full_service);
  if (!query)
    return real_getaddrinfo(node, service, hints, _res);

  srv_code = ruli_sync_srv_code(query);
  if (srv_code) {
    ruli_sync_delete(query);
    return real_getaddrinfo(node, service, hints, _res);
  }

  srv_list = ruli_sync_srv_list(query);
  srv_list_size = ruli_list_size(srv_list);
  if (srv_list_size < 1) {
    ruli_sync_delete(query);
    return real_getaddrinfo(node, service, hints, _res);
  }

  for (i = 0; i < srv_list_size; i++) {
    ruli_srv_entry_t *entry = ruli_list_get(srv_list, i);
    ruli_list_t *addr_list = &entry->addr_list;
    int addr_list_size = ruli_list_size(addr_list);
    int j;
    char canonname[RULI_LIMIT_DNAME_TEXT_BUFSZ];
    int canonlen;

    if (ruli_dname_decode(canonname, RULI_LIMIT_DNAME_TEXT_BUFSZ,
			  &canonlen, entry->target, entry->target_len)) {
      continue;
    }

    if (entry->port == -1) {
      struct servent *serv = 0;

      if (hints->ai_socktype == SOCK_STREAM)
	serv = getservbyname(service, "tcp");
      else if (hints->ai_socktype == SOCK_DGRAM)
	serv = getservbyname(service, "udp");
			
      if (!serv)
	continue;

      entry->port = ntohs(serv->s_port);
    }

    for (j = 0; j < addr_list_size; ++j) {
      ruli_addr_t *addr = ruli_list_get(addr_list, j);
      struct in_addr addr4;
      struct sockaddr_in *ai_addr;

      res1 = calloc(1, sizeof(struct addrinfo) +
		    sizeof(struct sockaddr_in) + canonlen + 1);
      if (!res0)
	res0 = res1;
      else
	res->ai_next = res1;
      res = res1;
			
      res->ai_family = AF_INET;
      res->ai_socktype = hints->ai_socktype;
      if (hints->ai_socktype == SOCK_STREAM)
	res->ai_protocol = IPPROTO_TCP;
      else if (hints->ai_socktype == SOCK_DGRAM)
	res->ai_protocol = IPPROTO_UDP;

      ai_addr = (struct sockaddr_in *) (res + sizeof(struct addrinfo));
      ai_addr->sin_port = htons(entry->port);
      assert(ruli_addr_family(addr) == PF_INET); /* IPv4 assumed here */
      addr4 = ruli_addr_inet(addr);
      memcpy(&ai_addr->sin_addr, &addr4, ruli_addr_size(addr));
      res->ai_addr = (struct sockaddr *) ai_addr;
      res->ai_addr->sa_family = res->ai_family;
      res->ai_addrlen = sizeof(struct sockaddr_in);

      if (hints->ai_flags & AI_CANONNAME) {
	res->ai_canonname = (char *) (res + sizeof(struct addrinfo) +
				      sizeof(struct sockaddr_in));
	strncpy(res->ai_canonname, canonname, canonlen);
      }
    }
  }
	
  ruli_sync_delete(query);

  /* Something went wrong and res0 isn't populated. */
  if (!res0)
    return real_getaddrinfo(node, service, hints, _res);

#if 0
  /* Get the normal getaddrinfo() results if possible and put them at
     the end of the linked list.*/
  if (!real_getaddrinfo(node, service, hints, &res1))
    res->ai_next = res1;
#endif

  *_res = res0;
  return 0;
}
