/*-GNU-GPL-BEGIN-*
RULI - Resolver User Layer Interface - Querying DNS SRV records
Copyright (C) 2003 Everton da Silva Marques

RULI is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

RULI is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with RULI; see the file COPYING.  If not, write to
the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.
*-GNU-GPL-END-*/

/*
  $Id: ruli_smtp.c,v 1.4 2004/03/11 19:35:15 evertonm Exp $
  */


#include <stdio.h>       /* FIXME: remove me [used for fprintf() debug] */

#include <assert.h>
#include <string.h>

#include <ruli_smtp.h>
#include <ruli_mem.h>
#include <ruli_txt.h>


static void f_query_done(ruli_srv_t *srv_qry) {
  assert(srv_qry->fall_query);
  
  ruli_res_query_delete(srv_qry->fall_query);
  ruli_free(srv_qry->fall_query);
  srv_qry->fall_query = 0;
}

static void *fall_query_done(ruli_srv_t *srv_qry, int srv_result_code)
{
#ifdef RULI_SMTP_DEBUG
      fprintf(stderr, 
	      "DEBUG: %s: %s(): query_result=%s [%d]\n", 
	      __FILE__, __PRETTY_FUNCTION__,
	      srv_result_code ? "FAIL" : "SUCCESS", srv_result_code);
#endif

  f_query_done(srv_qry);

  return _ruli_srv_query_done(srv_qry, srv_result_code);
}

static void *on_mx_answer(ruli_res_query_t *qry, void *qry_arg)
{
  ruli_srv_t *srv_qry           = (ruli_srv_t *) qry_arg;
  int        prev_srv_list_size = ruli_list_size(&srv_qry->answer_srv_list);

#ifdef RULI_SMTP_DEBUG
  {
    char txt_dname_buf[RULI_LIMIT_DNAME_TEXT_BUFSZ];
    int  txt_dname_len;
    int  result;

    result = ruli_dname_decode(txt_dname_buf, RULI_LIMIT_DNAME_TEXT_BUFSZ,
			       &txt_dname_len, 
			       (const char *) qry->full_dname, 
			       qry->full_dname_len);
    assert(!result);

    fprintf(stderr, 
	    "DEBUG: %s: %s(): domain=%s domain_len=%d\n",
	    __FILE__, __PRETTY_FUNCTION__,
	    txt_dname_buf, txt_dname_len);

    fprintf(stderr, 
	    "DEBUG: %s: %s(): id=%d answer_code=%d rcode=%d\n",
	    __FILE__, __PRETTY_FUNCTION__,
	    qry->query_id, qry->answer_code, qry->answer_header.rcode);
  }
#endif

  assert(qry->answer_code != RULI_SRV_CODE_VOID);
  assert(srv_qry->fall_query);
  assert(srv_qry->fall_query == qry);
  assert(!prev_srv_list_size);

  /*
   * Query failed?
   */
  if (qry->answer_code) {
    srv_qry->f_query_rcode = RULI_RCODE_VOID;

    if (qry->answer_code == RULI_CODE_TIMEOUT)
      return fall_query_done(srv_qry, RULI_SRV_CODE_FALL_ALARM);

    return fall_query_done(srv_qry, RULI_SRV_CODE_FALL_QUERY);
  }

  /*
   * Bad RCODE ?
   */
  {
    ruli_uint16_t rcode = qry->answer_header.rcode;

    srv_qry->f_query_rcode = rcode;

    if (rcode != RULI_RCODE_NOERROR)
      return fall_query_done(srv_qry, RULI_SRV_CODE_FALL_RCODE);
  }

  /*
   * Parse answer
   */
  {
    ruli_uint8_t *wanted_owner;
    int          wanted_owner_len;

    ruli_parse_t parse;

    if (ruli_parse_new(&parse))
      return fall_query_done(srv_qry, RULI_SRV_CODE_FALL_OTHER);

    {
      int result;

      assert(sizeof(ruli_uint8_t) == sizeof(char));

      result = ruli_parse_message(&parse, &qry->answer_header,
				  (ruli_uint8_t *) qry->answer_buf,
                                  qry->answer_msg_len);
      if (result) {
	ruli_parse_delete(&parse);
	return fall_query_done(srv_qry, RULI_SRV_CODE_FALL_PARSE);
      }
    }

    assert(sizeof(ruli_uint8_t) == sizeof(char));

    wanted_owner     = (ruli_uint8_t *) qry->full_dname;
    wanted_owner_len = qry->full_dname_len;

    /*
     * Search for MX records in answer section
     */
    {
      ruli_list_t      *an_list     = &parse.answer_list;
      int              an_list_size = ruli_list_size(an_list);
      int              i;

#ifdef RULI_SMTP_DEBUG
      {
	char wanted_txt[RULI_LIMIT_DNAME_TEXT_BUFSZ];
	int  wanted_txt_len;
	int  result;
	
	assert(sizeof(ruli_uint8_t) == sizeof(char));
	
	result = ruli_dname_decode(wanted_txt, RULI_LIMIT_DNAME_TEXT_BUFSZ, 
				   &wanted_txt_len, 
				   (const char *) wanted_owner, 
				   wanted_owner_len);
	assert(!result);
	
	fprintf(stderr, 
		"DEBUG: %s: %s(): "
		"wanted owner=(%d)%s\n", 
		__FILE__, __PRETTY_FUNCTION__,
		wanted_txt_len, wanted_txt);
      }
#endif

      /*
       * Scan answer section for IN MX records
       */
      for (i = 0; i < an_list_size; ++i) {
	ruli_rr_t *rr = ruli_list_get(an_list, i);

	if (rr->class != RULI_RR_CLASS_IN)
	  continue;
	if (rr->type != RULI_RR_TYPE_MX)
	  continue;

	/*
	 * Owner matches?
	 */

        if (ruli_dname_compare(rr->owner,
                               (ruli_uint8_t *) qry->answer_buf,
                               qry->answer_msg_len,
                               wanted_owner,
                               wanted_owner_len))
          continue;

	/*
	 * Find MX target
	 */

	{
	  int              j;
	  ruli_mx_rdata_t  mx_rdata;
	  ruli_list_t      *ad_list     = &parse.additional_list;
	  int              ad_list_size = ruli_list_size(ad_list);
	  ruli_srv_entry_t *srv_entry   = 0;

	  if (ruli_parse_rr_mx(&mx_rdata, rr->rdata, rr->rdlength,
			       (ruli_uint8_t *) qry->answer_buf,
			       qry->answer_msg_len)) {
	    ruli_parse_delete(&parse);
	    return fall_query_done(srv_qry, RULI_SRV_CODE_FALL_PARSE);
	  }

	  /*
	   * Search MX target addresses in additional section
	   */
	  
	  for (j = 0; j < ad_list_size; ++j) {
	    ruli_rr_t      *ad_rr = ruli_list_get(ad_list, j);
	    struct in_addr *addr;

	    if (ad_rr->class != RULI_RR_CLASS_IN)
	      continue;
	    if (ad_rr->type != RULI_RR_TYPE_A)
	      continue;

	    /*
	     * MX owner matches?
	     */

	    if (ruli_dname_compare(ad_rr->owner,
				   (ruli_uint8_t *) qry->answer_buf,
				   qry->answer_msg_len,
				   mx_rdata.target,
				   mx_rdata.target_len))
	      continue;

	    /*
	     * Save MX target address
	     */

	    if (!srv_entry) {
	      
	      /* Create SRV record */
	      srv_entry = ruli_malloc(sizeof(ruli_srv_entry_t));
	      if (!srv_entry) {
		ruli_parse_delete(&parse);
		return fall_query_done(srv_qry, RULI_SRV_CODE_FALL_OTHER);
	      }
	      
	      /* Create record addr list */
	      if (ruli_list_new(&srv_entry->addr_list)) {
		ruli_free(srv_entry);
		ruli_parse_delete(&parse);
		return fall_query_done(srv_qry, RULI_SRV_CODE_FALL_OTHER);
	      }
	      
	      /* Save record */
	      if (ruli_list_push(&srv_qry->answer_srv_list, srv_entry)) {
		ruli_list_delete(&srv_entry->addr_list);
		ruli_free(srv_entry);
		ruli_parse_delete(&parse);
		return fall_query_done(srv_qry, RULI_SRV_CODE_FALL_OTHER);
	      }
	      
	      /* 
	       * Init record 
	       */
	      srv_entry->priority = mx_rdata.preference;
	      srv_entry->weight   = -1;
	      srv_entry->port     = srv_qry->srv_fallback_port;
	      
	      assert(srv_qry->srv_domain_len <= RULI_LIMIT_DNAME_ENCODED);
	      srv_entry->target_len = mx_rdata.target_len;
	      memcpy(srv_entry->target, mx_rdata.target, 
		     srv_entry->target_len);
	    } /* Create SRV record */
	    
	    /* Allocate space */
	    addr = ruli_malloc(sizeof(struct in_addr));
	    if (!addr) {
	      ruli_parse_delete(&parse);
	      return fall_query_done(srv_qry, RULI_SRV_CODE_FALL_OTHER);
	    }
	    
	    /* Save space */
	    if (ruli_list_push(&srv_entry->addr_list, addr)) {
	      ruli_free(addr); 
	      ruli_parse_delete(&parse);
	      return fall_query_done(srv_qry, RULI_SRV_CODE_FALL_OTHER);
	    }
	    
	    /* Write address into space */
	    {
	      int result = ruli_parse_rr_a(addr, 
					   ad_rr->rdata, ad_rr->rdlength);
	      assert(!result); /* IN A parsing can't fail */
	    }
	  } /* Save MX target addresses */
	  
	} /* Search MX target in additional section */
	
      } /* for: IN MX scan of answer section */
      
      /* 
       * Done if at least one SRV record has been built 
       */
      {
	int curr_srv_list_size = ruli_list_size(&srv_qry->answer_srv_list);
	
	assert(curr_srv_list_size >= prev_srv_list_size);
	
	if (curr_srv_list_size > prev_srv_list_size) {
	  ruli_parse_delete(&parse);
	  return fall_query_done(srv_qry, RULI_SRV_CODE_OK);
	}
      }

#ifdef RULI_SMTP_DEBUG
      fprintf(stderr, 
	      "DEBUG: %s: %s(): id=%d answer_code=%d, rcode=%d: "
	      "BUT: no matching IN MX record\n",
	      __FILE__, __PRETTY_FUNCTION__,
	      qry->query_id, qry->answer_code, qry->answer_header.rcode);
#endif

    } /* MX search */    

    ruli_parse_delete(&parse);
    
  } /* parsed answer context */
  
  /* Dispose failed IN MX fallback query */
  f_query_done(srv_qry);

  /* Try default IN A fallback query */
  return _srv_answer_fallback_addr(srv_qry);
}

/*
  Fallback to 'smtp'
 */
static void *srv_answer_fallback_smtp(ruli_srv_t *srv_qry)
{
#ifdef RULI_SMTP_DEBUG
  {
    char txt_dname_buf[RULI_LIMIT_DNAME_TEXT_BUFSZ];
    int  txt_dname_len;
    int  result;

    result = ruli_dname_decode(txt_dname_buf, RULI_LIMIT_DNAME_TEXT_BUFSZ,
			       &txt_dname_len, 
			       srv_qry->srv_domain, srv_qry->srv_domain_len);
    assert(!result);

    fprintf(stderr, 
	    "DEBUG: %s: %s(): domain=%s len=%d\n",
	    __FILE__, __PRETTY_FUNCTION__,
	    txt_dname_buf, txt_dname_len);
  }
#endif

  assert(!srv_qry->fall_query);
  assert(!ruli_list_size(&srv_qry->answer_srv_list));

  /*
   * Allocate space for fallback query
   */
  srv_qry->fall_query = ruli_malloc(sizeof(ruli_res_query_t));
  if (!srv_qry->fall_query)
    return _ruli_srv_query_done(srv_qry, RULI_SRV_CODE_FALL_OTHER);

  /*
   * Initialize fallback query arguments
   */
  srv_qry->fall_query->q_on_answer     = on_mx_answer;
  srv_qry->fall_query->q_on_answer_arg = srv_qry;
  srv_qry->fall_query->q_domain        = srv_qry->srv_domain;
  srv_qry->fall_query->q_domain_len    = srv_qry->srv_domain_len;
  srv_qry->fall_query->q_class         = RULI_RR_CLASS_IN;
  srv_qry->fall_query->q_type          = RULI_RR_TYPE_MX;
  srv_qry->fall_query->q_options       = srv_qry->srv_options;

  /*
   * Submit fallback query
   */
  if (ruli_res_query_submit(srv_qry->srv_resolver, srv_qry->fall_query)) {
    ruli_free(srv_qry->fall_query);
    srv_qry->fall_query = 0;
    return _ruli_srv_query_done(srv_qry, RULI_SRV_CODE_FALL_OTHER);
  }

  /* Wait query answer */
  return OOP_CONTINUE;
}

ruli_search_srv_t *ruli_search_smtp_submit(ruli_res_t *resolver, 
					   void *(*call)(ruli_search_srv_t *search, void *arg),
					   void *call_arg,
					   long options,
					   const char *txt_domain)
{
  return _ruli_search_srv_submit(srv_answer_fallback_smtp,
				 resolver,
				 call,
				 call_arg,
				 options,
				 "_smtp._tcp",
				 txt_domain,
				 25);
}

