/*
 * Copyright (C) 2003 INRIA
 *
 *	INRIA
 *	Domaine de Voluceau
 *	Rocquencourt - B.P. 105
 *	78153 Le Chesnay Cedex - France
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Author: Loic Dachary <loic@gnu.org>
 * 
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <signal.h>
#include <string.h>
#include <memory.h>

#include "rfid.h"
#include "iso15693.h"

typedef struct {
  u_int8_t null_uid[ISO15693_UID_SIZE];
  u_int8_t uid[ISO15693_UID_SIZE];
  char all_A[ISO15693_BLOCK_SIZE_MAX];
  char all_B[ISO15693_BLOCK_SIZE_MAX];
  rfid_transponder_t* transponder;
  iso15693_t* iso15693;
  rfid_reader_t* reader;
  rfid_t rfid;
  int timeout;
  int destructive;
  int verbose;
} context_t;

enum {
  INVENTORY = 0,
  STAY_QUIET,
  READ_SINGLE_BLOCK,
  WRITE_SINGLE_BLOCK,
  LOCK_BLOCK,
  READ_MULTIPLE_BLOCKS,
  WRITE_MULTIPLE_BLOCKS,
  SELECT,
  RESET_TO_READY,
  WRITE_AFI,
  LOCK_AFI,
  WRITE_DSFID,
  LOCK_DSFID,
  GET_SYSTEM_INFORMATION,
  GET_MULTIPLE_BLOCK_SECURITY_STATUS,

  COMMANDS_MAX
};

static int commands[COMMANDS_MAX] = { 0, };

static context_t context;

static int report_error(context_t* context, const char* message)
{
  rfid_reader_t* reader = context->reader;

  if(context->reader) {
    char* str = reader->strerror_f(reader);
    fprintf(stderr, "%s: %s\n", message, str);
    free(str);

    /*
     * Some %$#@@# RFID transponders have no error codes: all error
     * codes are mapped to UNKNOWN.
     */
    if(reader->error_f(reader) == RFID_ERROR_NOT_SUPPORTED ||
       reader->error_f(reader) == RFID_ERROR_UNKNOWN_ERROR) {
      fprintf(stderr, "(non fatal because optional command)\n");
      return 0;
    } else {
      return -1;
    }
  } else {
    char* str = rfid_strerror(&context->rfid);
    fprintf(stderr, "%s: %s\n", message, str);
    free(str);
    return -1;
  }
}

static RETSIGTYPE timeout_f(int sig)
{
  fprintf(stderr, "\
\n\
FAILURE: Timeout attempting to establish communication with RFID reader at %s.\n\
FAILURE: Make sure a RFID reader is plugged in and turned on.\n\
\n\
", context.reader->io->device);
  exit(-1);
}

#define show_result(what) \
	if(context->verbose) { \
		fprintf(stderr, "%s : ", what); \
		if(retval < 0) \
			fprintf(stderr, " failed\n"); \
		else \
			fprintf(stderr, " OK\n"); \
	}

static int test_inventory(context_t* context)
{
  /* direct INVENTORY (mandatory) */
  iso15693_inventory_result_t result;
  iso15693_inventory_t inventory;
  int slot;
  int retval = -1;

  alarm(context->timeout);

  memset(&inventory, '\0', sizeof(iso15693_inventory_t));
  /*  inventory.one_slot = 1;*/
  if(iso15693_inventory(context->iso15693, &inventory, &result) < 0) {
    report_error(context, "iso15693_inventory");
    goto err;
  }

  if(context->verbose > 1) {
    printf("==============================\niso15693_inventory\n");
    for(slot = 0; slot < ISO15693_NB_SLOTS_DEFAULT; slot++) {
      int slot_bit = (1 << slot);
      printf("slot 0x%02x: ", slot);
    
      if(result.valid & slot_bit) {
	printf("dsfid = 0x%02x, uid = 0x%02x%02x%02x%02x%02x%02x%02x%02x\n",
	       result.data[slot].dsfid,
	       result.data[slot].uid[7],
	       result.data[slot].uid[6],
	       result.data[slot].uid[5],
	       result.data[slot].uid[4],
	       result.data[slot].uid[3],
	       result.data[slot].uid[2],
	       result.data[slot].uid[1],
	       result.data[slot].uid[0]);
      } else if(result.collide & slot_bit) {
	printf("collision\n");
      } else {
	printf("empty\n");
      }
    }
    printf("==============================\n");
  }

  commands[INVENTORY] = 1;
  retval = 0;

 err:

  alarm(0);

  show_result("inventory");

  return retval;
}

static int test_inventory_driver(context_t* context)
{
  /* driver INVENTORY (mandatory) */

  rfid_reader_t* reader = context->reader;
  rfid_transponders_t transponders;
  int i;
  int retval = -1;

  alarm(context->timeout);

  memset(&transponders, '\0', sizeof(transponders));

  if(reader->inventory_f(context->reader, &transponders) < 0) {
    report_error(context, "reader->inventory_f");
    goto err;
  }

  if(transponders.length < 1) {
    fprintf(stderr, "Make sure there is at least one ISO transponder on the reader\n");
    goto err;
  }

  if(context->verbose > 1) {
    printf("==============================\nreader->inventory_f\n");
    for(i = 0; i < transponders.length; i++) {
      char* description = reader->transponder_describe_f(reader, transponders.list[i]);
      printf("==============================\n%s", description);
      free(description);
    }
    printf("==============================\n");
  }

  /* Keep the first transponder as a reference for other tests. */
  {
    iso15693_transponder_t* tmp = (iso15693_transponder_t*)transponders.list[0]->private;
    reader->transponder_copy_f(reader, context->transponder, transponders.list[0]);
    memcpy(context->uid, tmp->uid, ISO15693_UID_SIZE);
  }

  retval = 0;
 err:
  if(transponders.alloc) {
    for(i = 0; i < transponders.size; i++) {
      reader->transponder_free_f(reader, transponders.list[i]);
    }
    free(transponders.list);
  }

  alarm(0);

  show_result("inventory_driver");

  return retval;
}

static int test_select(context_t* context)
{
  int retval = -1;

  /* direct SELECT (optional)  */
  alarm(context->timeout);

  if(iso15693_select(context->iso15693, context->uid) < 0) {
    retval = report_error(context, "iso15693_select");
    goto err;
  }

  retval = 0;

  commands[SELECT]++;

 err:
  alarm(0);

  show_result("select");

  return retval;
}

static int test_reset_to_ready(context_t* context)
{
  int retval = -1;

  /* direct RESET_TO_READY (optional)  */
  alarm(context->timeout);

  /*
   * Addressed mode
   */
  if(iso15693_reset_to_ready(context->iso15693, context->uid) < 0) {
    retval = report_error(context, "iso15693_reset_to_ready (addressed)");
    goto err;
  }
  /*
   * Non addressed mode
   */
  if(iso15693_reset_to_ready(context->iso15693, context->null_uid) < 0) {
    retval = report_error(context, "iso15693_reset_to_ready (non addressed)");
    goto err;
  }
    
  retval = 0;

  commands[RESET_TO_READY]++;

 err:
  alarm(0);

  show_result("reset_to_ready");

  return retval;
}

static int test_get_system_information(context_t* context)
{
  /* direct GET_SYSTEM_INFORMATION (optional) */
  rfid_transponder_t* transponder = 0;
  rfid_reader_t* reader = context->reader;
  iso15693_t* iso15693 = context->iso15693;
  int retval = -1;
  
  alarm(context->timeout);

  reader->transponder_alloc_f(reader, &transponder);

  /*
   * Addressed mode get_system_information
   */
  reader->transponder_copy_f(reader, transponder, context->transponder);

  iso15693_transponder_uid_set(transponder, context->uid);
  if(iso15693_get_system_information(iso15693, transponder) < 0) {
    retval = report_error(context, "iso15693_get_system_information (addressed)");
    goto err;
  }

  if(context->verbose > 1) {
    char* description;
    printf("==============================\nget_system_information (addressed)\n");
    description = reader->transponder_describe_f(reader, transponder);
    printf("==============================\n%s", description);
    free(description);
  }

  if(reader->transponder_cmp_f(reader, context->transponder, transponder)) {
    report_error(context, "iso15693_get_system_information in addressed mode returned a transponder that is different from reference transponder");
    return -1;
  }

  reader->transponder_copy_f(reader, context->transponder, transponder);

  reader->transponder_clear_f(reader, transponder);

  /*
   * Non addressed mode get_system_information
   */
  if(iso15693_get_system_information(iso15693, transponder) < 0) {
    retval = report_error(context, "iso15693_get_system_information");
    goto err;
  }

  if(context->verbose > 1) {
    char* description;
    printf("==============================\nget_system_information (non addressed)\n");
    description = reader->transponder_describe_f(reader, transponder);
    printf("==============================\n%s", description);
    free(description);
  }

  if(reader->transponder_cmp_f(reader, transponder, context->transponder)) {
    report_error(context, "iso15693_get_system_information in non addressed mode did not get the expected transponder");
    goto err;
  }

  commands[GET_SYSTEM_INFORMATION]++;

  retval = 0;

 err:

  if(transponder)
    reader->transponder_free_f(reader, transponder);

  alarm(0);

  show_result("get_system_information");

  return retval;
}


static int test_transponder_present_driver(context_t* context)
{
  /* driver transponder_present */
  rfid_transponder_t* transponder = 0;
  rfid_reader_t* reader = context->reader;
  int retval = -1;
  
  alarm(context->timeout);

  reader->transponder_alloc_f(reader, &transponder);

  if(reader->transponder_present_f(reader, transponder) < 0) {
    retval = report_error(context, "transponder_present_f");
    goto err;
  }

  if(context->verbose > 1) {
    char* description;
    printf("==============================\ntransponder_present\n");
    description = reader->transponder_describe_f(reader, transponder);
    printf("==============================\n%s", description);
    free(description);
  }

  if(reader->transponder_cmp_f(reader, transponder, context->transponder)) {
    report_error(context, "transponder_present_f did not get the expected transponder");
    goto err;
  }

  retval = 0;

 err:

  if(transponder)
    reader->transponder_free_f(reader, transponder);

  alarm(0);

  show_result("transponder_present_driver");

  return retval;
}

static int test_write_single_block(context_t* context)
{
  rfid_block_t block;
  u_int8_t storage[ISO15693_BLOCK_SIZE_MAX];
  iso15693_t* iso15693 = context->iso15693;
  rfid_transponder_t* transponder = context->transponder;
  int retval = -1;

  alarm(context->timeout);

  memset(&block, '\0', sizeof(rfid_block_t));
  block.data = storage;
  block.block_number = transponder->blocks - 1;

  memset(block.data, 'A', transponder->bytes_per_block);

  /*
   * Write the same block with AAAA...
   */
  if(iso15693_write_single_block(iso15693, transponder, &block) < 0) {
    retval = report_error(context, "iso15693_write_single_block (all A)");
    goto err;
  }

  commands[WRITE_SINGLE_BLOCK]++;
  retval = 0;

 err:

  alarm(0);

  show_result("write_single_block");

  return retval;
}

static int test_write_multiple_block(context_t* context)
{
  int retval = -1;
  int i;
  iso15693_t* iso15693 = context->iso15693;
  rfid_transponder_t* transponder = context->transponder;
  int blocks_length = 2;
  rfid_block_t* blocks;
  u_int8_t* pointer = transponder->data;

  alarm(context->timeout);

  memset(transponder->data, 'A', RFID_TRANSPONDER_DATA_SIZE);

  blocks = (rfid_block_t*)malloc(sizeof(rfid_block_t) * blocks_length);

  for(i = 0; i < blocks_length; i++) {
    blocks[i].data = pointer;
    pointer += transponder->bytes_per_block;
  }

  blocks[0].block_number = 0;
      
  if(iso15693_write_multiple_blocks(iso15693, transponder, blocks, blocks_length) < 0) {
    retval = report_error(context, "iso15693_write_multiple_blocks");
    goto err;
  }

  retval = 0;
  commands[WRITE_MULTIPLE_BLOCKS]++;

 err:

  free(blocks);

  alarm(0);

  show_result("write_multiple_blocks");

  return retval;
}

static int test_read_single_block(context_t* context)
{
  rfid_block_t block;
  u_int8_t storage[ISO15693_BLOCK_SIZE_MAX];
  iso15693_t* iso15693 = context->iso15693;
  rfid_transponder_t* transponder = context->transponder;
  int retval = -1;

  alarm(context->timeout);

  memset(&block, '\0', sizeof(rfid_block_t));
  block.data = storage;
  block.block_number = transponder->blocks - 1;
  block.security_status = 1;

  memset(block.data, '\0', transponder->bytes_per_block);
    
  if(iso15693_read_single_block(iso15693, transponder, &block) < 0) {
    retval = report_error(context, "iso15693_read_single_block)");
    goto err;
  }

  if(commands[WRITE_SINGLE_BLOCK]) {
    if(memcmp(block.data, context->all_A, transponder->bytes_per_block)) {
      fprintf(stderr, "Expected a block filled with A but got something else instead (read_single_block).\n");
      goto err;
    }
  }

  commands[READ_SINGLE_BLOCK]++;
  retval = 0;

 err:

  alarm(0);

  show_result("read_single_block");

  return retval;
}

static int test_read_multiple_blocks(context_t* context)
{
  int retval = -1;
  int i;
  iso15693_t* iso15693 = context->iso15693;
  rfid_transponder_t* transponder = context->transponder;
  int blocks_length = transponder->blocks;
  rfid_block_t* blocks;
  u_int8_t* pointer = transponder->data;

  alarm(context->timeout);

  blocks = (rfid_block_t*)malloc(sizeof(rfid_block_t) * blocks_length);

  for(i = 0; i < blocks_length; i++) {
    blocks[i].data = pointer;
    pointer += transponder->bytes_per_block;
  }

  blocks[0].block_number = 0;
  blocks[0].security_status = 0;
      
  if(iso15693_read_multiple_blocks(iso15693, transponder, blocks, blocks_length) < 0) {
    retval = report_error(context, "iso15693_read_multiple_blocks");
    goto err;
  }
  if(commands[WRITE_SINGLE_BLOCK]) {
    if(memcmp(blocks[transponder->blocks - 1].data, context->all_A, transponder->bytes_per_block)) {
      fprintf(stderr, "Expected the first block filled with A but got something else instead (read_multiple_blocks).\n");
      goto err;
    }
  }

  retval = 0;
  commands[READ_MULTIPLE_BLOCKS]++;

 err:

  alarm(0);

  free(blocks);

  show_result("read_multiple_blocks");

  return retval;
}

static int test_lock_block(context_t* context)
{
  /* direct LOCK_BLOCK (optional) */
  int retval = -1;
  iso15693_t* iso15693 = context->iso15693;
  rfid_transponder_t* transponder = context->transponder;
  rfid_block_t block;
  u_int8_t storage[ISO15693_BLOCK_SIZE_MAX];

  if(!context->destructive) {
    if(context->verbose) fprintf(stderr, "lock_block : SKIP (destructive)\n");
    return 0;
  }

  alarm(context->timeout);

  memset(&block, '\0', sizeof(rfid_block_t));
  block.data = storage;
  block.block_number = transponder->blocks - 2;

  memset(block.data, 'A', transponder->bytes_per_block);

  /*
   * Write the block with AAAA...
   */
  if(iso15693_write_single_block(iso15693, transponder, &block) < 0) {
    int error = iso15693->error;
    /*
     * Since the ISO-15693-3 error codes are not mandatory, we can
     * only assume that an UNKNOWN_ERROR is really a BLOCK_READ_ONLY
     * error.
     */
    if(error != RFID_ERROR_BLOCK_READ_ONLY &&
       error != RFID_ERROR_UNKNOWN_ERROR) {
      retval = report_error(context, "iso15693_lock_block (write_single_block on possibly locked block)");
      goto err;
    }
    if(context->verbose > 1)
      fprintf(stderr, "lock_block : the block %d is already locked because we can't write data in it.\n", block.block_number);
  } else {
    if(context->verbose > 1)
      fprintf(stderr, "lock_block : the block %d is read/write.\n", block.block_number);
  }

  /*
   * The test is considered successfull if the block is already locked
   * : that allows to use the same RFID transponder multiple times
   * instead of wasting a new tag for each test.
   */
  if(iso15693_lock_block(iso15693, transponder, &block) < 0) {
    int error = iso15693->error;
    /*
     * Since the ISO-15693-3 error codes are not mandatory, we can
     * only assume that an UNKNOWN_ERROR is really a BLOCK_ALREADY_LOCKED
     * error.
     */
    if(error != RFID_ERROR_BLOCK_ALREADY_LOCKED &&
       error != RFID_ERROR_UNKNOWN_ERROR) {
      retval = report_error(context, "iso15693_lock_block");
      goto err;
    }
    if(context->verbose > 1)
      fprintf(stderr, "lock_block : the block %d is already locked because we can't lock it.\n", block.block_number);
  } else {
    if(context->verbose > 1)
      fprintf(stderr, "lock_block : the block %d is now locked.\n", block.block_number);
  }

  retval = 0;
  commands[LOCK_BLOCK]++;

 err:
  alarm(0);

  show_result("lock_block");

  return retval;
}

static int test_read_driver(context_t* context)
{
  int retval = -1;
  rfid_transponder_t* transponder = context->transponder;
  rfid_reader_t* reader = context->reader;

  alarm(context->timeout);

  memset(transponder->data, '\0', RFID_TRANSPONDER_DATA_SIZE);
  if(reader->read_f(reader, 0, transponder->blocks, transponder) < 0) {
    retval = report_error(context, "read_f");
    goto err;
  }

  if(commands[WRITE_SINGLE_BLOCK]) {
    if(memcmp(transponder->data + transponder->bytes_per_block * (transponder->blocks - 1), context->all_A, transponder->bytes_per_block)) {
      fprintf(stderr, "Expected last block filled with A but got something else instead (read_f).\n");
      goto err;
    }
  }

  retval = 0;
 err:

  alarm(0);

  show_result("read_driver");

  return retval;
}

static int test_write_driver(context_t* context)
{
  int retval = -1;
  rfid_transponder_t* transponder = context->transponder;
  rfid_reader_t* reader = context->reader;
  int nblocks = transponder->blocks / 2;

  alarm(context->timeout);

  memset(transponder->data, 'B', RFID_TRANSPONDER_DATA_SIZE);
      
  if(reader->write_f(reader, 0, nblocks, transponder) < 0) {
    report_error(context, "write_f");
    goto err;
  }

  memset(transponder->data, '\0', RFID_TRANSPONDER_DATA_SIZE);
  if(reader->read_f(reader, 0, 1, transponder) < 0) {
    report_error(context, "read_f (testing write_f)");
    goto err;
  }
      
  if(memcmp(transponder->data, context->all_B, transponder->bytes_per_block)) {
    fprintf(stderr, "Expected a block filled with B but got something else instead (testing write_f).\n");
    goto err;
  }

  retval = 0;

 err:

  alarm(0);

  show_result("write_driver");

  return retval;
}

static int test_lock_afi(context_t* context)
{
  /* direct LOCK_AFI (optional) */
  rfid_transponder_t* transponder = context->transponder;
  iso15693_t* iso15693 = context->iso15693;
  int retval = -1;

  if(!context->destructive) {
    if(context->verbose) fprintf(stderr, "lock_afi : SKIP (destructive)\n");
    return 0;
  }

  alarm(context->timeout);

  if(iso15693_lock_afi(iso15693, context->uid) < 0) {
    /*
     * There is not error message dedicated to an attempt
     * to re-lock AFI. 
     */
    if(iso15693->error == RFID_ERROR_UNKNOWN_ERROR) {
      if(context->verbose) fprintf(stderr, "lock_afi : UNKNOWN (AFI is locked ?)\n");
      return 0;
    }
    
    retval = report_error(context, "iso15693_lock_afi");
    goto err;
  }

  iso15693_transponder_afi_set(transponder, ISO15693_AFI_GAMING);
  if(iso15693_write_afi(iso15693, transponder) < 0) {
    /*
     * There is not error message dedicated to an attempt
     * to write a locked AFI. 
     */
    if(iso15693->error != RFID_ERROR_UNKNOWN_ERROR) {
      retval = report_error(context, "iso15693_lock_afi (write_afi)");
      goto err;
    }
  } else {
    if(context->verbose) fprintf(stderr, "lock_afi : success writing AFI after it has been locked with lock_afi\n");
    goto err;
  }
  iso15693_transponder_afi_set(transponder, ISO15693_AFI_NONE);

  commands[LOCK_AFI]++;
  retval = 0;

 err:
  
  alarm(0);

  show_result("lock_afi");

  return retval;
}

static int test_write_afi(context_t* context)
{
  /* direct WRITE_AFI (optional) */
  rfid_transponder_t* transponder = context->transponder;
  iso15693_inventory_t inventory;
  iso15693_inventory_result_t results;
  iso15693_t* iso15693 = context->iso15693;
  int retval = -1;

  alarm(context->timeout);

  iso15693_transponder_afi_set(transponder, ISO15693_AFI_GAMING);
  if(iso15693_write_afi(iso15693, transponder) < 0) {
    /*
     * There is not error message dedicated to an attempt
     * to write a locked AFI. 
     */
    if(iso15693->error == RFID_ERROR_UNKNOWN_ERROR) {
      if(context->verbose) fprintf(stderr, "write_afi : UNKNOWN (AFI is locked ?)\n");
      return 0;
    }
    
    retval = report_error(context, "iso15693_write_afi");
    goto err;
  }
      
  memset(&inventory, '\0', sizeof(iso15693_inventory_t));
  inventory.afi = ISO15693_AFI_MEDICAL;
  if(iso15693_inventory(iso15693, &inventory, &results) < 0) {
    report_error(context, "iso15693_inventory (test write_afi)");
    goto err;
  }

  if(results.nvalid != 0) {
    report_error(context, "iso15693_inventory found a RFID transponder after writing AFI with GAMING and requesting AFI MEDICAL");
    goto err;
  }

  memset(&inventory, '\0', sizeof(iso15693_inventory_t));
  inventory.afi = ISO15693_AFI_GAMING;
  if(iso15693_inventory(iso15693, &inventory, &results) < 0) {
    report_error(context, "iso15693_inventory (test write_afi)");
    goto err;
  }

  if(results.nvalid != 1) {
    report_error(context, "iso15693_inventory did not find a RFID transponder after writing AFI with GAMING and requesting AFI GAMING");
    goto err;
  }

  iso15693_transponder_afi_set(transponder, ISO15693_AFI_NONE);
  if(iso15693_write_afi(iso15693, transponder) < 0) {
    retval = report_error(context, "iso15693_write_afi");
    goto err;
  }

  commands[WRITE_AFI]++;
  retval = 0;

 err:
  
  alarm(0);

  show_result("write_afi");

  return retval;
}

static int test_lock_dsfid(context_t* context)
{
  /* direct LOCK_DSFID (optional) */
  rfid_transponder_t* transponder = context->transponder;
  iso15693_t* iso15693 = context->iso15693;
  int retval = -1;

  if(!context->destructive) {
    if(context->verbose) fprintf(stderr, "lock_dsfid : SKIP (destructive)\n");
    return 0;
  }

  alarm(context->timeout);

  if(iso15693_lock_dsfid(iso15693, context->uid) < 0) {
    /*
     * There is not error message dedicated to an attempt
     * to re-lock DSFID. 
     */
    if(iso15693->error == RFID_ERROR_UNKNOWN_ERROR) {
      if(context->verbose) fprintf(stderr, "lock_dsfid : UNKNOWN (DSFID is locked ?)\n");
      return 0;
    }
    
    retval = report_error(context, "iso15693_lock_dsfid");
    goto err;
  }

  iso15693_transponder_dsfid_set(transponder, 0xAB);
  if(iso15693_write_dsfid(iso15693, transponder) < 0) {
    /*
     * There is not error message dedicated to an attempt
     * to write a locked DSFID. 
     */
    if(iso15693->error != RFID_ERROR_UNKNOWN_ERROR) {
      retval = report_error(context, "iso15693_lock_dsfid (write_dsfid)");
      goto err;
    }
  }

  commands[LOCK_DSFID]++;
  retval = 0;

 err:
  
  alarm(0);

  show_result("lock_dsfid");

  return retval;
}

static int test_get_multiple_block_security_status(context_t* context)
{
  /* direct GET_MULTIPLE_BLOCK_SECURITY_STATUS (optional) */
  rfid_transponder_t* transponder = context->transponder;
  iso15693_t* iso15693 = context->iso15693;
  rfid_block_t* blocks = 0;
  int retval = -1;

  alarm(context->timeout);

  blocks = (rfid_block_t*)malloc(sizeof(rfid_block_t) * transponder->blocks);
  memset(blocks, '\0', sizeof(rfid_block_t) * transponder->blocks);
  if(iso15693_get_multiple_block_security_status(iso15693, context->uid, blocks, transponder->blocks) < 0) {
    retval = report_error(context, "iso15693_get_multiple_block_security_status");
    goto err;
  }

  if(context->verbose > 1) {
    int i;
    fprintf(stderr, "get_multiple_block_security_status : ");
    for(i = 0; i < transponder->blocks; i++) {
      fprintf(stderr, "%s", (blocks[i].security_status ? "1" : "0"));
    }
    fprintf(stderr, "\n");
  }

  commands[GET_MULTIPLE_BLOCK_SECURITY_STATUS]++;
  retval = 0;

 err:
  
  alarm(0);

  free(blocks);

  show_result("get_multiple_block_security_status");

  return retval;
}

static int test_write_dsfid(context_t* context)
{
  /* direct WRITE_DSFID (optional) */
  rfid_transponder_t* transponder = context->transponder;
  iso15693_t* iso15693 = context->iso15693;
  int retval = -1;

  alarm(context->timeout);

  iso15693_transponder_dsfid_set(transponder, 0xAB);
  if(iso15693_write_dsfid(iso15693, transponder) < 0) {
    /*
     * There is not error message dedicated to an attempt
     * to write a locked DSFID. 
     */
    if(iso15693->error == RFID_ERROR_UNKNOWN_ERROR) {
      if(context->verbose) fprintf(stderr, "write_dsfid : UNKNOWN (DSFID is locked ?)\n");
      return 0;
    }
    
    retval = report_error(context, "iso15693_write_dsfid");
    goto err;
  }
      
  if(commands[GET_SYSTEM_INFORMATION]) {
    u_int8_t dsfid;

    iso15693_transponder_dsfid_set(transponder, 0x00);
    
    if(iso15693_get_system_information(iso15693, transponder) < 0) {
      retval = report_error(context, "iso15693_get_system_information (write dsfid)");
      goto err;
    }

    iso15693_transponder_dsfid_get(transponder, dsfid);

    if(dsfid != 0xAB) {
      report_error(context, "iso15693_write_dsfid: get_system_information returns DSFIDthat is not the expected 0xAB");
      goto err;
    }
  }

  commands[WRITE_DSFID]++;
  retval = 0;

 err:
  
  alarm(0);

  show_result("write_dsfid");

  return retval;
}

static int test_stay_quiet(context_t* context)
{
  /* direct STAY_QUIET (mandatory) */
  iso15693_inventory_t inventory;
  iso15693_inventory_result_t results;
  iso15693_t* iso15693 = context->iso15693;
  int slot;
  int retval = -1;

  alarm(context->timeout);

  if(iso15693_stay_quiet(iso15693, context->uid) < 0) {
    report_error(context, "iso15693_stay_quiet");
    goto err;
  }

      
  memset(&inventory, '\0', sizeof(iso15693_inventory_t));
  if(iso15693_inventory(iso15693, &inventory, &results) < 0) {
    report_error(context, "iso15693_inventory (test stay_quiet)");
    goto err;
  }

  for(slot = 0; slot < ISO15693_NB_SLOTS_DEFAULT; slot++) {
    int slot_bit = (1 << slot);
    if(results.valid & slot_bit) {
      if(!memcmp(results.data[slot].uid, context->uid, ISO15693_UID_SIZE)) {
	fprintf(stderr, "The transponder ignored the STAY QUIET command\n");
	goto err;
      }
    }
  }

  retval = 0;

 err:
  
  alarm(0);

  show_result("stay_quiet");

  return retval;
}

int main(int argc, char** argv)
{
  char* driver = 0;
  char* device = 0;

  if(argc > 1)
    driver = argv[1];
  if(argc > 2)
    device = argv[2];

  signal(SIGALRM, timeout_f);

  memset(&context, '\0', sizeof(context_t));
  if(argc > 3)
    context.destructive = 1;
  context.timeout = 5000;
  context.rfid.verbose = context.verbose = 0;

  if(rfid_init(&context.rfid, driver, device) < 0) {
    report_error(&context, "rfid_init");
    return -1;
  }
  context.reader = context.rfid.reader;
  context.iso15693 = (iso15693_t*)context.reader->private;

  context.reader->transponder_alloc_f(context.reader, &context.transponder);
  memset(context.all_A, 'A', ISO15693_BLOCK_SIZE_MAX);
  memset(context.all_B, 'B', ISO15693_BLOCK_SIZE_MAX);

  if(test_inventory(&context)) return -1;
  /* Registers the first transponder found for
     later tests (reference transponder). */
  if(test_inventory_driver(&context)) return -1;
  /* Find out the memory layout of the reference
     transponder. */
  if(test_get_system_information(&context)) return -1;
  if(test_transponder_present_driver(&context)) return -1;

  if(test_write_afi(&context)) return -1;
  if(test_write_single_block(&context)) return -1;
  if(test_write_multiple_block(&context)) return -1;
  if(test_read_single_block(&context)) return -1;
  if(test_read_multiple_blocks(&context)) return -1;
  if(test_read_driver(&context)) return -1;
  if(test_write_driver(&context)) return -1;
  if(test_lock_block(&context)) return -1;
  if(test_lock_afi(&context)) return -1;
  if(test_write_dsfid(&context)) return -1;
  if(test_lock_dsfid(&context)) return -1;
  if(test_get_multiple_block_security_status(&context)) return -1;

  if(test_stay_quiet(&context)) return -1;
  if(test_reset_to_ready(&context)) return -1;
  if(test_select(&context)) return -1;

  context.reader->transponder_free_f(context.reader, context.transponder);

  rfid_end(&context.rfid);

  return 0;
}
