/*
 * Copyright (C) 2003 INRIA
 *
 *	INRIA
 *	Domaine de Voluceau
 *	Rocquencourt - B.P. 105
 *	78153 Le Chesnay Cedex - France
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Author: Loic Dachary <loic@gnu.org>
 * 
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <signal.h>

#include "rfid_error.h"
#include "m2xxh.h"

static void report_error(rfid_reader_t* reader, const char* message)
{
  if(reader) {
    char* str = reader->strerror_f(reader);
    fprintf(stderr, "%s: %s\n", message, str);
    free(str);
  } else {
    fprintf(stderr, "no RFID reader found\n");
  }
}

static m2xxh_t* m2xxh;
static rfid_reader_t* reader;

static RETSIGTYPE alarm_f(int sig)
{
  fprintf(stderr, "\
\n\
FAILURE: Timeout attempting to establish communication with RFID reader at %s.\n\
FAILURE: Make sure a RFID reader is plugged in and turned on.\n\
\n\
", reader->io->device);
  exit(-1);
}

int m2xxh_reader_alloc(rfid_reader_t** reader);

int main(int argc, char** argv)
{
  int timeout = 5;
  int verbose = 0;
  int speed;

  rfid_io_t io;

  if(m2xxh_reader_alloc(&reader) < 0)
    return -1;

  m2xxh = m2xxh_from_reader(reader);
  m2xxh->verbose = verbose;
  reader->io = &io;

  {
    char* device;
    if(argc > 1)
      device = argv[1];
    else
      device = "/dev/ttyS0";
    rfid_io_init(&io, device);
  }

  signal(SIGALRM, alarm_f);

  alarm(timeout);
  if(reader->probe_f(reader, "m2xxh") < 0) {
    report_error(reader, "probe_f");
    return -1;
  }
  speed = io.speed;
  alarm(0);

  alarm(timeout);
  if(reader->init_f(reader, "m2xxh") < 0) {
    report_error(reader, "init_f");
    return -1;
  }
  alarm(0);

  /* m2xxh_read_status */
  {
    int where[] = {
      M2XXH_PARAMETER_LOCATION_RAM,
      M2XXH_PARAMETER_LOCATION_EEPROM
    };
    int i;

    for(i = 0 ; i < sizeof(where)/sizeof(int); i++) {
      m2xxh_reader_status_t status;
      status.parameter = where[i];
      status.fields = M2XXH_READER_STATUS_ALL;
      if(where[i] == M2XXH_PARAMETER_LOCATION_RAM)
	status.fields &= ~M2XXH_READER_STATUS_STATE_POWER_ON;
      if(m2xxh_read_status(m2xxh, &status)) {
	report_error(reader, "m2xxh_read_status");
	return -1;
      }

      if(verbose) {
	fprintf(stderr, "read_status from %s\n", (status.parameter & M2XXH_PARAMETER_LOCATION_RAM) ? "RAM" : "EEPROM");
	fprintf(stderr, "\tspeed : ");
	switch(status.speed) {
	case M2XXH_SERIAL_SPEED_9600:
	  fprintf(stderr, "9600");
	  break;
	case M2XXH_SERIAL_SPEED_19200:
	  fprintf(stderr, "19200");
	  break;
	case M2XXH_SERIAL_SPEED_38400:
	  fprintf(stderr, "38400");
	  break;
	case M2XXH_SERIAL_SPEED_57600:
	  fprintf(stderr, "57600");
	  break;
	case M2XXH_SERIAL_SPEED_115200:
	  fprintf(stderr, "115200");
	  break;
	case M2XXH_SERIAL_SPEED_424000:
	  fprintf(stderr, "424000");
	  break;
	default:
	  fprintf(stderr, "??? (0x%02x)", status.speed);
	  break;
	}
	fprintf(stderr, "\n");
	fprintf(stderr, "\toutput : out1=%s out2=%s\n", ((status.io_output & M2XXH_IO_OUTPUT_OUT1) ? "ON" : "OFF"), ((status.io_output & M2XXH_IO_OUTPUT_OUT2) ? "ON" : "OFF"));
	fprintf(stderr, "\tinput : in1=%s\n", (status.io_input & M2XXH_IO_INPUT_IN1) ? "ON" : "OFF");
	fprintf(stderr, "\tprotocol : ");
	switch(status.protocol) {
	case M2XXH_PROTOCOL_ISO_15693:
	  fprintf(stderr, "iso-15693");
	  break;
	case M2XXH_PROTOCOL_ISO_14443A:
	  fprintf(stderr, "iso-14443A");
	  break;
	case M2XXH_PROTOCOL_ISO_14443B:
	  fprintf(stderr, "iso-14443B");
	  break;
	default:
	  fprintf(stderr, "unknown (%d)", status.protocol);
	  break;
	}
	fprintf(stderr, "\n");
	fprintf(stderr, "\ttimeouts : 00 => %dus,\n\t\t   01 => %dus,\n\t\t   10 => %dus,\n\t\t   11 => %dus\n", status.timeouts[0], status.timeouts[1], status.timeouts[2], status.timeouts[3]);
      }
    }
  }

  /* m2xxh_select_card */
  {
    u_int8_t serial[M2XXH_SELECT_CARD_SERIAL_LENGTH] = { 0 };
    u_int8_t protocol = M2XXH_SELECT_CARD_ISO_ALL;

    if(m2xxh_select_card(m2xxh, serial, &protocol)) {
      if(reader->error_f(reader) == RFID_ERROR_TRANSPONDER_NOT_FOUND) {
	fprintf(stderr, "\
\n\
FAILURE: The test program will attempt to probe a transponder,\n\
FAILURE: please put a transponder on the RFID reader.\n\
\n\
");
      } else {
	report_error(reader, "m2xxh_select_card");
      }
      return -1;
    }
  }

  /* m2xxh_set_status SERIAL_SPEED */
  {
    m2xxh_reader_status_t status;
    status.parameter = M2XXH_PARAMETER_LOCATION_RAM;
    status.fields = M2XXH_READER_STATUS_SPEED;
    status.speed = M2XXH_SERIAL_SPEED_57600;

    if(m2xxh_set_status(m2xxh, &status)) {
      report_error(reader, "m2xxh_set_status SERIAL_SPEED_57600");
      return -1;
    }
  }

  /* m2xxh_set_status RESET_MAGNETIC_FIELD */
  {
    m2xxh_reader_status_t status;
    status.parameter = M2XXH_PARAMETER_RESET_MAGNETIC_FIELD;
    status.fields = M2XXH_READER_STATUS_NONE;

    if(m2xxh_set_status(m2xxh, &status)) {
      report_error(reader, "m2xxh_set_status RESET_MAGNETIC_FIELD");
      return -1;
    }
  }

  /* m2xxh_set_status RESET_COUPLER */
  {
    m2xxh_reader_status_t status;
    status.parameter = M2XXH_PARAMETER_RESET_COUPLER;
    status.fields = M2XXH_READER_STATUS_NONE;

    if(m2xxh_set_status(m2xxh, &status)) {
      report_error(reader, "m2xxh_set_status RESET_COUPLER");
      return -1;
    }

    /* Back to the original speed because EEPROM reloaded. */
    m2xxh->io->speed = speed;
    if(rfid_io_configure(m2xxh->io) < 0) {
      perror("rfid_io_configure");
      return -1;
    }
  }

  reader->end_f(reader);
  rfid_free(reader);
  
  return 0;
}
