/*
 * Copyright (C) 2003 INRIA
 *
 *	INRIA
 *	Domaine de Voluceau
 *	Rocquencourt - B.P. 105
 *	78153 Le Chesnay Cedex - France
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Author: Loic Dachary <loic@gnu.org>
 * 
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <signal.h>
#include <string.h>
#include <memory.h>
#include <errno.h>

#include "iso15693.h"

static iso15693_t* iso15693;
static rfid_reader_t* reader;

static void report_error(rfid_reader_t* reader, const char* message)
{
  char* str = reader->strerror_f(reader);
  fprintf(stderr, "%s: %s\n", message, str);
  free(str);
}

static RETSIGTYPE timeout(int sig)
{
  fprintf(stderr, "\
\n\
FAILURE: Timeout attempting to establish communication with RFID reader at %s.\n\
FAILURE: Make sure a RFID reader is plugged in and turned on.\n\
\n\
", reader->device);
  exit(-1);
}

int main(int argc, char** argv)
{
  iso15693_inventory_result_t* results;
  int results_length;

  if(iso15693_reader_alloc(&reader) < 0)
    return -1;
  iso15693 = (iso15693_t*)reader->private;

  if(argc > 1)
    reader->device = argv[1];
  else
    reader->device = "/dev/ttyS0";

  if(signal(SIGALRM, timeout) < 0) {
    perror("signal(SIGALRM)");
    return -1;
  }

  alarm(5);
  if(reader->init_f(reader) < 0) {
    report_error(reader, "init");
    return -1;
  }
  alarm(0);
  /* reader->verbose_f(reader, 2); */

  alarm(5);
  {
    iso15693_inventory_t inventory;
    int i;
    memset(&inventory, '\0', sizeof(iso15693_inventory_t));
    if(iso15693_inventory(iso15693, &inventory, &results, &results_length) < 0) {
      report_error(reader, "iso15693_inventory");
      return -1;
    }

    if(results_length < 1) {
      fprintf(stderr, "Make sure there is at least one ISO transponder on the reader\n");
      return -1;
    }
    
    printf("==============================\niso15693_inventory\n");
    for(i = 0; i < results_length; i++) {
      printf("dsfid = 0x%02x, uid = 0x%02x%02x%02x%02x%02x%02x%02x%02x\n",
	      results[i].dsfid,
	      results[i].uid[7],
	      results[i].uid[6],
	      results[i].uid[5],
	      results[i].uid[4],
	      results[i].uid[3],
	      results[i].uid[2],
	      results[i].uid[1],
	      results[i].uid[0]);
    }
    printf("==============================\n");
  }
  alarm(0);

  alarm(5);
  {
    rfid_transponder_t** transponders = 0;
    int transponders_length = 0;
    int transponders_size = 0;
    int i;

    if(reader->inventory_f(reader, &transponders, &transponders_length, &transponders_size) < 0) {
      report_error(reader, "reader->inventory_f");
      return -1;
    }

    if(results_length < 1) {
      fprintf(stderr, "Make sure there is at least one ISO transponder on the reader\n");
      return -1;
    }
    
    printf("==============================\nreader->inventory_f\n");
    for(i = 0; i < results_length; i++) {
      char* description = reader->transponder_describe_f(reader, transponders[i]);
      printf("==============================\n%s", description);
      free(description);
      free(transponders[i]);
    }
    printf("==============================\n");

    free(transponders);
  }
  alarm(0);

  /*  alarm(5);*/
  {
#if 0
    if(iso15693_stay_quiet(iso15693, results[0].uid) < 0) {
      report_error(reader, "iso15693_stay_quiet");
      return -1;
    }
#endif

#if 0 /* TRY AGAIN WITH TWO TRANSPONDERS. THE TRANSPONDER NOT FOUND ERROR SHOULD NOT HAPPEN */
    {
      iso15693_inventory_t inventory;
      iso15693_inventory_result_t* other;
      int other_length;
      
      memset(&inventory, '\0', sizeof(iso15693_inventory_t));
      if(iso15693_inventory(iso15693, &inventory, &other, &other_length) < 0) {
	report_error(reader, "iso15693_inventory");
	return -1;
      }

      if(other_length != results_length - 1) {
	fprintf(stderr, "The transponder ignored the STAY QUIET command\n");
	return -1;
      }

      if(other)
	free(other);
    }
#endif
  }
  alarm(0);

#if 0 /* RESET TO READY not supportted by S6350 */
  alarm(5);
  {
    u_int8_t null_uid[ISO15693_UID_SIZE] = { 0, 0, 0, 0, 0, 0, 0, 0 };
    /*
     * Non addressed mode
     */
    if(iso15693_reset_to_ready(iso15693, null_uid) < 0) {
      report_error(reader, "iso15693_reset_to_ready (non addressed)");
      return -1;
    }
    /*
     * Addressed mode
     */
    if(iso15693_reset_to_ready(iso15693, results[0].uid) < 0) {
      report_error(reader, "iso15693_reset_to_ready (addressed)");
      return -1;
    }
    
  }
  alarm(0);
#endif

  {
    rfid_transponder_t* transponder;
    char all_A[ISO15693_BLOCK_SIZE_MAX];
    char all_B[ISO15693_BLOCK_SIZE_MAX];

    reader->transponder_alloc_f(reader, &transponder);
    iso15693_transponder_uid_set(transponder, results[0].uid);
    transponder->blocks = ISO15693_BLOCK_COUNT_MAX;
    transponder->bytes_per_block = ISO15693_BLOCK_SIZE_MAX;
    memset(all_A, 'A', ISO15693_BLOCK_SIZE_MAX);
    memset(all_B, 'B', ISO15693_BLOCK_SIZE_MAX);

    {
      rfid_block_t block;
      char storage[ISO15693_BLOCK_SIZE_MAX];

      memset(&block, '\0', sizeof(rfid_block_t));
      block.data = storage;
      block.block_number = 2;
      block.security_status = 1;

      /*
       * Read one block
       */
      if(iso15693_read_single_block(iso15693, transponder, &block) < 0) {
	report_error(reader, "iso15693_read_single_block (probe)");
	return -1;
      }

      printf("Number of bytes per block in transponder : %d\n", transponder->bytes_per_block);
      memset(block.data, 'A', transponder->bytes_per_block);

      /*
       * Write the same block with AAAA...
       */
      if(iso15693_write_single_block(iso15693, transponder, &block) < 0) {
	report_error(reader, "iso15693_write_single_block (all A)");
	return -1;
      }

      /*
       * Read back the block and checks that it contains AAAA...
       */
      memset(block.data, '\0', transponder->bytes_per_block);
    
      if(iso15693_read_single_block(iso15693, transponder, &block) < 0) {
	report_error(reader, "iso15693_read_single_block (check)");
	return -1;
      }

      if(memcmp(block.data, all_A, transponder->bytes_per_block)) {
	fprintf(stderr, "Expected a block filled with A but got something else instead (read_single_block).\n");
	return -1;
      }
    }

    /*
     * Rely on previous test to a) set the bytes_per_block with an
     * appropriate value, b) fill block 2 (third block) with A.
     */
    {
      int blocks_length = 5;
      rfid_block_t blocks[blocks_length];
      int i;
      u_int8_t* pointer = transponder->data;

      for(i = 0; i < blocks_length; i++) {
	blocks[i].data = pointer;
	pointer += transponder->bytes_per_block;
      }

      blocks[0].block_number = 0;
      blocks[0].security_status = 0;
      
      if(iso15693_read_multiple_blocks(iso15693, transponder, blocks, blocks_length) < 0) {
	report_error(reader, "iso15693_read_multiple_blocks");
	return -1;
      }

      if(memcmp(transponder->data + transponder->bytes_per_block * 2, all_A, transponder->bytes_per_block)) {
	fprintf(stderr, "Expected a block filled with A but got something else instead (read_multiple_blocks).\n");
	return -1;
      }
    }

    /*
     * Same as previous (multiple read) using the reader interface.
     */
    {
      memset(transponder->data, '\0', RFID_TRANSPONDER_DATA_SIZE);
      if(reader->read_f(reader, 0, 5, transponder) < 0) {
	report_error(reader, "read_f");
	return -1;
      }

      if(memcmp(transponder->data + transponder->bytes_per_block * 2, all_A, transponder->bytes_per_block)) {
	fprintf(stderr, "Expected a block filled with A but got something else instead (read_f).\n");
	return -1;
      }
    }

    /*
     * Write first 5 blocks with 'B' & check the result, using the
     * reader interface.
     */
    {
      memset(transponder->data, 'B', RFID_TRANSPONDER_DATA_SIZE);
      
      if(reader->write_f(reader, 0, 5, transponder) < 0) {
	report_error(reader, "write_f");
	return -1;
      }

      memset(transponder->data, '\0', RFID_TRANSPONDER_DATA_SIZE);
      if(reader->read_f(reader, 0, 1, transponder) < 0) {
	report_error(reader, "read_f (testing write_f)");
	return -1;
      }
      
      if(memcmp(transponder->data, all_B, transponder->bytes_per_block)) {
	fprintf(stderr, "Expected a block filled with B but got something else instead (testing write_f).\n");
	return -1;
      }
    }

    /*
     * Grab the first transponder available.
     */
    reader->transponder_clear_f(reader, transponder);
    if(reader->transponder_present_f(reader, transponder) < 0) {
      report_error(reader, "transponder_present_f");
      return -1;
    }

    {
      char* description = reader->transponder_describe_f(reader, transponder);
      printf("==============================\n%s", description);
      free(description);
    }

    /*
     * Non addressed mode get_system_information
     */
    reader->transponder_clear_f(reader, transponder);
    if(iso15693_get_system_information(iso15693, transponder) < 0) {
      report_error(reader, "iso15693_get_system_information");
      return -1;
    }

    {
      char* description = reader->transponder_describe_f(reader, transponder);
      printf("==============================\n%s", description);
      free(description);
    }

    {
      u_int8_t uid[ISO15693_UID_SIZE];
      iso15693_transponder_uid_get(transponder, uid);
      if(memcmp(uid, results[0].uid, ISO15693_UID_SIZE)) {
	report_error(reader, "iso15693_get_system_information in non addressed mode did not get the expected UID");
	return -1;
      }
    }

    {
      rfid_transponder_t* transponder_2 = 0;
      reader->transponder_alloc_f(reader, &transponder_2);
      /*
       * Save the transponder.
       */
      reader->transponder_copy_f(reader, transponder_2, transponder);

      /*
       * Addressed mode get_system_information
       */
      reader->transponder_clear_f(reader, transponder);
      iso15693_transponder_uid_set(transponder, results[0].uid);
      if(iso15693_get_system_information(iso15693, transponder) < 0) {
	report_error(reader, "iso15693_get_system_information (addressed)");
	return -1;
      }

      {
	char* description = reader->transponder_describe_f(reader, transponder);
	printf("==============================\n%s", description);
	free(description);
      }

      if(reader->transponder_cmp_f(reader, transponder, transponder_2)) {
	report_error(reader, "iso15693_get_system_information in addressed mode returned a transponder description that is different from the description returned in non addressed mode");
	return -1;
      }
      reader->transponder_free_f(reader, transponder_2);
    }

    reader->transponder_free_f(reader, transponder);
  }
  alarm(0);

  if(results)
    free(results);

  reader->end_f(reader);
  reader->free_f(reader);
  
  return 0;
}
