/*
 * Copyright (C) 2003 _INRIA_
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Author: Loic Dachary <loic@gnu.org>
 * 
 */

#ifndef _RFID_H_
#define _RFID_H_

#include <signal.h>
#include <s6350.h>

/*
 * Assume that a RFID transponder can hold a maximum of 32 bytes.
 */
#define RFID_TRANSPONDER_DATA_SIZE 32

/*
 * RFID transponder identity and data.
 */
typedef struct {
  s6350_transponder_t info;		/* Identity and storage
					   layout. */
  u_int8_t data[RFID_TRANSPONDER_DATA_SIZE + 1]; /* + 1 for trailing NULL */
  int data_length;			/* Length of data in the
					   data field. */
} rfid_transponder_t;

/*
 * Values of the options field of the rfid_t structure.
 */
/*
 * If set, the RFID transponder will polled every 5 milliseconds and
 * will be reported each time it is found.  If not set, a RFID
 * transponder placed on a RFID reader is only reported once.
 * For instance, the rfid_read function will only call the callback
 * once.
 */
#define RFID_OPTION_POLL		0x00000001

/*
 * Handler holding context for a specific RFID device and used by
 * all rfid functions.
 */
typedef struct {
  s6350_t s6350;		/* S6350 Midrange Reader handler. */
  int options;			/* Flags controling the behavior of
				   rfid functions. */
  int error;			/* Errors set by rfid functions. */
  char* device;			/* Path name of the device connecting
				   to the RFID reader. */
  rfid_transponder_t last;	/* Copy of the latest RFID transponder
				   reported by the RFID reader. */
  int verbose;			/* Control verbosity level of
				   functions. */
  sigset_t sigset;		/* Saved signal handlers state. */
} rfid_t;

/*
 * Opens rfid->device that must be a tty and initializes it in raw
 * mode. Probe the presence of a RFID reader and sets the carrier on.
 */
int rfid_init(rfid_t* rfid);

/*
 * Cleanup the context and closes the device used to dialog with
 * the RFID reader.
 */
void rfid_end(rfid_t* rfid);

/*
 * Valid return values of rfid_read_callback_t functions.
 */
#define RFID_CALLBACK_OK	0	/* Everything is fine, keep going. */
#define RFID_CALLBACK_END	1	/* Everything is fine, end outer loop. */
#define RFID_CALLBACK_ERROR	-1	/* Something went wrong, end outer loop on error. */

/*
 * Callback function for the rfid_read function.
 * Each time a transponder is found, call the callback with transponder information
 * (transponder argument) and the anonymous pointer given as the third argument of
 * the read function (data argument).
 *
 * The transponder->data array is always NULL terminated.
 * 
 */
typedef int rfid_read_callback_t(rfid_t* rfid, const rfid_transponder_t* transponder, void* data);

/*
 * Infinite loop polling the RFID reader (using rfid_poll). When
 * a RFID transponder is found, call the callback function, passing it
 * the data argument. If rfid->options is set to RFID_OPTION_POLL, 
 * call the callback function each time the poll reports the presence
 * of the RFID transponder. If RFID_OPTION_POLL is not set, the callback
 * function is only called if a RFID transponder change occurs. 
 *
 */
int rfid_read(rfid_t* rfid, rfid_read_callback_t* callback, void* data);

/*
 * Query the RFID reader for a transponder. If one is found, fill the
 * transponder argument with information and content. Return on error
 * if no transponder is found, and set rfid->error to ENOENT.
 *
 * The transponder->data array is always NULL terminated.
 * 
 */
int rfid_poll(rfid_t* rfid, rfid_transponder_t* transponder);

/*
 * Write transponder->data_length bytes from transponder->data to
 * a RFID transponder. If the length of the data written does
 * not fill the available space on the RFID transponder, the
 * remaining bytes are set to NULL.
 *
 * The tranponder->info is filled with the informations describing
 * the RFID transponder on which the data was written.
 */
int rfid_write(rfid_t* rfid, rfid_transponder_t* transponder);

/*
 * Build a string describing the error condition and return it. It is
 * the responsibility of the caller to free(3) the returned string.
 */
char* rfid_strerror(rfid_t* rfid);

/*
 * Return the latest error number. Positive values are errno(3)
 * values, negative values are RFID reader specific error numbers.
 */
int rfid_error(rfid_t* rfid);

/*
 * Return the file descriptor used to dialog with the RFID reader.
 */
#define rfid_fd(rfid) ((rfid)->s6350.fd)

/*
 * Set the options to value.
 */
#define rfid_options(rfid,value) ((rfid)->options = (value))

/*
 * Return the version of the library as a string (1.0.0 for instance).
 */
const char* rfid_version(void);

#endif /* _RFID_H_ */
