/*
 * Copyright (C) 2003 INRIA
 *
 *	INRIA
 *	Domaine de Voluceau
 *	Rocquencourt - B.P. 105
 *	78153 Le Chesnay Cedex - France
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Author: Loic Dachary <loic@gnu.org>
 * 
 */

/*
 * For a detailed description see rfid(3)
 */

#ifndef _RFID_H_
#define _RFID_H_

#include <signal.h>
#include <rfid_reader.h>
#include <rfid_error.h>

#if defined(__cplusplus)
extern "C" {
#endif

/*
 * Values of the options field of the rfid_t structure.
 */
#define RFID_OPTION_FREE_IO		0x00000002

struct rfid;
/*
 * Called when a new RFID transponder enters the range of the 
 * RFID reader (event == RFID_EVENT_NEW), leaves the range of the
 * RFID reader (event == RFID_EVENT_DELETE) or is still in the
 * range of the RFID reader (event == RFID_EVENT_AGAIN).
 * The transponder argument fully describes the RFID transponder,
 * including meta information. The data argument is an opaque pointer
 * identical to the data argument of the rfid_event_loop function.
 */
typedef int rfid_transponder_callback_t(struct rfid* rfid, int event, const rfid_transponder_t* transponder, void* data);

/*
 * Called at the end of each rfid_event_loop iteration. The
 * transponders argument is an array of pointers to rfid_transponder_t
 * objects that fully describe the RFID transponders including meta
 * information. The data argument is an opaque pointer
 * identical to the data argument of the rfid_event_loop function.
 */
typedef int rfid_inventory_callback_t(struct rfid* rfid, rfid_transponder_t** transponders, int transponders_length, void* data);

/*
 * Context of the event loop also used for polling.
 */
typedef struct {
  int events_mask;
  rfid_transponder_callback_t* transponder_callback;
  rfid_inventory_callback_t* inventory_callback;
  void* data;
  
  rfid_transponders_t current;
  rfid_transponders_t cache;
} rfid_events_context_t;

/*
 * Handler holding context for a specific RFID device and used by
 * all rfid functions.
 */
typedef struct rfid {
  rfid_reader_t* reader;	/* RFID reader handler. */
  rfid_events_context_t events_context; /* State of the event loop. */
  int options;			/* Flags controling the behavior of
				   rfid functions. */
  int error;			/* Errors set by rfid functions. */
  int verbose;			/* Control verbosity level. */
  sigset_t sigset;		/* Saved signal handlers state. */
} rfid_t;

/*
 * Placeholders for the driver and device arguments of the
 * rfid_init function.
 */
#define RFID_DRIVER_GUESS	0
#define RFID_DEVICE_GUESS	0

/*
 * Load the RFID drivers modules and call their initialization function.
 */
int rfid_init(rfid_t* rfid, const char* driver, const char* device);

/*
 * Call the destructors of the RFID drivers modules and
 * cleanup internal state.
 */
void rfid_end(rfid_t* rfid);

/*
 * Return a new rfid_transponder_t object that must be
 * freed by the caller using rfid_transponder_free.
 */
rfid_transponder_t* rfid_transponder_alloc(rfid_t* rfid);
/*
 * Free a rfid_transponder_t object previously allocated by
 * rfid_transponder_alloc.
 */
void rfid_transponder_free(rfid_t* rfid, rfid_transponder_t* transponder);
/*
 * Return a malloc'ed string containing a null terminated human readable
 * description of the transponder given in argument.
 */
char* rfid_transponder_describe(rfid_t* rfid, rfid_transponder_t* transponder);
/*
 * Return a null terminated string containing an ascii representation
 * of the unique identifier of the transponder given in argument.
 */
int rfid_transponder_id_get(rfid_t* rfid, const rfid_transponder_t* transponder, char** idp);
/*
 * Set the unique identifier of the tranponder given in argument 
 * from the ascii representation provided in the null terminated string id.
 */
int rfid_transponder_id_set(rfid_t* rfid, rfid_transponder_t* transponder, char* id);

int rfid_fd(rfid_t* rfid);

/*
 * Callback functions for the rfid_event_loop function.
 */

/*
 * Valid return values of rfid_transponder_callback_t functions.
 */
#define RFID_CALLBACK_OK	0	/* Everything is fine, keep
					   going. */
#define RFID_CALLBACK_END	1	/* Everything is fine, end
					   outer loop. */
#define RFID_CALLBACK_ERROR	-1	/* Something went wrong, end
					   outer loop on error. */

/*
 * Values of the event argument of the rfid_transponder_callback_t
 * funtion and of the event_mask argument of the rfid_event_loop 
 * function.
 */
#define RFID_EVENT_NEW		0x01	/* A transponder entered the
					   range of the RFID
					   reader. */
#define RFID_EVENT_DELETE	0x02	/* A transponder left the
					   range of the RFID
					   reader. */
#define RFID_EVENT_AGAIN	0x04    /* A transponder is still in
					   the range of the RFID
					   reader. */
#define RFID_EVENT_INVENTORY	0x08	/* An inventory of all
					   transponders in the range
					   of the RFID reader was
					   completed. */
#define RFID_EVENT_ALL		0xFF
#define RFID_EVENT_NONE		0x00

/*
 * Prepare the context used by the rfid_event_poll and rfid_event_loop
 * functions. The events_mask argument controls the calls to the
 * transponder_callback and inventory_callback functions.  If the
 * RFID_EVENT_NEW or RFID_EVENT_DELETE or RFID_EVENT_AGAIN bits are
 * set, the transponder_callback function will be called. See the
 * rfid_transponder_callback_t description for more information. If
 * the RFID_EVENT_INVENTORY bit is set, the inventory_callback
 * function will be called at the end of each iteration. See the
 * rfid_inventory_callback_t description for more information. The
 * data argument is an opaque pointer that is passed to the callback
 * functions.
 */
int rfid_events_context_init(rfid_t* rfid, int events_mask, rfid_transponder_callback_t* transponder_callback, rfid_inventory_callback_t* inventory_callback, void* data);

/*
 * Free the context allocated by rfid_events_context_init.
 */
int rfid_events_context_end(rfid_t* rfid);

/*
 * Process a single event. The rfid_events_context_init function
 * must be called prior to the first call to rfid_event_poll to
 * specify events of interest and callbacks.
 */
int rfid_event_poll(rfid_t* rfid);

/*
 * Infinite loop polling the RFID reader. The arguments are
 * fed to rfid_events_context_init and rfid_event_poll is
 * called in an infinite loop.
 */
int rfid_event_loop(rfid_t* rfid, int events_mask, rfid_transponder_callback_t* transponder_callback, rfid_inventory_callback_t* inventory_callback, void* data);

/*
 * Write transponder->data_length bytes from transponder->data to
 * a RFID transponder. If the length of the data written does
 * not fill the space available on the RFID transponder, the
 * remaining bytes are set to NULL.
 */
int rfid_write(rfid_t* rfid, rfid_transponder_t* transponder);

/*
 * Build a string describing the error condition and return it. It is
 * the responsibility of the caller to free(3) the returned string.
 */
char* rfid_strerror(rfid_t* rfid);

/*
 * Return the latest error number. 
 */
int rfid_error(rfid_t* rfid);

/*
 * Return the version of the library as a string ("1.0.0" for instance).
 */
const char* rfid_version(void);

#if defined(__cplusplus)
}
#endif

#endif /* _RFID_H_ */
