/*
 * Copyright (C) 2003 INRIA
 *
 *	INRIA
 *	Domaine de Voluceau
 *	Rocquencourt - B.P. 105
 *	78153 Le Chesnay Cedex - France
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Author: Loic Dachary <loic@gnu.org>
 * 
 * Thanks to Bruno Charrat and Frederic Capasso for their help in
 * understanding the m2xxh harware and their explanations of
 * undocumented features.
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <memory.h>
#include <errno.h>

#include <rfid_error.h>
#include <rfid_reader.h>
#include <iso15693.h>
#include <m2xxh.h>

/*
 * Although the M2XXH series claim to implement the low level
 * communication protocol on the basis of ISO-7816, it is believed
 * that it is customized to such an extend that it would be useless to
 * support it with a generic module that could be used for other RFID
 * readers. Where appropriate, variables and constants are prefixed
 * with "iso7816" when they match the ISO-7816 definition.
 */

/*
 * ISO7816 header structure.
 *
     - CLA is an instruction class. The value FF is reserved for PTS.

     - INS is an instruction code in the instruction class. The instruction
       code is valid only if the least significant bit is 0, and the most
       significant half byte is neither 6 nor 9.

     - P1, P2 are a reference (e.g. an address) completing the instruction
       code

     - P3 codes the number n of data bytes (D1, ... , Dn) which are to be
       transmitted during the command. The direction of movement of these
       data is a function of the instruction. In an outgoing data transfer
       command, P3=0 introduces a 256 byte data transfer from the card. In an
       incoming data transfer command, P3=0 introduces no transfer of data.

*/
#define ISO7816_CLASS_INDEX		0
#define ISO7816_INSTRUCTION_INDEX	1
#define ISO7816_P1_INDEX		2
#define ISO7816_P2_INDEX		3
#define ISO7816_P3_INDEX		4

#define ISO7816_HEADER_SIZE		5

/*
 * DS_M21x_2-11.pdf, page 19.
 */
#define M2XXH_COMMAND_SELECT_CARD		0xA4
#define M2XXH_COMMAND_SELECT_PAGE		0xA6
#define M2XXH_COMMAND_TRANSMIT			0xC2
#define M2XXH_COMMAND_GET_RESPONSE		0xC0
#define M2XXH_COMMAND_READ_STATUS		0xF2
#define M2XXH_COMMAND_SET_STATUS		0xF4
#define M2XXH_COMMAND_DISABLE_COUPLER		0xAD
#define M2XXH_COMMAND_ENABLE_COUPLER		0xAE
#define M2XXH_COMMAND_ASK_RANDOM		0x84
#define M2XXH_COMMAND_SELECT_CURRENT_KEY	0x52


/*
 * DS_M21x_2-11.pdf, page 16.
 * CLASS : always 80h
 */
#define M2XXH_ISO7816_CLASS			0x80

#define clear_error(o) ((o)->error = 0)

/*
 * DS_M21x_2-11.pdf, page 16.
 *
 * Case       Host => Coupler  Host <=  Coupler    ISO Type
 * -----------------------------------------------------------
 * 1           None               None             ISO None
 * 2           None               Yes              ISO Out
 * 3           Yes                None             ISO In
 * 4           Yes                Yes              ISO In / Out
 *
 */
#define M2XXH_TYPE_ISO_NONE	1
#define M2XXH_TYPE_ISO_OUT	2
#define M2XXH_TYPE_ISO_IN	3
#define M2XXH_TYPE_ISO_IN_OUT	4

#define M2XXH_STATUS_SIZE	2
#define M2XXH_STATUS_MSB	0
#define M2XXH_STATUS_LSB	1

#define M2XXH_ACK_SIZE		1

/*
 * DS_M21x_2-11.pdf, page 63.
 */
#define M2XXH_STATUS_SUCCESS					0x9000
#define M2XXH_STATUS_DATA_LENGTH_P3_INCORRECT			0x6700
#define M2XXH_STATUS_PARAMETERS_P1_P2_INCORRECT			0x6B00
#define M2XXH_STATUS_INSTRUCTION_NOT_RECOGNIZED_PARITY_ERROR	0x6D00
/* This error code is not documented and shows when sending class 0x00 */
#define M2XXH_STATUS_CLASS_NOT_RECOGNIZED			0x6E00
#define M2XXH_STATUS_CRC_OR_AUTHENTICATION_FAILED		0x6982
#define M2XXH_STATUS_COMMAND_FLOW_INCORRECT			0x9835
#define M2XXH_STATUS_CARD_NOT_FOUND				0x6A82
#define M2XXH_STATUS_EEPROM_ERROR				0x6200
/* 
 * DS_M21x_2-11.pdf, page 29.
 */
#define M2XXH_STATUS_COUPLER_RESET				0x3B00
/* This error code is not documented and shows when sending a COUPLER_RESET */
#define M2XXH_STATUS_COUPLER_RESET_UNKNOWN			0xE0E0

static int status_check(m2xxh_t* m2xxh, u_int8_t status_msb, int status_msb_valid)
{
  u_int8_t status_words[M2XXH_STATUS_SIZE];
  unsigned short status;
  int status_offset;

  if(status_msb_valid) {
    status_words[M2XXH_STATUS_MSB] = status_msb;
    status_offset = 1;
  } else {
    status_offset = 0;
  }
  
  if(rfid_io_read(m2xxh->io, status_words + status_offset, M2XXH_STATUS_SIZE - status_offset) < 0) {
    m2xxh->error = errno;
    return -1;
  }

  status = (status_words[M2XXH_STATUS_MSB] << 8) |
    status_words[M2XXH_STATUS_LSB];

  if(m2xxh->verbose > 1) {
    fprintf(stderr, "M2XXH status : 0x%04x\n", status);
  }

  if(status != M2XXH_STATUS_SUCCESS) {
    switch(status) {
    case M2XXH_STATUS_DATA_LENGTH_P3_INCORRECT:
      m2xxh->error = RFID_ERROR_M2XXH_P3_INCORRECT;
      break;
    case M2XXH_STATUS_PARAMETERS_P1_P2_INCORRECT:
      m2xxh->error = RFID_ERROR_M2XXH_P1_P2_INCORRECT;
      break;
    case M2XXH_STATUS_INSTRUCTION_NOT_RECOGNIZED_PARITY_ERROR:
    case M2XXH_STATUS_CLASS_NOT_RECOGNIZED:
      m2xxh->error = RFID_ERROR_NOT_SUPPORTED;
      break;
    case M2XXH_STATUS_CRC_OR_AUTHENTICATION_FAILED:
      m2xxh->error = RFID_ERROR_CHECKSUM;
      break;
    case M2XXH_STATUS_COMMAND_FLOW_INCORRECT:
      m2xxh->error = RFID_ERROR_UNKNOWN_ERROR;
      break;
    case M2XXH_STATUS_CARD_NOT_FOUND:
      m2xxh->error = RFID_ERROR_TRANSPONDER_NOT_FOUND;
      break;
    case M2XXH_STATUS_EEPROM_ERROR:
      m2xxh->error = RFID_ERROR_EEPROM;
      break;
    case M2XXH_STATUS_COUPLER_RESET:
    case M2XXH_STATUS_COUPLER_RESET_UNKNOWN:
      m2xxh->error = RFID_ERROR_SUCCESS;
      break;
    default:
      m2xxh->error = RFID_ERROR_UNKNOWN_ERROR;
      break;
    }
    return -1;
  }

  return 0;
}

static int ack_check(m2xxh_t* m2xxh, u_int8_t expected_ack)
{
  u_int8_t ack;
  if(rfid_io_read(m2xxh->io, &ack, M2XXH_ACK_SIZE) < 0) {
    m2xxh->error = errno;
    RFID_ERROR_STACK(m2xxh->verbose);
    return -1;
  }

  if(ack == expected_ack) {
    if(m2xxh->verbose > 1)
      fprintf(stderr, "M2XXH ACK\n");
    return 0;
  } else {
    /*
     * When expecting an ACK, it is an error to get a status
     * word even if the status word says the operation was
     * successfull.
     */
    status_check(m2xxh, ack, 1 /* status check valid */);
    return -1;
  }
}

static int handle_request(m2xxh_t* m2xxh, int iso_type, u_int8_t* buffer_in, int buffer_in_size, u_int8_t* buffer_out, int buffer_out_size)
{
  int wrote;

  buffer_in[ISO7816_CLASS_INDEX] = M2XXH_ISO7816_CLASS;

  if(m2xxh->verbose > 1) {
    fprintf(stderr, "M2XXH header request : ");
    rfid_dump_frame(buffer_in, ISO7816_HEADER_SIZE);
    fprintf(stderr, "\n");
  }
    
  if((wrote = write(m2xxh->io->fd, buffer_in, ISO7816_HEADER_SIZE)) != ISO7816_HEADER_SIZE) {
    m2xxh->error = errno;
    RFID_ERROR_STACK(m2xxh->verbose);
    return -1;
  }

  switch(iso_type) {
  case M2XXH_TYPE_ISO_NONE:
    if(ack_check(m2xxh, buffer_in[ISO7816_INSTRUCTION_INDEX]) < 0) {
      return -1;
    }
    break;
  case M2XXH_TYPE_ISO_OUT:
    {
      if(ack_check(m2xxh, buffer_in[ISO7816_INSTRUCTION_INDEX]) < 0) {
	return -1;
      }

      if(buffer_in[ISO7816_P3_INDEX] > buffer_out_size) {
	m2xxh->error = RFID_ERROR_M2XXH_P3_INCORRECT;
	return -1;
      }
      if(rfid_io_read(m2xxh->io, buffer_out, buffer_in[ISO7816_P3_INDEX]) < 0) {
	m2xxh->error = errno;
	return -1;
      }

      if(m2xxh->verbose > 1) {
	fprintf(stderr, "M2XXH read data : ");
	rfid_dump_frame(buffer_out, buffer_in[ISO7816_P3_INDEX]);
	fprintf(stderr, "\n");
      }
    }
    break;
  case M2XXH_TYPE_ISO_IN:
    {
      if(ack_check(m2xxh, buffer_in[ISO7816_INSTRUCTION_INDEX]) < 0) {
	return -1;
      }
      
      if(buffer_in[ISO7816_P3_INDEX] > buffer_out_size) {
	m2xxh->error = RFID_ERROR_M2XXH_P3_INCORRECT;
	return -1;
      }

      if(m2xxh->verbose > 1) {
	fprintf(stderr, "M2XXH write data : ");
	rfid_dump_frame(buffer_out, buffer_in[ISO7816_P3_INDEX]);
	fprintf(stderr, "\n");
      }

      if(write(m2xxh->io->fd, buffer_out, buffer_in[ISO7816_P3_INDEX]) != buffer_in[ISO7816_P3_INDEX]) {
	m2xxh->error = errno;
	return -1;
      }
    }
    break;
  case M2XXH_TYPE_ISO_IN_OUT:
    {
      if(ack_check(m2xxh, buffer_in[ISO7816_INSTRUCTION_INDEX]) < 0) {
	return -1;
      }

      /* Send data from host to reader. */
      if(buffer_in[ISO7816_P3_INDEX] + ISO7816_HEADER_SIZE > buffer_in_size) {
	m2xxh->error = RFID_ERROR_M2XXH_P3_INCORRECT;
	return -1;
      }

      if(m2xxh->verbose > 1) {
	fprintf(stderr, "M2XXH write data : ");
	rfid_dump_frame(buffer_in + ISO7816_HEADER_SIZE, buffer_in[ISO7816_P3_INDEX]);
	fprintf(stderr, "\n");
      }

      if(write(m2xxh->io->fd, buffer_in + ISO7816_HEADER_SIZE, buffer_in[ISO7816_P3_INDEX]) != buffer_in[ISO7816_P3_INDEX]) {
	m2xxh->error = errno;
	return -1;
      }

      if(ack_check(m2xxh, buffer_in[ISO7816_INSTRUCTION_INDEX]) < 0) {
	return -1;
      }

      /* Read the answer. */
      if(buffer_in[ISO7816_P2_INDEX] > buffer_out_size) {
	m2xxh->error = RFID_ERROR_M2XXH_P1_P2_INCORRECT;
	return -1;
      }
      if(rfid_io_read(m2xxh->io, buffer_out, buffer_in[ISO7816_P2_INDEX]) < 0) {
	m2xxh->error = errno;
	return -1;
      }

      if(m2xxh->verbose > 1) {
	fprintf(stderr, "M2XXH read data : ");
	rfid_dump_frame(buffer_out, buffer_in[ISO7816_P2_INDEX]);
	fprintf(stderr, "\n");
      }
    }
    break;
  default:
    m2xxh->error = EINVAL;
    return -1;
    break;
  }

  return status_check(m2xxh, 0 /* nop */, 0 /* nop */);
}

static char* m2xxh_strerror(m2xxh_t* m2xxh)
{
  const char* error_string = rfid_internal_strerror(m2xxh->error);
  char* message = malloc(strlen(error_string) + 32);
  sprintf(message, "%d: %s", m2xxh->error, error_string);
  return message;
}

static int m2xxh_error(m2xxh_t* m2xxh)
{
  return m2xxh->error;
}

/*
 * DS_M21x_2-11.pdf, page 19.
 *    READ_STATUS (0xF2)
 */

#define M2XXH_RW_STATUS_LOCATION_EEPROM		0x00
#define M2XXH_RW_STATUS_LOCATION_IO		0x01
#define M2XXH_RW_STATUS_LOCATION_RAM		0x03

#define M2XXH_RW_STATUS_ADDRESS_STATE_POWER_ON	0x42
#define M2XXH_RW_STATUS_ADDRESS_SERIAL_SPEED	0x6D
#define M2XXH_RW_STATUS_ADDRESS_IO_OUTPUT	0x05
#define M2XXH_RW_STATUS_ADDRESS_IO_INPUT	0x07

#define M2XXH_RW_STATUS_P3			0x01

#define M2XXH_RW_STATUS_ADDRESS_PROTOCOL_LSB		0x5E
#define M2XXH_RW_STATUS_ADDRESS_PROTOCOL_MSB		0x5F
#define M2XXH_RW_STATUS_ADDRESS_TIMEOUT_00		0x68
#define M2XXH_RW_STATUS_ADDRESS_TIMEOUT_01		0x69
#define M2XXH_RW_STATUS_ADDRESS_TIMEOUT_10		0x6A
#define M2XXH_RW_STATUS_ADDRESS_TIMEOUT_11		0x6B
#define M2XXH_RW_STATUS_ADDRESS_ISO14443A_MAGIC_1	0x64
#define M2XXH_RW_STATUS_ADDRESS_ISO14443A_MAGIC_2	0x65
#define M2XXH_RW_STATUS_ADDRESS_ISO14443B_MAGIC_1	0x6F

/* Values of M2XXH_RW_STATUS_ADDRESS_PROTOCOL_LSB and
   M2XXH_RW_STATUS_ADDRESS_PROTOCOL_MSB respectively. */
#define M2XXH_RW_STATUS_PROTOCOL_ISO15693_LSB		0x21
#define M2XXH_RW_STATUS_PROTOCOL_ISO15693_MSB		0x31
#define M2XXH_RW_STATUS_PROTOCOL_ISO14443A_LSB		0x32
#define M2XXH_RW_STATUS_PROTOCOL_ISO14443A_MSB		0x12
#define M2XXH_RW_STATUS_PROTOCOL_ISO14443B_LSB		0x44
#define M2XXH_RW_STATUS_PROTOCOL_ISO14443B_MSB		0x31

/* Values of M2XXH_RW_STATUS_ADDRESS_*_MAGIC_* */
#define M2XXH_RW_STATUS_PROTOCOL_ISO14443A_MAGIC_1	0x16
#define M2XXH_RW_STATUS_PROTOCOL_ISO14443A_MAGIC_2	0x63
#define M2XXH_RW_STATUS_PROTOCOL_ISO14443B_MAGIC_1	0x63

/* Values of M2XXH_SET_STATUS_ADDRESS_TIMEOUT_* */
/* ISO 14443 (A-B) : If X is the content of the timeout byte, the actual 
   timeout = X . 380s + 200s */
#define M2XXH_RW_STATUS_TIMEOUT_ISO14443_FROM_uS(us) ((us - 200) / 380)
#define M2XXH_RW_STATUS_TIMEOUT_ISO14443_TO_uS(b) ((b * 380) + 200)
/* ISO 14443 (A-B) : If X is the content of the timeout byte, the actual 
   timeout = (X << 2) . 380s + 200s */
#define M2XXH_RW_STATUS_TIMEOUT_ISO15693_FROM_uS(us) (((us - 200) >> 2) / 380)
#define M2XXH_RW_STATUS_TIMEOUT_ISO15693_TO_uS(b) (((b << 2) * 380) + 200)

int m2xxh_read_status(m2xxh_t* m2xxh, m2xxh_reader_status_t* status)
{
  u_int8_t request[ISO7816_HEADER_SIZE];
  u_int8_t eeprom_or_ram = 
    (status->parameter & M2XXH_PARAMETER_LOCATION_MASK) == M2XXH_PARAMETER_LOCATION_EEPROM ?
    M2XXH_RW_STATUS_LOCATION_EEPROM :
    M2XXH_RW_STATUS_LOCATION_RAM;

  clear_error(m2xxh);

  request[ISO7816_INSTRUCTION_INDEX] = M2XXH_COMMAND_READ_STATUS;
  request[ISO7816_P3_INDEX] = M2XXH_RW_STATUS_P3;

  /* Get state power on from eeprom only */
  if(status->fields & M2XXH_READER_STATUS_STATE_POWER_ON) {
    if(eeprom_or_ram == M2XXH_RW_STATUS_LOCATION_RAM) {
      m2xxh->error = EINVAL;
      RFID_ERROR_STACK(m2xxh->verbose);
      return -1;
    }
    request[ISO7816_P1_INDEX] = M2XXH_RW_STATUS_LOCATION_EEPROM;
    request[ISO7816_P2_INDEX] = M2XXH_RW_STATUS_ADDRESS_STATE_POWER_ON;

    if(handle_request(m2xxh, M2XXH_TYPE_ISO_OUT, request, ISO7816_HEADER_SIZE, &status->state_power_on, 1) < 0) {
      RFID_ERROR_STACK(m2xxh->verbose);
      return -1;
    }
  }

  /* Get communication speed */
  if(status->fields & M2XXH_READER_STATUS_SPEED) {
    request[ISO7816_P1_INDEX] = eeprom_or_ram;
    request[ISO7816_P2_INDEX] = M2XXH_RW_STATUS_ADDRESS_SERIAL_SPEED;

    if(handle_request(m2xxh, M2XXH_TYPE_ISO_OUT, request, ISO7816_HEADER_SIZE, &status->speed, 1) < 0) {
      RFID_ERROR_STACK(m2xxh->verbose);
       return -1;
     }
  }

  /* Get I/O state of output */
  if(status->fields & M2XXH_READER_STATUS_IO_OUTPUT) {
    request[ISO7816_P1_INDEX] = M2XXH_RW_STATUS_LOCATION_IO;
    request[ISO7816_P2_INDEX] = M2XXH_RW_STATUS_ADDRESS_IO_OUTPUT;

    if(handle_request(m2xxh, M2XXH_TYPE_ISO_OUT, request, ISO7816_HEADER_SIZE, &status->io_output, 1) < 0) {
      RFID_ERROR_STACK(m2xxh->verbose);
      return -1;
    }
  }

  /* Get I/O state of input */
  if(status->fields & M2XXH_READER_STATUS_IO_INPUT) {
    request[ISO7816_P1_INDEX] = M2XXH_RW_STATUS_LOCATION_IO;
    request[ISO7816_P2_INDEX] = M2XXH_RW_STATUS_ADDRESS_IO_INPUT;

    if(handle_request(m2xxh, M2XXH_TYPE_ISO_OUT, request, ISO7816_HEADER_SIZE, &status->io_input, 1) < 0) {
      RFID_ERROR_STACK(m2xxh->verbose);
      return -1;
    }
  }

  /* Get protocol */
  if(status->fields & M2XXH_READER_STATUS_PROTOCOL) {
    u_int8_t lsb = 0;
    u_int8_t msb = 0;
    request[ISO7816_P1_INDEX] = eeprom_or_ram;

    request[ISO7816_P2_INDEX] = M2XXH_RW_STATUS_ADDRESS_PROTOCOL_LSB;
    if(handle_request(m2xxh, M2XXH_TYPE_ISO_OUT, request, ISO7816_HEADER_SIZE, &lsb, 1) < 0) {
      RFID_ERROR_STACK(m2xxh->verbose);
      return -1;
    }

    request[ISO7816_P2_INDEX] = M2XXH_RW_STATUS_ADDRESS_PROTOCOL_MSB;
    if(handle_request(m2xxh, M2XXH_TYPE_ISO_OUT, request, ISO7816_HEADER_SIZE, &msb, 1) < 0) {
      RFID_ERROR_STACK(m2xxh->verbose);
      return -1;
    }

    if(lsb == M2XXH_RW_STATUS_PROTOCOL_ISO15693_LSB &&
       msb == M2XXH_RW_STATUS_PROTOCOL_ISO15693_MSB) {
      status->protocol = M2XXH_PROTOCOL_ISO_15693;
    } else if(lsb == M2XXH_RW_STATUS_PROTOCOL_ISO14443A_LSB &&
	      msb == M2XXH_RW_STATUS_PROTOCOL_ISO14443A_MSB) {
      status->protocol = M2XXH_PROTOCOL_ISO_14443A;
    } else if(lsb == M2XXH_RW_STATUS_PROTOCOL_ISO14443B_LSB &&
	      msb == M2XXH_RW_STATUS_PROTOCOL_ISO14443B_MSB) {
      status->protocol = M2XXH_PROTOCOL_ISO_14443B;
    } else {
      status->protocol = M2XXH_PROTOCOL_UNKNOWN;
    }

    /* Get timeouts if the protocol is known. */
    if(status->protocol != M2XXH_PROTOCOL_UNKNOWN &&
       status->fields & M2XXH_READER_STATUS_TIMEOUT_ALL) {
      int i;
      int bit;

      request[ISO7816_P1_INDEX] = eeprom_or_ram;

      for(i = 0, bit = M2XXH_READER_STATUS_TIMEOUT_00;
	  i < M2XXH_READER_STATUS_NTIMEOUT;
	  i++, bit <<= 1) {
	if(status->fields & bit) {
	  u_int8_t byte;
	  request[ISO7816_P2_INDEX] = M2XXH_RW_STATUS_ADDRESS_TIMEOUT_00 + i;

	  if(handle_request(m2xxh, M2XXH_TYPE_ISO_OUT, request, ISO7816_HEADER_SIZE, &byte, 1) < 0) {
	    RFID_ERROR_STACK(m2xxh->verbose);
	    return -1;
	  }

	  if(status->protocol == M2XXH_PROTOCOL_ISO_15693) 
	    status->timeouts[i] = M2XXH_RW_STATUS_TIMEOUT_ISO15693_TO_uS(byte);
	  else
	    status->timeouts[i] = M2XXH_RW_STATUS_TIMEOUT_ISO14443_TO_uS(byte);

	}
      }
    }
    
  }
  
  return 0;
}

#define M2XXH_SET_STATUS_ADDRESS_RESET_1		0x3E
#define M2XXH_SET_STATUS_ADDRESS_RESET_2		0x7E

/*
 * DS_M21x_2-11.pdf, page 14
 */
int m2xxh_set_status(m2xxh_t* m2xxh, m2xxh_reader_status_t* status)
{
  u_int8_t request[ISO7816_HEADER_SIZE];
  u_int8_t nop = '\0';
  u_int8_t eeprom_or_ram =
    (status->parameter & M2XXH_PARAMETER_LOCATION_MASK) == M2XXH_PARAMETER_LOCATION_EEPROM ?
    M2XXH_RW_STATUS_LOCATION_EEPROM :
    M2XXH_RW_STATUS_LOCATION_RAM;

  clear_error(m2xxh);

  request[ISO7816_INSTRUCTION_INDEX] = M2XXH_COMMAND_SET_STATUS;
  request[ISO7816_P3_INDEX] = M2XXH_RW_STATUS_P3;

  if(status->parameter & M2XXH_PARAMETER_RESET_COUPLER) {
    request[ISO7816_P1_INDEX] = M2XXH_PARAMETER_RESET_COUPLER;

    /*
     * DS_M21x_2-11.pdf, page 29 : status code returned for success
     * is *not* 0x9000 but 0x3B00 (mapped to RFID_ERROR_SUCCESS). However, it turns
     * out that 0xE0E0 is returned instead, although this status code
     * is undocumented (it is mapped to EBUSY too as a workaround).
     * Inside technical support says it should not happen and that this
     * behaviour is inconsistent.
     *
     * DS_M21x_2-11.pdf, page 14 : says that *two* reset must be sent,
     * with P2 == 3E and P2 == 7E (the exact meaning of these is not
     * documented and inside technical support say it does not
     * matter). It turns out that sending only the first
     * reset does the job but sending the two does not hurt therefore
     * we comply to the documentation.
     */
    request[ISO7816_P2_INDEX] = M2XXH_SET_STATUS_ADDRESS_RESET_1;
    if(handle_request(m2xxh, M2XXH_TYPE_ISO_IN, request, ISO7816_HEADER_SIZE, &nop, 1) < 0) {
      if(m2xxh->error != RFID_ERROR_SUCCESS) {
	RFID_ERROR_STACK(m2xxh->verbose);
	return -1;
      } else
	clear_error(m2xxh);
    } else {
      RFID_ERROR_STACK(m2xxh->verbose);
      return -1;
    }

#if 0
M2XXH header request : 0x80 0xf4 0x80 0x3e 0x01 
M2XXH ACK
M2XXH write data : 0x00 
M2XXH status : 0xe0e0
M2XXH header request : 0x80 0xf4 0x80 0x7e 0x01 
M2XXH status : 0x0000

  YERK SOMETHING WRONG

    request[ISO7816_P2_INDEX] = M2XXH_SET_STATUS_ADDRESS_RESET_2;
    if(handle_request(m2xxh, M2XXH_TYPE_ISO_IN, request, ISO7816_HEADER_SIZE, &nop, 1) < 0) {
      if(m2xxh->error != RFID_ERROR_SUCCESS) {
	RFID_ERROR_STACK(m2xxh->verbose);
	return -1;
      } else
	clear_error(m2xxh);
    } else {
      RFID_ERROR_STACK(m2xxh->verbose);
      return -1;
    }
#endif

  }

  if(status->parameter & M2XXH_PARAMETER_RESET_MAGNETIC_FIELD) {
    request[ISO7816_P1_INDEX] = M2XXH_PARAMETER_RESET_MAGNETIC_FIELD;
    request[ISO7816_P2_INDEX] = nop;

    if(handle_request(m2xxh, M2XXH_TYPE_ISO_IN, request, ISO7816_HEADER_SIZE, &nop, 1) < 0) {
      RFID_ERROR_STACK(m2xxh->verbose);
      return -1;
    }
  }

  if(status->fields & M2XXH_READER_STATUS_STATE_POWER_ON) {
    if(eeprom_or_ram == M2XXH_RW_STATUS_LOCATION_RAM) {
      m2xxh->error = EINVAL;
      RFID_ERROR_STACK(m2xxh->verbose);
      return -1;
    }
    request[ISO7816_P1_INDEX] = M2XXH_RW_STATUS_LOCATION_EEPROM;
    request[ISO7816_P2_INDEX] = M2XXH_RW_STATUS_ADDRESS_STATE_POWER_ON;

    if(handle_request(m2xxh, M2XXH_TYPE_ISO_IN, request, ISO7816_HEADER_SIZE, &status->state_power_on, 1) < 0) {
      RFID_ERROR_STACK(m2xxh->verbose);
      return -1;
    }
  }

  /* Set communication speed */
  if(status->fields & M2XXH_READER_STATUS_SPEED) {
    /*
     * RS232 does not support this speed.
     */
    if(status->speed == M2XXH_SERIAL_SPEED_424000) {
      m2xxh->error = RFID_ERROR_SPEED;
      RFID_ERROR_STACK(m2xxh->verbose);
      return -1;
    }

    request[ISO7816_P1_INDEX] = eeprom_or_ram;
    request[ISO7816_P2_INDEX] = M2XXH_RW_STATUS_ADDRESS_SERIAL_SPEED;

    if(handle_request(m2xxh, M2XXH_TYPE_ISO_IN, request, ISO7816_HEADER_SIZE, &status->speed, 1) < 0) {
      RFID_ERROR_STACK(m2xxh->verbose);
      return -1;
    }

    /*
     * Reconfigure device according to the new speed.
     */
    {
      int speed;
      switch(status->speed) {
      case M2XXH_SERIAL_SPEED_9600: speed = B9600; break;
      case M2XXH_SERIAL_SPEED_19200: speed = B19200; break;
      case M2XXH_SERIAL_SPEED_38400: speed = B38400; break;
      case M2XXH_SERIAL_SPEED_57600: speed = B57600; break;
      case M2XXH_SERIAL_SPEED_115200: speed = B115200; break;
      default:
	m2xxh->error = EINVAL;
	RFID_ERROR_STACK(m2xxh->verbose);
	return -1;
	break;
      }

      m2xxh->io->speed = speed;
      if(rfid_io_configure(m2xxh->io) < 0) {
	m2xxh->error = errno;
	RFID_ERROR_STACK(m2xxh->verbose);
	return -1;
      }
    }
  }

  /* Set I/O state of output */
  if(status->fields & M2XXH_READER_STATUS_IO_OUTPUT) {
    request[ISO7816_P1_INDEX] = M2XXH_RW_STATUS_LOCATION_IO;
    request[ISO7816_P2_INDEX] = M2XXH_RW_STATUS_ADDRESS_IO_OUTPUT;

    if(handle_request(m2xxh, M2XXH_TYPE_ISO_IN, request, ISO7816_HEADER_SIZE, &status->io_output, 1) < 0) {
      RFID_ERROR_STACK(m2xxh->verbose);
      return -1;
    }
  }

  /* Set I/O state of input */
  if(status->fields & M2XXH_READER_STATUS_IO_INPUT) {
    request[ISO7816_P1_INDEX] = M2XXH_RW_STATUS_LOCATION_IO;
    request[ISO7816_P2_INDEX] = M2XXH_RW_STATUS_ADDRESS_IO_INPUT;

    if(handle_request(m2xxh, M2XXH_TYPE_ISO_IN, request, ISO7816_HEADER_SIZE, &status->io_input, 1) < 0) {
      RFID_ERROR_STACK(m2xxh->verbose);
      return -1;
    }
  }

  if(status->fields & M2XXH_READER_STATUS_PROTOCOL) {
    request[ISO7816_P1_INDEX] = eeprom_or_ram;

    switch(status->protocol) {
    case M2XXH_PROTOCOL_ISO_15693:
      {
	u_int8_t byte;

	request[ISO7816_P2_INDEX] = M2XXH_RW_STATUS_ADDRESS_PROTOCOL_LSB;
	byte = M2XXH_RW_STATUS_PROTOCOL_ISO15693_LSB;
	if(handle_request(m2xxh, M2XXH_TYPE_ISO_IN, request, ISO7816_HEADER_SIZE, &byte, 1) < 0) {
	  RFID_ERROR_STACK(m2xxh->verbose);
	  return -1;
	}

	request[ISO7816_P2_INDEX] = M2XXH_RW_STATUS_ADDRESS_PROTOCOL_MSB;
	byte = M2XXH_RW_STATUS_PROTOCOL_ISO15693_MSB;
	if(handle_request(m2xxh, M2XXH_TYPE_ISO_IN, request, ISO7816_HEADER_SIZE, &byte, 1) < 0) {
	  RFID_ERROR_STACK(m2xxh->verbose);
	  return -1;
	}
      }
      break;
    case M2XXH_PROTOCOL_ISO_14443A: /* Not implemented. */
    case M2XXH_PROTOCOL_ISO_14443B: /* Not implemented. */
    default:
      m2xxh->error = RFID_ERROR_PROTOCOL_UNKNOWN;
      RFID_ERROR_STACK(m2xxh->verbose);
      return -1;
      break;
    }
    
    /*
     * Calculation of the timeout byte depends on the protocol,
     * therefore we must consider this field only when the protocol
     * field is filled.
     */
    if(status->fields & M2XXH_READER_STATUS_TIMEOUT_ALL) {
      int i;
      int bit;

      request[ISO7816_P1_INDEX] = eeprom_or_ram;

      for(i = 0, bit = M2XXH_READER_STATUS_TIMEOUT_00;
	  i < M2XXH_READER_STATUS_NTIMEOUT;
	  i++, bit <<= 1) {
	if(status->fields & bit) {
	  int value;
	  u_int8_t byte;
	  if(status->protocol == M2XXH_PROTOCOL_ISO_15693) {
	    value = M2XXH_RW_STATUS_TIMEOUT_ISO15693_FROM_uS(status->timeouts[i]);
	  } else {
	    value = M2XXH_RW_STATUS_TIMEOUT_ISO14443_FROM_uS(status->timeouts[i]);
	  }

	  if(value > 0xFF) value = 0xFF;
	  if(value <= 0) value = 0x01;

	  request[ISO7816_P2_INDEX] = M2XXH_RW_STATUS_ADDRESS_TIMEOUT_00 + i;

	  byte = value & 0xFF;
	  if(handle_request(m2xxh, M2XXH_TYPE_ISO_IN, request, ISO7816_HEADER_SIZE, &byte, 1) < 0) {
	    RFID_ERROR_STACK(m2xxh->verbose);
	    return -1;
	  }
	}
      }
    }
  }

  return 0;
}

#define M2XXH_TRANSMIT_SEND_MAX			32
#define M2XXH_TRANSMIT_RECEIVE_MAX		35

int m2xxh_transmit(m2xxh_t* m2xxh, u_int8_t param, u_int8_t* buffer_in, int buffer_in_length, u_int8_t* buffer_out, int buffer_out_length)
{
  u_int8_t iso_type;
  int retval;

  clear_error(m2xxh);

  buffer_in[ISO7816_INSTRUCTION_INDEX] = M2XXH_COMMAND_TRANSMIT;
  buffer_in[ISO7816_P1_INDEX] = param;
  buffer_in[ISO7816_P2_INDEX] = buffer_out_length;
  buffer_in[ISO7816_P3_INDEX] = buffer_in_length - ISO7816_HEADER_SIZE;

  if(buffer_in[ISO7816_P2_INDEX] > M2XXH_TRANSMIT_RECEIVE_MAX ||
     buffer_in[ISO7816_P3_INDEX] > M2XXH_TRANSMIT_SEND_MAX) {
    m2xxh->error = RFID_ERROR_TOO_BIG;
    RFID_ERROR_STACK(m2xxh->verbose);
    return -1;
  }

  if(buffer_in[ISO7816_P3_INDEX] == 0) {
    /* Related to the degenerated case used to send an EOF,
       i.e. a TRANSMIT with 0 data bytes sent. */
    iso_type = M2XXH_TYPE_ISO_NONE;
  } else {
    if(param & M2XXH_TRANSMIT_ISO_TYPE_IN_OUT)
      iso_type = M2XXH_TYPE_ISO_IN_OUT;
    else
      iso_type = M2XXH_TYPE_ISO_IN;
  }
  if(iso_type == M2XXH_TYPE_ISO_IN) {
    buffer_out = buffer_in + ISO7816_HEADER_SIZE;
    buffer_out_length = buffer_in_length - ISO7816_HEADER_SIZE;
  }
  
  retval = handle_request(m2xxh, iso_type, buffer_in, buffer_in_length, buffer_out, buffer_out_length);

  /*
   * Related to the degenerated case used to send an EOF: when the EOF
   * is successfull, the 9000 status word is returned instead of the
   * ACK, which is interpreted as an error with no error number where
   * it should be a plain success.
   */
  if(retval < 0 && m2xxh->error == 0) {
    retval = 0;
  }

  return retval;
}

#define M2XXH_GET_RESPONSE_RECEIVE_MAX		35

int m2xxh_get_response(m2xxh_t* m2xxh, u_int8_t* buffer_out, int buffer_out_length)
{
  u_int8_t request[ISO7816_HEADER_SIZE];
  
  clear_error(m2xxh);

  request[ISO7816_INSTRUCTION_INDEX] = M2XXH_COMMAND_GET_RESPONSE;
  request[ISO7816_P1_INDEX] = M2XXH_NOP;
  request[ISO7816_P2_INDEX] = M2XXH_NOP;
  request[ISO7816_P3_INDEX] = buffer_out_length;

  if(request[ISO7816_P3_INDEX] > M2XXH_GET_RESPONSE_RECEIVE_MAX) {
    m2xxh->error = RFID_ERROR_TOO_BIG;
    RFID_ERROR_STACK(m2xxh->verbose);
    return -1;
  }

  return handle_request(m2xxh, M2XXH_TYPE_ISO_OUT, request, ISO7816_HEADER_SIZE, buffer_out, buffer_out_length);
}

#if 0 /* Not yet implemented. */
/*
 * 1: Wait until a card is selected or a character received from the
 * host (e.g. PC).
 *
 * 0: Exit if no card is detected after 3 attempts.
 *
 * Note: When SELECT_CARD uses the option LOOP, the coupler sends
 * ACK=60h after each unsuccessful selection until a card is
 * selected. When a card is selected, '9000h' is returned. In order
 * to stop this scanning, host has to send a byte through the RS232
 * interface.
 *
 */
#define M2XXH_SELECT_CARD_LOOP			0x80
/*
 * 1: Halt card after selection for fast serial numbers capture.
 * 0: No halt after selection.
 */
#define M2XXH_SELECT_CARD_HALT			0x40
/*
 * 1: Increase pre-selection with INSIDE CONTACTLESS anti-collision
 * and a large number of cards.
 *
 * 0: Standard anti-collision (best for 5 cards max.).
 *
 */
#define M2XXH_SELECT_CARD_PRE		0x10
/*
 * 1: Perform a standard INSIDE authentication.
 * Authentication is performed if the key is set as the current key.
 *
 * 0: Do not perform an authentication.
 */
#define M2XXH_SELECT_CARD_AUTH		0x08
/*
 * 1: Authenticates with Debit Key (Kd = Key 1) if AUTH is set.
 * 0: Authenticates with Credit Key (Kc = Key 2) if AUTH is set.
 */
#define M2XXH_SELECT_CARD_KEY		0x04

#endif /* Not yet implemented */

#define M2XXH_SELECT_CARD_OUT_SIZE	9

#define M2XXH_SELECT_CARD_PROTOCOL_INDEX	0
#define M2XXH_SELECT_CARD_SERIAL_INDEX		1

int m2xxh_select_card(m2xxh_t* m2xxh, u_int8_t* serial, u_int8_t* protocol)
{
  u_int8_t request[ISO7816_HEADER_SIZE];
  u_int8_t answer[M2XXH_SELECT_CARD_OUT_SIZE];

  clear_error(m2xxh);

  request[ISO7816_INSTRUCTION_INDEX] = M2XXH_COMMAND_SELECT_CARD;
  request[ISO7816_P1_INDEX] = 0x00;
  request[ISO7816_P2_INDEX] = *protocol;
  request[ISO7816_P3_INDEX] = M2XXH_SELECT_CARD_OUT_SIZE;

  if(handle_request(m2xxh, M2XXH_TYPE_ISO_OUT, request, ISO7816_HEADER_SIZE, answer, M2XXH_SELECT_CARD_OUT_SIZE) < 0) {
    RFID_ERROR_STACK(m2xxh->verbose);
    return -1;
  }

  /*
   * Convert the protocol number returned into a value compatible with
   * the input.
   */
  *protocol = (1 << answer[M2XXH_SELECT_CARD_PROTOCOL_INDEX]);

  memcpy(serial, answer + M2XXH_SELECT_CARD_SERIAL_INDEX, M2XXH_SELECT_CARD_SERIAL_LENGTH);

  return 0;
}

#define M2XXH_DISABLE_COUPLER_P1 0xAD
#define M2XXH_DISABLE_COUPLER_P2 0xBC
#define M2XXH_DISABLE_COUPLER_P3 0xDA

int m2xxh_disable_coupler(m2xxh_t* m2xxh)
{
  u_int8_t request[ISO7816_HEADER_SIZE];
  
  clear_error(m2xxh);

  request[ISO7816_INSTRUCTION_INDEX] = M2XXH_COMMAND_DISABLE_COUPLER;
  request[ISO7816_P1_INDEX] = M2XXH_DISABLE_COUPLER_P1;
  request[ISO7816_P2_INDEX] = M2XXH_DISABLE_COUPLER_P2;
  request[ISO7816_P3_INDEX] = M2XXH_DISABLE_COUPLER_P3;

  return handle_request(m2xxh, M2XXH_TYPE_ISO_NONE, request, ISO7816_HEADER_SIZE, 0, 0);
}

#define M2XXH_ENABLE_COUPLER_P1 0xAE
#define M2XXH_ENABLE_COUPLER_P2 0xDA
#define M2XXH_ENABLE_COUPLER_P3 0xBC

int m2xxh_enable_coupler(m2xxh_t* m2xxh)
{
  u_int8_t request[ISO7816_HEADER_SIZE];
  
  clear_error(m2xxh);

  request[ISO7816_INSTRUCTION_INDEX] = M2XXH_COMMAND_ENABLE_COUPLER;
  request[ISO7816_P1_INDEX] = M2XXH_ENABLE_COUPLER_P1;
  request[ISO7816_P2_INDEX] = M2XXH_ENABLE_COUPLER_P2;
  request[ISO7816_P3_INDEX] = M2XXH_ENABLE_COUPLER_P3;

  return handle_request(m2xxh, M2XXH_TYPE_ISO_NONE, request, ISO7816_HEADER_SIZE, 0, 0);
}

/*
 * Predefined timeouts in micro seconds
 */
#define M2XXH_TIMEOUT_00	1720
#define M2XXH_TIMEOUT_01	4760
#define M2XXH_TIMEOUT_10       24520
#define M2XXH_TIMEOUT_11       39720

static int init_iso15693(m2xxh_t* m2xxh)
{
  m2xxh_reader_status_t status;
  memset(&status, '\0', sizeof(m2xxh_reader_status_t));

  status.fields = M2XXH_READER_STATUS_TIMEOUT_00 | M2XXH_READER_STATUS_TIMEOUT_01 |
    M2XXH_READER_STATUS_TIMEOUT_10 | M2XXH_READER_STATUS_TIMEOUT_11 |
    M2XXH_READER_STATUS_PROTOCOL | M2XXH_READER_STATUS_SPEED;
  status.parameter = M2XXH_PARAMETER_LOCATION_RAM;

  if(m2xxh_read_status(m2xxh, &status) < 0) {
    RFID_ERROR_STACK(m2xxh->verbose);
    return -1;
  }

  if(status.protocol != M2XXH_PROTOCOL_ISO_15693 ||
     status.timeouts[M2XXH_READER_STATUS_TIMEOUT_INDEX_00] != M2XXH_TIMEOUT_00 ||
     status.timeouts[M2XXH_READER_STATUS_TIMEOUT_INDEX_01] != M2XXH_TIMEOUT_01 ||
     status.timeouts[M2XXH_READER_STATUS_TIMEOUT_INDEX_10] != M2XXH_TIMEOUT_10 ||
     status.timeouts[M2XXH_READER_STATUS_TIMEOUT_INDEX_11] != M2XXH_TIMEOUT_11 ||
     status.speed != M2XXH_SERIAL_SPEED_57600) {
    status.protocol = M2XXH_PROTOCOL_ISO_15693;
    status.timeouts[M2XXH_READER_STATUS_TIMEOUT_INDEX_00] = M2XXH_TIMEOUT_00;
    status.timeouts[M2XXH_READER_STATUS_TIMEOUT_INDEX_01] = M2XXH_TIMEOUT_01;
    status.timeouts[M2XXH_READER_STATUS_TIMEOUT_INDEX_10] = M2XXH_TIMEOUT_10;
    status.timeouts[M2XXH_READER_STATUS_TIMEOUT_INDEX_11] = M2XXH_TIMEOUT_11;
    status.speed = M2XXH_SERIAL_SPEED_57600;

    status.parameter = M2XXH_PARAMETER_LOCATION_EEPROM;
    if(m2xxh_set_status(m2xxh, &status) < 0) {
      RFID_ERROR_STACK(m2xxh->verbose);
      return -1;
    }

    status.fields = M2XXH_READER_STATUS_NONE;
    status.parameter = M2XXH_PARAMETER_RESET_COUPLER;
    if(m2xxh_set_status(m2xxh, &status) < 0) {
      RFID_ERROR_STACK(m2xxh->verbose);
      return -1;
    }
  }

  return 0;
}

/*
 * Sending a wrong byte (0x00) as a start of command must trigger
 * the emission of an error message from the reader. If it does not,
 * it means that it is understood as part of a previously unfinished
 * command sequence. If we keep sending wrong bytes, the reader will
 * eventually resynchronize.
 */
static int resync(m2xxh_t* m2xxh)
{

  u_int8_t nul = '\0';
  int ok = 0;
  int i;
  int timeout = m2xxh->io->timeout;
  m2xxh->io->timeout = 100;
  for(i = 0; !ok && i < ISO7816_HEADER_SIZE * 2; i++) {
    if(m2xxh->verbose > 1)
      fprintf(stderr, "M2XXH send null byte\n");
    if(write(m2xxh->io->fd, &nul, 1) != 1) {
      m2xxh->error = errno;
      m2xxh->io->timeout = timeout;
      RFID_ERROR_STACK(m2xxh->verbose);
      return -1;
    }
    status_check(m2xxh, 0 /* nop */, 0 /* nop */);
    if(m2xxh->error == RFID_ERROR_TRY_AGAIN ||
       m2xxh->error == RFID_ERROR_UNKNOWN_ERROR ||
       m2xxh->error == M2XXH_STATUS_COUPLER_RESET_UNKNOWN) {
      if(m2xxh->verbose > 1)
	fprintf(stderr, "M2XXH timeout or unknown error\n");
      /* Timeout or unknown error, just try again. */
    } else if(m2xxh->error == RFID_ERROR_NOT_SUPPORTED || m2xxh->error == 0) {
      /* This is the expected answer. */
      if(m2xxh->verbose > 1)
	fprintf(stderr, "M2XXH synchronized\n");
      ok = 1;
    } else {
      m2xxh->io->timeout = timeout;
      RFID_ERROR_STACK(m2xxh->verbose);
      return -1;
    }
  }

  m2xxh->io->timeout = timeout;

  if(!ok) {
    m2xxh->error = RFID_ERROR_READER_NOT_FOUND;
    RFID_ERROR_STACK(m2xxh->verbose);
    return -1;
  } else
    return 0;
}

#define M2XXH_READER_ID_M210H	0x01

static int m2xxh_probe(m2xxh_t* m2xxh, char** id_stringp, int* idp)
{
  int retval = -1;
  int speeds[] = {
    B57600, B9600, B19200, B38400, B115200
  };
  int speed;

  clear_error(m2xxh);

  if(rfid_io_set(m2xxh->io, B9600, CS8, PARENB, CSTOPB, RFID_IO_LOCK) < 0) {
    m2xxh->error = errno;
    RFID_ERROR_STACK(m2xxh->verbose);
    return -1;
  }

  if(rfid_io_open(m2xxh->io) < 0) {
    m2xxh->error = errno;
    RFID_ERROR_STACK(m2xxh->verbose);
    return -1;
  }

  if(m2xxh->verbose)
    fprintf(stderr, "M2XXH probing %s : ", m2xxh->io->device);
  for(speed = 0; speed < sizeof(speeds)/sizeof(int); speed++) {
    m2xxh->io->speed = speeds[speed];
    if(rfid_io_configure(m2xxh->io) < 0) {
      m2xxh->error = errno;
      RFID_ERROR_STACK(m2xxh->verbose);
      return -1;
    }

    if(m2xxh->verbose) {
      switch(speeds[speed]) {
      case B9600:
	fprintf(stderr, "9600 ");
	break;
      case B19200:
	fprintf(stderr, "19200 ");
	break;
      case B38400:
	fprintf(stderr, "38400 ");
	break;
      case B57600:
	fprintf(stderr, "57600 ");
	break;
      case B115200:
	fprintf(stderr, "115200 ");
	break;
      default:
	fprintf(stderr, "??? (0x%02x) ", speeds[speed]);
	break;
      }
    }

    if(resync(m2xxh) < 0) {
      if(m2xxh->error != RFID_ERROR_READER_NOT_FOUND)
	break;
    } else {
      retval = 0;
      break;
    }
  }

  rfid_io_end(m2xxh->io);

  if(m2xxh->verbose)
    fprintf(stderr, "%s\n", retval < 0 ? "failed" : "found");

  if(retval < 0)
    m2xxh->error = RFID_ERROR_READER_NOT_FOUND;
  else {
    *idp = M2XXH_READER_ID_M210H;
    *id_stringp = "M210H";
  }

  RFID_ERROR_STACK(retval < 0 && m2xxh->verbose);

  return retval;
}

static int m2xxh_init(m2xxh_t* m2xxh)
{
  clear_error(m2xxh);
  m2xxh->iso15693_command_length = 0;
  m2xxh->iso15693_response_length = 0;

  if(rfid_io_set(m2xxh->io, m2xxh->io->speed, CS8, PARENB, CSTOPB, RFID_IO_LOCK) < 0) {
    m2xxh->error = errno;
    RFID_ERROR_STACK(m2xxh->verbose);
    return -1;
  }

  if(rfid_io_open(m2xxh->io) < 0) {
    m2xxh->error = errno;
    RFID_ERROR_STACK(m2xxh->verbose);
    return -1;
  }

  return resync(m2xxh);
}

static void m2xxh_end(m2xxh_t* m2xxh)
{
  if(m2xxh) {
    if(m2xxh->io)
      rfid_io_end(m2xxh->io);
  }
}

/*
 * rfid_reader driver wrappers.
 */
static int drv_probe(struct rfid_reader* reader, const char* drivers)
{
  m2xxh_t* m2xxh = m2xxh_from_reader(reader);
  m2xxh->io = reader->io;
  strcpy(reader->drivers, drivers);
  return m2xxh_probe(m2xxh, &reader->id_string, &reader->id);
}

static int drv_init(struct rfid_reader* reader, const char* drivers)
{
  m2xxh_t* m2xxh = m2xxh_from_reader(reader);
  m2xxh->io = reader->io;
  strcpy(reader->drivers, drivers);
  if(m2xxh_init(m2xxh) < 0) {
    RFID_ERROR_STACK(m2xxh->verbose);
    return -1;
  }

  /*
   * Make sure the RFID reader is properly configured
   * for ISO-15693-3 dialog.
   */
  if(!strncmp("iso15693", drivers, strlen("iso15693")))
    return init_iso15693(m2xxh);

  return 0;
}

static void drv_end(struct rfid_reader* reader)
{
  m2xxh_t* m2xxh = m2xxh_from_reader(reader);
  m2xxh_end(m2xxh);
}

static char* drv_strerror(struct rfid_reader* reader)
{
  m2xxh_t* m2xxh = m2xxh_from_reader(reader);
  return m2xxh_strerror(m2xxh);
}

static int drv_error(struct rfid_reader* reader)
{
  m2xxh_t* m2xxh = m2xxh_from_reader(reader);
  return m2xxh_error(m2xxh);
}

static const char* drv_version(struct rfid_reader* reader)
{
  return "M2XXH " M2XXH_VERSION;
}

static void drv_verbose(struct rfid_reader* reader, int verbosity)
{
  m2xxh_t* m2xxh = m2xxh_from_reader(reader);

  m2xxh->verbose = verbosity;
}

static int drv_iso15693_command(struct rfid_reader* reader, u_int8_t* data, int data_length, const rfid_transponder_t* transponder)
{
  m2xxh_t* m2xxh = m2xxh_from_reader(reader);
  u_int8_t eof_param =
    M2XXH_TRANSMIT_TIMEOUT_00 |
    M2XXH_TRANSMIT_PROTOCOL_ISO_OTHER;
  rfid_reader_t* iso15693_reader = reader->iso15693_reader;
  int retval = -1;
  u_int8_t* request = 0;
  int request_length = 0;
  u_int8_t param =
    M2XXH_TRANSMIT_ISO_TYPE_IN_OUT |
    M2XXH_TRANSMIT_PROTOCOL_ISO_OTHER;

  /*
   * Massage ISO-15693 flags to match m2xxh requirements.
   */
  data[ISO15693_FLAGS_INDEX] |= ISO15693_COMMAND_FLAGS_DATA_RATE;
  data[ISO15693_FLAGS_INDEX] &= ~ISO15693_COMMAND_FLAGS_SUB_CARRIER;
  /*
   * Changing the ISO-15693-3 frame without recalculating the CRC is
   * calling for trouble.
   */
  iso15693_crc16_set(data, data_length);

  request = (u_int8_t*)malloc(ISO7816_HEADER_SIZE + data_length);

  {
    int response_size = 0;
    if(iso15693_response_size(reader->iso15693_reader, data, data_length, &response_size, transponder) < 0) {
      m2xxh->error = iso15693_reader->error_f(iso15693_reader);
      RFID_ERROR_STACK(m2xxh->verbose);
      goto err;
    }
    m2xxh->iso15693_response_length = response_size;

  }

  request_length = ISO7816_HEADER_SIZE + data_length;

  memcpy(request + ISO7816_HEADER_SIZE, data, data_length);

  memcpy(m2xxh->iso15693_command, data, data_length);
  m2xxh->iso15693_command_length = data_length;

  switch(data[ISO15693_COMMAND_INDEX]) {
  case ISO15693_COMMAND_GET_SYSTEM_INFORMATION:
    /*
     * The GET_SYSTEM_INFORMATION command returns a variable number
     * of bytes (the exact number of bytes being specified in the 
     * first byte of the response). The command is therefore transmitted
     * with the maximum possible and the RFID reader will be happy with
     * such a request. 
     */
    param |= M2XXH_TRANSMIT_TIMEOUT_10; /* Assume it's somewhat fast. */
    if(m2xxh_transmit(m2xxh, param, request, request_length, m2xxh->iso15693_response, m2xxh->iso15693_response_length) < 0) {
      RFID_ERROR_STACK(m2xxh->verbose);
      goto err;
    } else if((m2xxh->iso15693_response[ISO15693_FLAGS_INDEX] & ISO15693_RESPONSE_FLAG_ERROR) == 0) {
      u_int8_t info = m2xxh->iso15693_response[ISO15693_INFO_FLAGS_INDEX];
      u_int8_t response_length = ISO15693_INFO_FLAGS_SIZE + ISO15693_INFO_FLAGS_SIZE + ISO15693_UID_SIZE + ISO15693_CRC_SIZE;
      if(info & ISO15693_INFO_FLAGS_DSFID) response_length += ISO15693_DSFID_SIZE;
      if(info & ISO15693_INFO_FLAGS_AFI) response_length += ISO15693_AFI_SIZE;
      if(info & ISO15693_INFO_FLAGS_VICC_MEMORY_SIZE) response_length += ISO15693_VICC_MEMORY_SIZE_SIZE;
      if(info & ISO15693_INFO_FLAGS_IC_REFERENCE) response_length += ISO15693_IC_REFERENCE_SIZE;

      m2xxh->iso15693_response_length = response_length;
    }
    break;

  case ISO15693_COMMAND_INVENTORY:
    /*
     * Implement the low level inventory logic (it should typically be
     * implemented in the RFID reader) as described in ISO/IEC
     * 15693-3:2001(E), page 21. That requires two features :
     *
     * - The ability to distinguish two responses "no transponder
     *   response for this slot" and "collision between two or more transponders
     *   for this slot".
     *
     * - A function to send a EOF, which is not a byte.
     *
     * No transponder for this slot : error with ENOENT or TIMEOUT.
     * Collision : CRC error.
     * Send a EOF : send a TRANSMIT with no data (0x80,0xC2,0x73,0x0A,0x00)
     */
    param |= M2XXH_TRANSMIT_TIMEOUT_10; /* Assume it's blasting fast. */

    {
      int slot = 0;
      int empty_slot = 0;
      int max_slot = (data[ISO15693_FLAGS_INDEX] & ISO15693_COMMAND_FLAGS_NB_SLOTS) ?
	ISO15693_NB_SLOTS_ONE : ISO15693_NB_SLOTS_DEFAULT;
      u_int16_t valid = 0;
      u_int16_t collide = 0;
      u_int8_t* inventory_response = m2xxh->iso15693_response +
	ISO15693_FLAGS_SIZE +
	NOT_ISO15693_VALID_DATA_FLAGS_SIZE +
	NOT_ISO15693_COLLISION_FLAGS_SIZE;
      int inventory_response_length = m2xxh->iso15693_response_length;
      
      /*
       * Use the TRANSMIT/GET_RESPONSE pair instead of standalone TRANSMIT.
       */
      param &= ~M2XXH_TRANSMIT_ISO_TYPE_IN_OUT;

      /*
       * Initial INVENTORY only needs to be sent once. The loop below
       * will ask all transponders to change slots (sending EOF) but
       * will not repeat the INVENTORY command.
       */
      if(m2xxh_transmit(m2xxh, param, request, request_length, 0, inventory_response_length) < 0) {
	if(m2xxh->error == RFID_ERROR_TRANSPONDER_NOT_FOUND)
	  empty_slot = 1;
	else {
	  RFID_ERROR_STACK(m2xxh->verbose);
	  goto err;
	}
      }

      do {
	if(!empty_slot) {
	  if(m2xxh_get_response(m2xxh, inventory_response, inventory_response_length) < 0) {
	    RFID_ERROR_STACK(m2xxh->verbose);
	    goto err;
	  } else {
	    if(iso15693_crc16_check(inventory_response, inventory_response_length)) {
	      /* Success: the ISO15693 frame should be filled as expected. */
	      valid |= (1 << slot);
	    } else {
	      m2xxh->error = RFID_ERROR_ISO15693_CHECKSUM;
	      RFID_ERROR_STACK(m2xxh->verbose);
	      goto err;
	    }
	  }

	  inventory_response += inventory_response_length - ISO15693_CRC_SIZE;
	}

	slot++;
	if(slot < max_slot) {
	  /*
	   * Ask all transponders to switch to next slot by sending
	   * EOF.  
	   */
	  if(m2xxh_transmit(m2xxh, eof_param, request, ISO7816_HEADER_SIZE, 0, inventory_response_length) < 0) {
	    inventory_response[ISO15693_FLAGS_INDEX] = ISO15693_RESPONSE_FLAG_ERROR;
	    switch(m2xxh->error) {
	    case RFID_ERROR_CHECKSUM:
	      /* Collision */
	      collide |= (1 << slot);
	      empty_slot = 1;
	      break;
	    case RFID_ERROR_TRANSPONDER_NOT_FOUND:
	      empty_slot = 1;
	      break;
	    default:
	      RFID_ERROR_STACK(m2xxh->verbose);
	      goto err;
	      break;
	    }
	  } else {
	    empty_slot = 0;
	  }
	}
      } while(slot < max_slot);

      m2xxh->iso15693_response[ISO15693_FLAGS_INDEX] = ISO15693_SUCCESS;
      m2xxh->iso15693_response[NOT_ISO15693_VALID_DATA_FLAGS_LSB_INDEX] = valid & 0xFF;
      m2xxh->iso15693_response[NOT_ISO15693_VALID_DATA_FLAGS_MSB_INDEX] = (valid >> 8) & 0xFF;
      m2xxh->iso15693_response[NOT_ISO15693_COLLISION_FLAGS_LSB_INDEX] = collide & 0xFF;
      m2xxh->iso15693_response[NOT_ISO15693_COLLISION_FLAGS_MSB_INDEX] = (collide >> 8) & 0xFF;
      m2xxh->iso15693_response_length = inventory_response - m2xxh->iso15693_response;
      m2xxh->iso15693_response_length += ISO15693_CRC_SIZE;
      iso15693_crc16_set(m2xxh->iso15693_response, m2xxh->iso15693_response_length);
    }
    break;

  case ISO15693_COMMAND_STAY_QUIET:
    param |= M2XXH_TRANSMIT_TIMEOUT_00; /* Assume it's blasting fast. */
    /*
     * It is expected that STAY_QUIET does not send a response.
     */
    if(m2xxh_transmit(m2xxh, param, request, request_length, m2xxh->iso15693_response, m2xxh->iso15693_response_length) != -1 || m2xxh->error != RFID_ERROR_TRANSPONDER_NOT_FOUND) {
      RFID_ERROR_STACK(m2xxh->verbose);
      goto err;
    }
    break;

  default:
    {
      /*
       * Instruct the command to grab enough bytes to have a well formed
       * error frame, even if it means asking for more bytes than what the
       * transponder will provide in case of success.
       */
      int response_length = m2xxh->iso15693_response_length > ISO15693_ERROR_FRAME_SIZE ? m2xxh->iso15693_response_length : ISO15693_ERROR_FRAME_SIZE;
      param |= M2XXH_TRANSMIT_TIMEOUT_11; /* Assume it's slow. */
      if(m2xxh_transmit(m2xxh, param, request, request_length, m2xxh->iso15693_response, response_length) < 0) {
	RFID_ERROR_STACK(m2xxh->verbose);
	goto err;
      }
    }
    break;
  }

  retval = 0;
 err:
  if(request) free(request);

  return retval;
}

static int drv_iso15693_response(struct rfid_reader* reader, u_int8_t** datap, int* data_lengthp, const rfid_transponder_t* transponder)
{
  m2xxh_t* m2xxh = m2xxh_from_reader(reader);
  int response_size = m2xxh->iso15693_response_length < ISO15693_ERROR_FRAME_SIZE ? ISO15693_ERROR_FRAME_SIZE : m2xxh->iso15693_response_length;
  u_int8_t* response = (u_int8_t*)malloc(response_size);

  if(m2xxh->iso15693_response[ISO15693_FLAGS_INDEX] & ISO15693_RESPONSE_FLAG_ERROR) {
    if(!iso15693_crc16_check(m2xxh->iso15693_response, ISO15693_ERROR_FRAME_SIZE) || m2xxh->iso15693_response[ISO15693_FLAGS_INDEX] == 0xFF) {
      /*
       * If the transponder left the RFID reader range immediately
       * after sending the SOF, the buffer returned will be filled
       * with 0xFF, which is the initialization value of the RFID
       * reader buffer. From the ISO-15693-3 point of view it looks
       * like an error. However, it is not a well formed error and
       * we must detect this case.
       *
       * If the buffer is filled with 0xFF we hit the pathological
       * CRC16 case : 0xFFFF is a valid checksum of 0xFF 0xFF. We
       * can just rely on the checksum to tell us if the error message
       * is garbled, we must specificaly take in account the case
       * where the first byte is 0xFF.
       *
       * If the error message is invalid because of the above,
       * pretend it is a TRANSPONDER_NOT_FOUND error.
       */
      m2xxh->iso15693_response[ISO15693_FLAGS_INDEX] = ISO15693_RESPONSE_FLAG_ERROR;
      m2xxh->iso15693_response[ISO15693_ERROR_INDEX] = NOT_ISO15693_ERROR_TRANSPONDER_NOT_FOUND;
      iso15693_crc16_set(m2xxh->iso15693_response, ISO15693_ERROR_FRAME_SIZE);
      
    }
    
    m2xxh->iso15693_response_length = ISO15693_ERROR_FRAME_SIZE;
  }
  
  memcpy(response, m2xxh->iso15693_response, m2xxh->iso15693_response_length);

  *datap = response;
  *data_lengthp = m2xxh->iso15693_response_length;

  return 0;
}

static int drv_free(rfid_reader_t* reader)
{
  free(reader->opaque);
  free(reader);
  return 0;
}

static int drv_alloc(rfid_reader_t** readerp)
{
  rfid_reader_t* reader = (rfid_reader_t*)malloc(sizeof(rfid_reader_t));
  m2xxh_t* m2xxh = (m2xxh_t*)malloc(sizeof(m2xxh_t));
  memset(m2xxh, '\0', sizeof(m2xxh_t));

  /* Opaque reader information. */
  reader->opaque = m2xxh;

  reader->alloc_f = drv_alloc;
  reader->free_f = drv_free;

  reader->probe_f = drv_probe;
  reader->init_f = drv_init;
  reader->end_f = drv_end;

  reader->strerror_f = drv_strerror;
  reader->error_f = drv_error;
  reader->version_f = drv_version;
  reader->verbose_f = drv_verbose;

  reader->read_f = 0;
  reader->write_f = 0;
  reader->inventory_f = 0;
  reader->iso15693_command_f = drv_iso15693_command;
  reader->iso15693_response_f = drv_iso15693_response;

  reader->transponder_describe_f = 0;
  reader->transponder_cmp_f = 0;
  reader->transponder_null_f = 0;
  reader->transponder_present_f = 0;
  reader->transponder_alloc_f = 0;
  reader->transponder_free_f = 0;
  reader->transponder_clear_f = 0;
  reader->transponder_copy_f = 0;
  reader->transponder_id_get_f = 0;
  reader->transponder_id_set_f = 0;

  reader->buffer_in_max = M2XXH_TRANSMIT_RECEIVE_MAX;
  reader->buffer_out_max = M2XXH_TRANSMIT_SEND_MAX;

  *readerp = reader;

  return 0;
}

int m2xxh_reader_alloc(rfid_reader_t** reader);

int m2xxh_reader_alloc(rfid_reader_t** reader)
{
  return drv_alloc(reader);
}
