/*
 * Copyright (C) 2003 INRIA
 *
 *	INRIA
 *	Domaine de Voluceau
 *	Rocquencourt - B.P. 105
 *	78153 Le Chesnay Cedex - France
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Author: Loic Dachary <loic@gnu.org>
 * 
 */

/*
 * For a detailed description see m2xxh(3)
 */

#ifndef _M2XXH_H_
#define _M2XXH_H_

#include <sys/types.h>

#include <iso15693.h>
#include <rfid_reader.h>

/*
 * Handler holding context for a specific RFID device and used by all
 * m2xxh function.
 */
typedef struct {
  rfid_io_t* io;		/* I/O physical device. */
  int error;			/* Last error, see errno(3) and
				   m2xxh(3) for meaning. */
                                /* Last ISO-15693 command sent. */
  u_int8_t iso15693_command[ISO15693_BUFFER_MAX];   
  int iso15693_command_length;  /* Length of command stored in
				   iso15693_command. */
				/* Cache for last response. */
  u_int8_t iso15693_response[ISO15693_BUFFER_MAX];   
  int iso15693_response_length; /* Length of response stored in
				   iso15693_response. */
  int verbose;			/* Verbosity level. */
} m2xxh_t;

/*
 * Access m2xxh internal structure.
 */
#define m2xxh_from_reader(reader) ((m2xxh_t*)(reader)->private)

#define M2XXH_NOP			0x00

/*
 * Values of the "fields" field 
 */
#define M2XXH_READER_STATUS_SPEED		0x0001
#define M2XXH_READER_STATUS_STATE_POWER_ON	0x0002
#define M2XXH_READER_STATUS_IO_OUTPUT		0x0004
#define M2XXH_READER_STATUS_IO_INPUT		0x0008
#define M2XXH_READER_STATUS_TIMEOUT_00		0x0010
#define M2XXH_READER_STATUS_TIMEOUT_01		0x0020
#define M2XXH_READER_STATUS_TIMEOUT_10		0x0040
#define M2XXH_READER_STATUS_TIMEOUT_11		0x0080
#define M2XXH_READER_STATUS_PROTOCOL		0x0100
#define M2XXH_READER_STATUS_ALL			0xFFFF
#define M2XXH_READER_STATUS_NONE		0x0000

#define M2XXH_READER_STATUS_TIMEOUT_ALL		0x00F0

/*
 * Values of the "state_power_on" field 
 */
#define M2XXH_STATE_POWER_ON_ENABLE	0x01
#define M2XXH_STATE_POWER_ON_DISABLE	0x00

/*
 * Values of the "speed" field 
 */
#define M2XXH_SERIAL_SPEED_9600		0x57
#define M2XXH_SERIAL_SPEED_19200	0x2D
#define M2XXH_SERIAL_SPEED_38400	0x15
#define M2XXH_SERIAL_SPEED_57600	0x0E
#define M2XXH_SERIAL_SPEED_115200	0x06
#define M2XXH_SERIAL_SPEED_424000	0x01

/*
 * Values of the "output" field 
 */
#define M2XXH_IO_OUTPUT_OUT1		0x04
#define M2XXH_IO_OUTPUT_OUT2		0x08

/*
 * Values of the "input" field 
 */
#define M2XXH_IO_INPUT_IN1		0x01

/*
 * Values of the "protocol" field 
 */
#define M2XXH_PROTOCOL_UNKNOWN		0x00
#define M2XXH_PROTOCOL_ISO_14443B	0x01
#define M2XXH_PROTOCOL_ISO_15693	0x02
#define M2XXH_PROTOCOL_ISO_14443A	0x04

/*
 * Values of the "parameter" field 
 */
#define M2XXH_PARAMETER_LOCATION_EEPROM		0x00
#define M2XXH_PARAMETER_LOCATION_RAM		0x03
#define M2XXH_PARAMETER_RESET_COUPLER		0x80
#define M2XXH_PARAMETER_RESET_MAGNETIC_FIELD	0x40

#define M2XXH_PARAMETER_LOCATION_MASK		0x03

/*
 * Size of the "timeouts" array
 */
#define M2XXH_READER_STATUS_NTIMEOUT		4

/*
 * Boundaries of the values of the "timeouts" array,
 * in micro seconds.
 */
#define M2XXH_READER_STATUS_TIMEOUT_MIN		  1720
#define M2XXH_READER_STATUS_TIMEOUT_MAX		400000

/*
 * Symbolic names for "timeouts" array indices
 */
#define M2XXH_READER_STATUS_TIMEOUT_INDEX_00	0
#define M2XXH_READER_STATUS_TIMEOUT_INDEX_01	1
#define M2XXH_READER_STATUS_TIMEOUT_INDEX_10	2
#define M2XXH_READER_STATUS_TIMEOUT_INDEX_11	3

typedef struct {
  int      fields;		/* Bit field of valid fields in the
				   structure. */
  int	   parameter;		/* Parameter P1 of {read,set}_status */
  u_int8_t speed;		/* Host <-> reader communication
				   speed. */
  u_int8_t state_power_on;	/* True if the reader must emit a
				   magnetic field when powered on. */
  u_int8_t io_output;		/* ON/OFF status of the reader output
				   pins. */
  u_int8_t io_input;		/* ON/OFF status of the reader input
				   pin. */
  u_int8_t protocol;		/* Read <-> transponder physical
				   protocol used */
  /*
   * Timeout in microseconds mapped M2XXH_TRANSMIT_TIMEOUT_* and used
   * by the TRANSMIT command.
   */
  int	   timeouts[M2XXH_READER_STATUS_NTIMEOUT];
} m2xxh_reader_status_t;

/*
 * Read the internal state of the RFID reader.
 * DS_M21x_2-11.pdf, page 27.
 */
int m2xxh_read_status(m2xxh_t* m2xxh, m2xxh_reader_status_t* status);

/*
 * Alter the internal state of the RFID reader.
 * DS_M21x_2-11.pdf, page 28.
 */
int m2xxh_set_status(m2xxh_t* m2xxh, m2xxh_reader_status_t* status);

/*
 * INSIDE couplers manage the following protocols :
 *
 * Protocol 0 : ISO 14 443 type B & Inside anticollision (only for
 * INSIDE chip)
 *
 * Protocol 1 : ISO 15 693 & Inside anticollision (only for INSIDE
 * chip)
 *
 * Protocol 2 : ISO 14 443 type A & Inside anticollision (only for
 * INSIDE chip)
 *
 * Protocol 3 : User defined protocol
 *
 * If the bit related to protocol x is set to one, the coupler will
 * run an anticollision sequence using this protocol. If several
 * protocols are selected, the coupler will test all of them, starting
 * from protocol 0 to protocol 3.
 *
 */
#define M2XXH_SELECT_CARD_ISO_1443B_2		0x01
#define M2XXH_SELECT_CARD_ISO_15693_2		0x02
#define M2XXH_SELECT_CARD_ISO_1443A_2		0x04
#define M2XXH_SELECT_CARD_ISO_OTHER		0x08
#define M2XXH_SELECT_CARD_ISO_ALL		0x0F

/*
 * Length in bytes of the serial number identifying a transponder.
 */
#define M2XXH_SELECT_CARD_SERIAL_LENGTH		8

/*
 * Check if a RFID transponder is in the range of the RFID reader and
 * return its serial number and the protocol used to dialog with
 * it. Return on error with RFID_ERROR_TRANSPONDER_NOT_FOUND if no
 * RFID transponder was found. The serial argument must point to a
 * memory area large enough to hold the result
 * (M2XXH_SELECT_CARD_SERIAL_LENGTH).  The protocol argument must be
 * set by the caller to the set of protocols that will be tried by the
 * RFID reader. On return it is set to the actual protocol used.
 *
 * DS_M21x_2-11.pdf, page 20
 */
int m2xxh_select_card(m2xxh_t* m2xxh, u_int8_t* serial, u_int8_t* protocol);

/*
 * 1: The coupler automatically sends the CRC (function of the Data
 * bytes) to the chip. Coupler uses the CRC associated to the choosen
 * protocol (bit 1 & 0) 
 * 0: Only P3 data bytes are sent.
 */
#define M2XXH_TRANSMIT_SEND_CRC			0x80
/*
 * 1: Compares the returned CRC with the expected value calculated by
 * the coupler (verify the data sent by the chip).  
 * 0: CRC is not checked.
 */
#define M2XXH_TRANSMIT_CHECK_CRC		0x40
/*
 * The time out value depends of the protocol used (see also
 * m2xxh_set_status).  The time out is the time from the commands EOF
 * (End Of Frame) to the chip response SOF (Start of Frame).
 */
#define M2XXH_TRANSMIT_TIMEOUT_00		0x00
#define M2XXH_TRANSMIT_TIMEOUT_01		0x10
#define M2XXH_TRANSMIT_TIMEOUT_10		0x20
#define M2XXH_TRANSMIT_TIMEOUT_11		0x30
/*
 * 1 : Communication is ISO IN-OUT. Coupler send back the data as soon
 * as it receives chip answer.
 *
 * 0 : Commucation between HOST and coupler follows the ISO 7816 T=0
 * protocol.  Thus TRANSMIT command is only ISO IN, and user has to
 * use the GET REPONSE command to retrieve chip DATA from the coupler.
 */
#define M2XXH_TRANSMIT_ISO_TYPE_IN_OUT		0x04
/*
 * Defines the contactless communication protocol number to be
 * used. When coupler's EEPROM is set with the default values, the
 * protocol types are as follows:
 *
 *           Contactless Communication Protocol
 *           00      ISO 14443 B PICO family chips
 *           01      ISO 15693 PICO family chips
 *           10      ISO 14443 A PICO family chips
 *           11      other ISO
 * 
 * The 11 protocol (ISO_OTHER) can be programmed using m2xxh_set_status.
 * 
 */
#define M2XXH_TRANSMIT_PROTOCOL_ISO_14443B_2	0x00
#define M2XXH_TRANSMIT_PROTOCOL_ISO_15693_2	0x01
#define M2XXH_TRANSMIT_PROTOCOL_ISO_14443A_2	0x02
#define M2XXH_TRANSMIT_PROTOCOL_ISO_OTHER	0x03

/*
 * Send the buffer_in_length bytes from buffer_in to the transponders
 * currently in the range of the reader. The physical protocol used to
 * send the message is defined in the param argument. The response from
 * the transponder is stored in the buffer_out array and is expected to be
 * exactly buffer_out_length bytes long.
 *
 * If the M2XXH_TRANSMIT_ISO_TYPE_IN_OUT bit of the param argument is cleared,
 * the response will not be returned in buffer_out. Instead the caller must
 * call m2xxh_get_response to retrieve the answer.
 *
 * DS_M21x_2-11.pdf, page 24
 */
int m2xxh_transmit(m2xxh_t* m2xxh, u_int8_t param, u_int8_t* buffer_in, int buffer_in_length, u_int8_t* buffer_out, int buffer_out_length);

/*
 * Retrieve buffer_out_length bytes from the RFID reader buffer and copy
 * them in the buffer_out array. Typically used after a m2xxh_transmit.
 *
 * DS_M21x_2-11.pdf, page 26
 */
int m2xxh_get_response(m2xxh_t* m2xxh, u_int8_t* buffer_out, int buffer_out_length);

/*
 * The RFID reader goes in sleep mode (low power consumption) and the
 * carrier is desactivated.  The RFID reader will not respond to any
 * command except the ENABLE_COUPLER command. (coupler == RFID reader).
 *
 * DS_M21x_2-11.pdf, page 31
 */
int m2xxh_disable_coupler(m2xxh_t* m2xxh);

/*
 * The carrier is activated and the RFID reader leaves sleep mode.
 * This command can only be used after a DISABLE_COUPLER command or if
 * the coupler is desactivated at power on (see state_power_on in
 * m2xxh_set_status).
 *
 * DS_M21x_2-11.pdf, page 32
 */
int m2xxh_enable_coupler(m2xxh_t* m2xxh);

#endif /* _M2XXH_H_ */
