/*  Copyright (C) 2011 Ben Asselstine

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU Library General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
  02110-1301, USA.
*/
#include <config.h>
#include <stdlib.h>
#include <glib.h>
#include <unistd.h>
#include <string.h>
#include "gettext-more.h"
#include "xvasprintf.h"
#include "read-file.h"
#include "reddit_priv.h"
#include "whoami.h"

struct argp reddit_whoami_argp = { NULL, noargs_parse_opt, "", N_("Show your profile information.") };

int 
reddit_whoami_parse_argp (struct reddit_state_t *state, int argc, char **argv, struct argp *argp)
{
  int err = 0;
  int parsing_errors;
  err = argp_parse (argp, argc, argv, ARGP_NO_EXIT,  0, &parsing_errors);
  if (!err && !parsing_errors && !argp_help_check (argc, argv))
    return reddit_whoami (state);
  else
    return 0;
}

static int
reddit_get_stats_from_data (struct reddit_state_t *state, char *data, int *link_karma, int *comment_karma, char **age)
{
  gchar **fields = g_regex_split_simple ("<span class=\"karma\"\\>", data, 0, 
                                         0);
  int num_fields =  g_strv_length (fields);
  if (num_fields < 3)
    {
      free (data);
      return -1;
    }
  *link_karma = atoi (fields[1]);
  *comment_karma = atoi (fields[2]);
  g_strfreev (fields);
  fields = g_regex_split_simple ("<span class=\"age\"\\>", data, 0, 0);
  if (num_fields < 2)
    {
      free (data);
      return -2;
    }
  char *lt = strchr (fields[1], '<');
  if (!lt)
    {
      free (data);
      return -3;
    }
  *lt = '\0';
  *age = strdup (fields[1]);
  g_strfreev (fields);
  free (data);
  return 0;
}

static int
reddit_get_stats_from_file (struct reddit_state_t *state, FILE *fileptr, 
                            int *link_karma, int *comment_karma, char **age)
{
  if (!fileptr)
    return 0;
  size_t data_len = 0;
  char *data = fread_file (fileptr, &data_len);
  fclose (fileptr);
  return reddit_get_stats_from_data (state, data, link_karma, comment_karma,
                                     age);
}

static int
reddit_get_stats (struct reddit_state_t *state, char *link, int *link_karma, int *comment_karma, char **age, int *resp)
{
  int err = 0;
  char tmp[sizeof(PACKAGE) + 13];
  snprintf (tmp, sizeof tmp, "/tmp/%s.XXXXXX", PACKAGE);
  int fd = mkstemp(tmp);
  close (fd);
  FILE *fileptr = fopen (tmp, "wb");
  curl_easy_setopt (state->curl, CURLOPT_URL, link);
  curl_easy_setopt (state->curl, CURLOPT_WRITEDATA, fileptr);
  curl_easy_perform(state->curl);
  fflush (fileptr);
  fsync (fileno (fileptr));
  fclose (fileptr);
  long response = 0;
  curl_easy_getinfo (state->curl, CURLINFO_RESPONSE_CODE, &response);

  if (response == 200)
    err = reddit_get_stats_from_file (state, fopen (tmp, "r"), link_karma,
                                      comment_karma, age);
  else
    err = -1;
  if (resp)
    *resp = response;
  remove (tmp);
  return err;
}

int 
reddit_whoami(struct reddit_state_t *state)
{
  if (state->logged_in == 0)
    {
      reddintf (state, _("You are not logged in."
                         "  Login using the `login' command.\n"));
      return 0;
    }
  int response = 0;
  int err = 0;
  char *link = xasprintf ("http://%s/user/%s/", state->site, state->username);
  int link_karma = 0;
  int comment_karma = 0;
  char *age = NULL;
  err = reddit_get_stats (state, link, &link_karma, &comment_karma, &age, 
                          &response);
  free (link);
  if (err)
    {
      reddintf (state, _("Error: Couldn't get the profile page.  "
                         "(HTTP /1.1 %d err=%d)\n"), response, err);
      err = 0;
    }
  else
    {
      reddintf (state, "%s\n", state->username);
      reddintf (state, _("Link Karma: %d\n"), link_karma);
      reddintf (state, _("Comment Karma: %d\n"), comment_karma);
      reddintf (state, "%s\n", age);
      reddintf (state, "\n");
    }
  free (age);
  return err;
}
