/*  Copyright (C) 2011 Ben Asselstine

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU Library General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
  02110-1301, USA.
*/
#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <string.h>
#include <unistd.h>
#include <stdarg.h>
#include <argz.h>
#include <argp.h>
#include "opts.h"
#include "xvasprintf.h"
#include "trim.h"
#ifdef HAVE_LIBREADLINE
#include <readline/readline.h>
#include <readline/history.h>
#endif
#include "gettext-more.h"
#include "read-file.h"
#include <curl/curl.h>
#include "reddit.h"
#include "reddit_priv.h"
#include "stories.h"

void 
reddit_line (struct reddit_state_t *state)
{
  if (state->banner == 0)
    return;
  int i = 0;
  int columns = 80;
  if (state->columns != 0)
    columns = state->columns;
  for (;i < columns; i++)
    fprintf (state->out, "-");
  fprintf (state->out, "\n");
}

void 
reddit_banner (struct reddit_state_t *state, char *title)
{
  if (state->banner == 0)
    return;
  char *edition = title;
  if (edition == NULL)
    edition = _("Canadian Edition!");
  if (state->columns < 40 && state->columns > 0)
    {
      fprintf (state->out, "    _o    \n");
      fprintf (state->out, "  _/_     \n");
      fprintf (state->out, " (o_o)    REDDIT\n [%s]\n", edition);
    }
  else if (state->columns < 80 && state->columns > 0)
    {
      fprintf (state->out, "    _o    \n");
      fprintf (state->out, "  _/_     \n");
      fprintf (state->out, " (o_o)    REDDIT  [%s]\n", edition);
    }
  else
    {
      fprintf (state->out, "    _o    ____  ____  ___   ___   _  ___\n");
      fprintf (state->out, "  _/_     |__/  |___  |  \\  |  \\  |   |\n");
      fprintf (state->out, " (o_o)    |  \\  |___  |__/  |__/  |   |  [%s]\n",
               edition);
    }
  fprintf (state->out, "\n");
  char *fmt; 
  if (state->columns < 40 && state->columns > 0)
    fmt = " [%s] [%s]\n" 
          " [%s] [%s]\n"
          " [%s]\n";
  else if (state->columns < 80 && state->columns > 0)
    fmt = "   [%s]  [%s]\n" 
          "   [%s]  [%s]  [%s]\n";
  else
    fmt = "     [%s]    [%s]    [%s]    [%s]    [%s]\n";
  int whats = state->whats;
  if (state->search)
    whats = -1;
  fprintf (state->out, fmt, 
           whats == REDDIT_WHATS_HOT ? _("what's hot") : _("hot"), 
           whats == REDDIT_WHATS_NEW ? _("what's new") : _("new"), 
           whats == REDDIT_WHATS_CONTROVERSIAL ? 
           _("most controversial") :_("controversial"), 
           whats == REDDIT_WHATS_TOP ? _("top scoring") : _("top"),
           whats == REDDIT_WHATS_SAVED ? _("*saved*") : _("saved")
           );
  fprintf (state->out, "\n");
  reddit_line (state);
}

int
reddit_history(struct reddit_state_t *state)
{
#ifdef HAVE_LIBREADLINE
  if (!history_is_stifled())
    fprintf (state->out, "%d\n", -1);
  else
    fprintf (state->out, "%d\n", history_max_entries);
#else
  fprintf (state->out, "0\n");
#endif
  return 0;
}

int
reddit_browser (struct reddit_state_t *state)
{
  fprintf (state->out, "%s\n", state->browser_binary);
  return 0;
}

int
reddit_set_browser (struct reddit_state_t *state, char *value)
{
  free (state->browser_binary);
  FILE *fileptr = fopen (value, "r");
  if (fileptr)
    {
      state->browser_binary = strdup (value);
      fclose (fileptr);
    }
  else
    reddintf (state, _("The program %s does not exist.\n"), value);
  return 0;
}

int
reddit_browser_options (struct reddit_state_t *state)
{
  fprintf (state->out, "%s\n", state->browser_options);
  return 0;
}

int
reddit_set_browser_options (struct reddit_state_t *state, char *value)
{
  free (state->browser_options);
  state->browser_options = strdup (value);
  return 0;
}

int
reddit_warranty (struct reddit_state_t *state)
{
  fprintf (state->out, "%s %s\n", PACKAGE, VERSION);
  fprintf (state->out, "Copyright (C) 2011 Ben Asselstine\n");
  fprintf (state->out, "\n");
  fprintf (state->out, _("\
    This program is free software; you can redistribute it and/or modify\n\
    it under the terms of the GNU General Public License as published by\n\
    the Free Software Foundation; either version 2 of the License, or\n\
    (at your option) any later version.\n\
\n\
    This program is distributed in the hope that it will be useful,\n\
    but WITHOUT ANY WARRANTY; without even the implied warranty of\n\
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n\
    GNU General Public License for more details.\n\
\n\
    You should have received a copy of the GNU General Public License \n\
    along with this program. If not, write to\n\
\n\
    The Free Software Foundation, Inc.\n\
    51 Franklin Street, Fifth Floor\n\
    Boston, MA 02110-1301  USA\n\n\
"));
  return 0;
}

void
reddit_unknown_command (struct reddit_state_t *state)
{
  reddintf (state, _("Unknown command `%s'\n"), state->command);
}

int
argp_help_check (int argc, char **argv)
{
  int i = 0;
  for (; i < argc; i++)
    {
      if (strcmp (argv[i], "-?") == 0)
        return 1;
      if (strcmp (argv[i], "--help") == 0 || strcmp (argv[i], "--usage") == 0)
        return 1;
    }
  return 0;
}


void
reddit_reshow_stories (struct reddit_state_t *state)
{
  reddit_show_stories (state);
  if (state->search)
    reddit_banner (state, "search results");
  else if (!reddit_is_in_subreddit (state))
    reddit_banner (state, state->subreddit);
  else
    reddit_banner (state, NULL);
}

int
reddit_set_banner (struct reddit_state_t *state, char *value)
{
  if (strcasecmp (value, "off") == 0)
    state->banner = 0;
  else if (strcasecmp (value, "on") == 0)
    state->banner = 1;
  else
    reddit_unknown_command (state);
  return 0;
}

#ifndef HAVE_LIBREADLINE
int
reddit_set_history (struct reddit_state_t *state, char *value)
{
  reddintf (state, _("Error: %s was not compiled with GNU readline.  "
                     "A history of commands cannot be kept.\n"), PACKAGE);
  return 0;
}

#else 

int
reddit_set_history (struct reddit_state_t *state, char *value)
{
  char *end = NULL;
  long int new_history_length = strtol (value, &end, 0);
  if ((end == NULL) || (end == value))
    reddintf (state, _("Error `%s' isn't a number.\n"), value);
  else if (new_history_length < -1)
    reddintf (state, _("Error `%s' is an invalid value.\n"), value);
  else
    {
      if (new_history_length == -1 && history_is_stifled())
        unstifle_history();
      else if (new_history_length == -1 && !history_is_stifled())
        ;
      else
        stifle_history (new_history_length);
    }
  return 0;
}
#endif

char *
reddit_construct_url (struct reddit_state_t *state, char *page, int w, char *subreddit, char *search)
{
  char *mysubreddit = subreddit;
  if (mysubreddit == NULL)
    mysubreddit = state->subreddit;
  char *mysearch = search;
  if (mysearch == NULL)
    mysearch = state->search;
  char *mypage = page;
  if (mypage == NULL)
    mypage = state->page;
  char *whats = NULL;
  int mywhats = w;
  if (mywhats == -1)
    mywhats = state->whats;
  if (mywhats != REDDIT_WHATS_HOT)
    whats = reddit_whats_to_string (mywhats);
  char *mypagesize = NULL;
  if (state->page_size)
    mypagesize = xasprintf ("limit=%d", state->page_size);
  char *url;
  if (mysearch && *mysearch)
    {
      char *encoded_search = curl_easy_escape (state->curl, mysearch, 0);
      url = xasprintf ("http://%s/search?q=%s%s%s%s%s", state->site, 
                       encoded_search,
                       mypage ? "&" : "",
                       mypage ? mypage : "",
                       mypagesize ? "&" : "",
                       mypagesize ? mypagesize : "");
      curl_free (encoded_search);
    }
  else if (mywhats == REDDIT_WHATS_SAVED)
    url = xasprintf ("http://%s/saved/%s%s%s%s", state->site, 
                     mypage || mypagesize ? "?" : "",
                     mypage ? mypage : "",
                     mypage && mypagesize ? "&" : "",
                     mypagesize ? mypagesize : "");
  else
    url = xasprintf ("http://%s/%s%s%s%s%s%s%s%s%s", state->site,
                     mysubreddit && *mysubreddit ? "r/" : "",
                     mysubreddit && *mysubreddit ? mysubreddit : "",
                     mysubreddit && *mysubreddit ? "/" : "",
                     whats ? whats : "",
                     whats ? "/" : "",
                     mypage || mypagesize ? "?" : "",
                     mypage ? mypage : "",
                     mypage && mypagesize ? "&" : "",
                     mypagesize ? mypagesize : "");
  free (mypagesize);
  return url;
}

int 
reddit_load_and_show_stories (struct reddit_state_t *state, char *page, int whats, char *subreddit, char *search, int *response)
{
  int err = 0;
  struct reddit_story_t *stories = NULL;
  char *url = reddit_construct_url (state, page, whats, subreddit, search);
  int oldwhats = state->whats;
  char *oldpage = state->page;
  char *oldsubreddit = state->subreddit;
  char *oldsearch = state->search;
  if (whats != -1)
    state->whats = whats;
  if (page != NULL)
    state->page = page;
  if (subreddit != NULL)
    state->subreddit = subreddit;
  if (search != NULL)
    state->search = search;

  int num_stories = reddit_load_stories (state, url, &stories, response);

  state->whats = oldwhats;
  state->page = oldpage;
  state->subreddit = oldsubreddit;
  state->search = oldsearch;

  if (num_stories <= 0)
    {
      free (url);
      free (stories);
      if (num_stories == 0)
        err = -1;
      else
        err = num_stories;
    }
  else
    {
      free (url);
      reddit_free_stories (state->stories, state->num_stories);
      state->num_stories = num_stories;
      state->stories = stories;
      if (search && search[0] == '\0')
        {
          free (state->search);
          state->search = NULL;
        }
      else if (search && search[0] != '\0')
        {
          free (state->search);
          state->search = strdup (search);
        }
      if (page && *page != 0)
        {
          free (state->page);
          state->page = strdup (page);
        }
      if (whats != -1)
        state->whats = whats;

      if ((subreddit && subreddit[0] == '\0') || 
          (subreddit && strcmp (subreddit, "all") == 0))
        {
          free (state->subreddit);
          state->subreddit = NULL;
        }
      else if (subreddit)
        {
          free (state->subreddit);
          state->subreddit = strdup (subreddit);
        }
      reddit_show_stories (state);

      if (reddit_is_in_subreddit (state))
        reddit_banner (state, state->subreddit);
      else
        reddit_banner (state, NULL);
    }
  return err;
}

int
reddit_update_modhash (struct reddit_state_t *state)
{
  struct reddit_story_t *stories = NULL;
  char *url = reddit_construct_url (state, NULL, -1, NULL, NULL);
  int num_stories = reddit_load_stories (state, url, &stories, NULL);
  free (url);
  if (num_stories > 0)
    reddit_free_stories (stories, num_stories);
  else
    return -1;
  return 0;
}

int
reddit_post (struct reddit_state_t *state, char *action, char *post, int *resp, char **msg)
{
  int err = 0;
  char tmp[sizeof(PACKAGE) + 13];
  snprintf (tmp, sizeof tmp, "/tmp/%s.XXXXXX", PACKAGE);

  int fd = mkstemp(tmp);
  close (fd);
  FILE *fileptr = fopen (tmp, "wb");
  char *url = xasprintf ("http://%s/api/%s", state->site, action);
  curl_easy_setopt (state->curl, CURLOPT_URL, url);
  curl_easy_setopt (state->curl, CURLOPT_POSTFIELDS, post);
  curl_easy_setopt (state->curl, CURLOPT_POSTFIELDSIZE, (long)strlen (post));
  curl_easy_setopt (state->curl, CURLOPT_WRITEDATA, fileptr);
  curl_easy_perform(state->curl);
  free (url);
  fflush (fileptr);
  fclose (fileptr);
  long response = 0;
  curl_easy_getinfo (state->curl, CURLINFO_RESPONSE_CODE, &response);
  size_t data_len = 0;
  char *text = read_file (tmp, &data_len);
  char *pass1 = htmltotext (state, text);
  char *pass2 = htmltotext (state, pass1);
  *msg = strdup (pass2);
  free (pass2);
  free (pass1);
  free (text);
  remove (tmp);
  if (resp)
    *resp = response;
  return err;
}

int reddintf (struct reddit_state_t *state, char *fmt, ...)
{
  if (state->silent)
    return 0;
  va_list ap;
  va_start (ap, fmt);
  int r = vfprintf (state->out, fmt, ap);
  va_end(ap);
  return r;
}

error_t 
noargs_parse_opt (int key, char *arg, struct argp_state *state)
{

  int *parsing_errors = (int*) state->input;
  switch (key)
    {
    case ARGP_KEY_ARG:
        {
          if (!*parsing_errors)
            {
              argp_failure (state, 0, 0, 
                            _("This command doesn't take an argument."));
              *parsing_errors = 1;
            }
        }
      break;
    case ARGP_KEY_INIT:
      *parsing_errors = 0;
      break;
    default:
      return ARGP_ERR_UNKNOWN;
    }
  return 0;
}

int
reddit_page_size (struct reddit_state_t *state)
{
  reddintf (state, "%d\n", state->page_size);
  return 0;
}

int
reddit_set_page_size (struct reddit_state_t *state, char *value)
{
  char *end = NULL;
  long int new_page_size = strtol (value, &end, 0);
  if ((end == NULL) || (end == value))
    reddintf (state, _("Error `%s' isn't a number.\n"), value);
  else if (new_page_size < 1)
    state->page_size = 0;
  else
    state->page_size = new_page_size;
  return 0;
}
