/*  Copyright (C) 2011 Ben Asselstine

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU Library General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
  02110-1301, USA.
*/
#include <config.h>
#include <stdlib.h>
#include <argz.h>
#include <string.h>
#include "gettext-more.h"
#include "xvasprintf.h"
#include "reddit_priv.h"
#include "grep.h"

static struct argp_option options[] = 
{
    {"nfsw", 'n', "1|0", OPTION_ARG_OPTIONAL, 
      N_("include/exclude stories that are \"not safe for work\"") },
    {"subreddit", 'r', "NAME", 0, 
      N_("return stories from the given subreddit") },
    {"site", 's', "DOMAIN", 0, N_("return stories from the given website") },
    {"url", 'u', "TEXT", 0, N_("search for TEXT in links") },
    {"selftext", 't', "TEXT", 0, 
      N_("search for TEXT in self-posts") },
    {"is-self", 'i', "1|0", OPTION_ARG_OPTIONAL, 
      N_("include/exclude stories from self-posts") },
    {"author", 'a', "USERNAME", 0, 
      N_("search for stories submitted by USERNAME") },
    {"local", 'l', NULL, 0, 
      N_("search for stories in the active subreddit") },
    {0}
};

static error_t 
parse_opt (int key, char *arg, struct argp_state *state)
{
  struct reddit_grep_options_t *opt = NULL;
  if (state)
    opt = (struct reddit_grep_options_t*) state->input;
  switch (key)
    {
    case 'l':
      if (opt->parsing_errors == 0)
        opt->search_in_this_reddit = 1;
      break;
    case 'a':
      if (opt->parsing_errors == 0)
        argz_add (&opt->authors, &opt->authors_len, arg);
      break;
    case 'n':
        {
          if (opt->parsing_errors == 0)
            {
              if (arg)
                {
                  char *end = NULL;
                  unsigned long val = strtoul (arg, &end, 0);
                  if (end == NULL || end == arg || (int)val < 0)
                    opt->nsfw = val;
                  else
                    {
                      argp_failure 
                        (state, 0, 0, 
                         _("The argument `%s' doesn't look like a number."), 
                         arg);
                      opt->parsing_errors = 1;
                    }
                }
              else
                opt->nsfw = 1;
            }
        }
      break;
    case 'i':
        {
          if (opt->parsing_errors == 0)
            {
              if (opt->state->logged_in)
                {
                  if (arg)
                    {
                      char *end = NULL;
                      unsigned long val = strtoul (arg, &end, 0);
                      if (end == NULL || end == arg || (int)val < 0)
                        opt->is_self = val;
                      else
                        {
                          argp_failure 
                            (state, 0, 0, 
                             _("The argument `%s' doesn't look like a number."), 
                             arg);
                          opt->parsing_errors = 1;
                        }
                    }
                  else
                    opt->is_self = 1;
                }
              else
                {
                  argp_failure (state, 0, 0, 
                                _("You need to login to use the -i option.  "
                                  "Login using the `login' command."));
                  opt->parsing_errors = 1;
                }
            }
        }
      break;
    case 'r':
      if (opt->parsing_errors == 0)
        argz_add (&opt->reddits, &opt->reddits_len, arg);
      break;
    case 's':
      if (opt->parsing_errors == 0)
        {
          if (opt->state->logged_in)
            argz_add (&opt->sites, &opt->sites_len, arg);
          else
            {
              argp_failure (state, 0, 0, 
                            _("You need to login to use the -s option.  "
                              "Login using the `login' command."));
              opt->parsing_errors = 1;
            }
        }
      break;
    case 'u':
      if (opt->parsing_errors == 0)
        argz_add (&opt->urls, &opt->urls_len, arg);
      break;
    case 't':
      if (opt->parsing_errors == 0)
        argz_add (&opt->selftext, &opt->selftext_len, arg);
      break;
    case ARGP_KEY_ARG:
      if (!opt->parsing_errors)
        argz_add (&opt->search_terms,  &opt->search_terms_len, arg);
      break;
    case ARGP_KEY_INIT:
      opt->parsing_errors = 0;
      opt->search_terms = NULL;
      opt->search_terms_len = 0;
      opt->reddits = NULL;
      opt->reddits_len= 0;
      opt->urls = NULL;
      opt->urls_len = 0;
      opt->sites = NULL;
      opt->sites_len = 0;
      opt->selftext = NULL;
      opt->selftext_len = 0;
      opt->authors = NULL;
      opt->authors_len = 0;
      opt->search_in_this_reddit = -1;
      opt->nsfw = -1;
      opt->is_self = -1;
      break;
    default:
      return ARGP_ERR_UNKNOWN;
    }
  return 0;
}

static char * 
help_filter (int key, const char *text, void *input)
{
  struct reddit_grep_options_t *o = (struct reddit_grep_options_t*) input;
  if (key == ARGP_KEY_HELP_POST_DOC && o)
    {
      char *s1;
      if (reddit_is_in_subreddit (o->state))
        s1 = xasprintf (_("  The active subreddit is `%s'."), 
                        o->state->subreddit);
      else
        s1 = strdup (_("  The --local option is ignored because "
                       "there is currently no active subreddit."));
      char *new_text = xasprintf (text, s1);
      free (s1);
      return new_text;
    }
  return (char *) text;
}

struct argp reddit_grep_argp = { options, parse_opt, "[SEARCH-TERMS...]", 
  N_("Search the site for stories.\v"
     "So-called \"self-posts\" are reddit stories that do-not link externally "
     "but instead link to themselves.%s"), 0, help_filter };

int 
reddit_grep_parse_argp (struct reddit_state_t *state, int argc, char **argv, struct argp *argp)
{
  int err = 0;
  struct reddit_grep_options_t grep_options;
  grep_options.state = state;
  err = argp_parse (argp, argc, argv, ARGP_NO_EXIT,  0, &grep_options);
  if (!err && !grep_options.parsing_errors && !argp_help_check (argc, argv))
    {
      if (grep_options.search_terms_len == 0 && 
          grep_options.reddits_len == 0 &&
          grep_options.sites_len == 0 &&
          grep_options.urls_len == 0 &&
          grep_options.selftext_len == 0 &&
          grep_options.authors_len == 0)
        {
          reddintf (state, _("Try `grep --help' or `grep --usage' "
                             "for more information.\n"));
          return 0;
        }
      else
        return reddit_grep (state, &grep_options);
    }
  else
    return 0;
}

static char *
generate_search (struct reddit_state_t *state, struct reddit_grep_options_t *options)
{
  char *prev = NULL;
  char *line;
  argz_stringify (options->search_terms, options->search_terms_len, ' ');
  char *search = strdup (options->search_terms);
  if (options->nsfw > -1)
    {
      char *s1 = xasprintf ("%s over18:%s", search, 
                            options->nsfw ? "yes" : "no");
      free (search);
      search = s1;
    }
  if (options->is_self > -1)
    {
      char *s1 = xasprintf ("%s is_self:%s", search, 
                            options->is_self? "yes" : "no");
      free (search);
      search = s1;
    }
  while ((line = argz_next(options->reddits, options->reddits_len, prev)))
    {
      prev = line;
      char *s1 = xasprintf ("%s reddit:\"%s\"", search, line);
      free (search);
      search = s1;
    }
  if (options->search_in_this_reddit && reddit_is_in_subreddit(state))
    {
      char *s1 = xasprintf ("%s site:\"%s\"", search, state->subreddit);
      free (search);
      search = s1;
    }
  prev = NULL;
  while ((line = argz_next(options->sites, options->sites_len, prev)))
    {
      prev = line;
      char *s1 = xasprintf ("%s site:\"%s\"", search, line);
      free (search);
      search = s1;
    }
  prev = NULL;
  while ((line = argz_next(options->urls, options->urls_len, prev)))
    {
      prev = line;
      char *s1 = xasprintf ("%s url:\"%s\"", search, line);
      free (search);
      search = s1;
    }
  prev = NULL;
  while ((line = argz_next(options->selftext, options->selftext_len, prev)))
    {
      prev = line;
      char *s1 = xasprintf ("%s selftext:\"%s\"", search, line);
      free (search);
      search = s1;
    }
  prev = NULL;
  while ((line = argz_next(options->authors, options->authors_len, prev)))
    {
      prev = line;
      char *s1 = xasprintf ("%s author:\"%s\"", search, line);
      free (search);
      search = s1;
    }
  return search;
}

int 
reddit_grep(struct reddit_state_t *state, struct reddit_grep_options_t *options)
{
  int response = 0;
  int err = 0;
  char *search = generate_search (state, options);
  err = reddit_load_and_show_stories (state, NULL, -1, NULL, search, &response);
  if (err)
    {
      reddintf (state, _("Error: Couldn't get the search results page.  "
                         "(HTTP /1.1 %d err=%d)\n"), response, err);
      err = 0;
    }
  free (options->search_terms);
  free (options->reddits);
  free (options->sites);
  free (options->urls);
  free (options->selftext);
  free (search);
  return err;
}
