/*  Copyright (C) 2011 Ben Asselstine

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU Library General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
  02110-1301, USA.
*/
#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include "xvasprintf.h"
#include "gettext-more.h"
#include "reddit_priv.h"
#include "discuss.h"
#include "view.h"
#include "stories.h"
#include "story.h"

static error_t 
parse_opt (int key, char *arg, struct argp_state *state)
{
  struct reddit_discuss_options_t *opt = NULL;
  if (state)
    opt = (struct reddit_discuss_options_t*) state->input;
  switch (key)
    {
    case ARGP_KEY_ARG:
        {
          if (opt->story_id == -1 && !opt->parsing_errors)
            opt->story_id = argp_parse_story_id (state, arg, opt->state);
          else if (!opt->parsing_errors)
            {
              argp_failure (state, 0, 0, 
                            _("You may only see the comments "
                              "for one story at a time."));
              opt->parsing_errors = 1;
            }
        }
      break;
    case ARGP_KEY_NO_ARGS:
      if (state->next <= 1)
        {
          argp_failure (state, 0, 0, 
                        _("You must specify a story number "
                          "to see the comments of."));
          argp_state_help (state, state->err_stream, ARGP_HELP_SEE);
          opt->parsing_errors = 1;
        }
      break;
    case ARGP_KEY_INIT:
      opt->story_id = -1;
      opt->parsing_errors = 0;
      break;
    default:
      return ARGP_ERR_UNKNOWN;
    }
  return 0;
}

static char *
help_filter (int key, const char *text, void *input)
{
  struct reddit_discuss_options_t *o = (struct reddit_discuss_options_t*) input;
  return argp_parse_story_help_filter (key, text, o ? o->state : NULL);
}

struct argp reddit_discuss_argp = { NULL, parse_opt, "STORY-NUMBER",N_("See the discussion for a given story.\v"), 0, help_filter};

int 
reddit_discuss_parse_argp (struct reddit_state_t *state, int argc, char **argv, struct argp *argp)
{
  int err = 0;
  struct reddit_discuss_options_t discuss_options;
  discuss_options.state = state;

  err = argp_parse (argp, argc, argv, ARGP_NO_EXIT,  0, &discuss_options);
  if (!err && discuss_options.story_id != -1 && !discuss_options.parsing_errors)
    return reddit_discuss (state, &discuss_options);
  else
    return 0;
}

int 
reddit_discuss (struct reddit_state_t *state, struct reddit_discuss_options_t *options)
{
  int err = 0;
  struct reddit_story_t *story = 
    reddit_find_story_with_id (state, options->story_id);
  if (story)
    {
      char *id = story->id;
      id+=3;
      char *link = xasprintf("http://%s/r/%s/comments/%s/.compact", 
                             state->site, story->subreddit, id);

      printf("link is `%s'\n", link);
      err = reddit_spawn_viewer_for_link (state, link);
      free (link);
    }
  return err;
}
