/*  Copyright (C) 2011 Ben Asselstine

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU Library General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
  02110-1301, USA.
*/
#include <config.h>
#include <stdlib.h>
#include <glib.h>
#include <argz.h>
#include "reddit_priv.h"
#include "view.h"
#include "story.h"
#include "stories.h"
#include "gettext-more.h"
#include "wait-process.h"

static error_t 
parse_opt (int key, char *arg, struct argp_state *state)
{
  struct reddit_view_options_t *opt = NULL;
  if (state)
    opt = (struct reddit_view_options_t*) state->input;
  switch (key)
    {
    case ARGP_KEY_ARG:
        {
          if (opt->story_id == -1 && !opt->parsing_errors)
            {
              opt->story_id = argp_parse_story_id (state, arg, opt->state);
              if (opt->story_id == -1)
                opt->parsing_errors = 1;
            }
          else if (!opt->parsing_errors)
            {
              argp_failure (state, 0, 0, 
                            _("You may only view one story at a time."));
              opt->parsing_errors = 1;
            }
        }
      break;
    case ARGP_KEY_NO_ARGS:
      if (state->next <= 1)
        {
          argp_failure (state, 0, 0, 
                        _("You must specify a story number to view."));
          argp_state_help (state, state->err_stream, ARGP_HELP_SEE);
          opt->parsing_errors = 1;
        }
      break;
    case ARGP_KEY_INIT:
      opt->story_id = -1;
      opt->parsing_errors = 0;
      break;
    default:
      return ARGP_ERR_UNKNOWN;
    }
  return 0;
}

static char *
help_filter (int key, const char *text, void *input)
{
  struct reddit_view_options_t *o = (struct reddit_view_options_t*) input;
  return argp_parse_story_help_filter (key, text, o ? o->state : NULL);
}

struct argp reddit_view_argp = { NULL, parse_opt, "STORY-NUMBER",N_("Increases the number of people who liked a given story.  Tallies your vote in favour of a story.\v"), 0, help_filter };

int 
reddit_view_parse_argp (struct reddit_state_t *state, int argc, char **argv, struct argp *argp)
{
  int err = 0;
  struct reddit_view_options_t view_options;
  view_options.state = state;
  err = argp_parse (argp, argc, argv, ARGP_NO_EXIT,  0, &view_options);
  if (!err && view_options.story_id != -1 && !view_options.parsing_errors)
    return reddit_view (state, &view_options);
  else
    return 0;
}

static int
spawn_viewer_for_story (struct reddit_state_t *state, struct reddit_story_t *story)
{
  char *prev = NULL;
  char *option;
  char *argz = NULL;
  size_t argz_len = 0;
  int err = 0;
  int flags = G_SPAWN_DO_NOT_REAP_CHILD | G_SPAWN_CHILD_INHERITS_STDIN;
  if (state->browser_options)
    argz_create_sep (strdup (state->browser_options), ' ', &argz, &argz_len);
  GError *error = NULL;
  gchar *argv[argz_count (argz, argz_len) + 3];
  int argc = 0;
  argv[argc++]= state->browser_binary;
  if (state->browser_options)
    {
      while ((option = argz_next (argz, argz_len, prev)))
        {
          prev = option;
          argv[argc++] = option;
        }
    }
  argv[argc++]= story->link;
  argv[argc] = NULL;
  GPid pid;

  g_spawn_async (NULL, argv, NULL, flags, NULL, NULL, &pid, &error);
  free (argz);
  if (error && error->code)
    err = -1;
  wait_subprocess ((pid_t) pid, state->browser_binary, true, true, false, false, NULL);
  return err;
}

int 
reddit_view (struct reddit_state_t *state, struct reddit_view_options_t *options)
{
  int err = 0;
  struct reddit_story_t *story = reddit_find_story_with_id (state, 
                                                            options->story_id);
  if (story)
    {
      err = spawn_viewer_for_story (state, story);
      if (err)
        {
          err = 0;
          reddintf (state, _("Error: Sorry, something went wrong "
                             "when viewing the story.\n"));
        }
    }
  return err;
}
