/*  Copyright (C) 2011 Ben Asselstine

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU Library General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
  02110-1301, USA.
*/
#include <config.h>
#include <stdlib.h>
#include <argz.h>
#include "gettext-more.h"
#include "xvasprintf.h"
#include "reddit_priv.h"
#include "ls.h"
#include "story.h"
#include "stories.h"

static error_t 
parse_opt (int key, char *arg, struct argp_state *state)
{
  struct reddit_ls_options_t *opt = NULL;
  if (state)
    opt = (struct reddit_ls_options_t*) state->input;
  switch (key)
    {
    case ARGP_KEY_ARG:
        {
          if (!opt->parsing_errors)
            {
              int id = argp_parse_story_id (state, arg, opt->state);
              if (id == -1)
                opt->parsing_errors = 1;
              else
                {
                  char *id_str = xasprintf ("%d", id);
                  argz_add (&opt->stories,  &opt->stories_len, id_str);
                  free (id_str);
                }
            }
        }
      break;
    case ARGP_KEY_INIT:
      opt->parsing_errors = 0;
      opt->stories = NULL;
      opt->stories_len = 0;
      break;
    default:
      return ARGP_ERR_UNKNOWN;
    }
  return 0;
}

struct argp reddit_ls_argp = { NULL, parse_opt, "[STORY-NUMBER...]", N_("See the list of stories that we already downloaded.") };

int 
reddit_ls_parse_argp (struct reddit_state_t *state, int argc, char **argv, struct argp *argp)
{
  int err = 0;
  struct reddit_ls_options_t ls_options;
  ls_options.state = state;

  err = argp_parse (argp, argc, argv, ARGP_NO_EXIT, 0, &ls_options);

  if (!err && !ls_options.parsing_errors && !argp_help_check (argc, argv))
    return reddit_ls (state, &ls_options);
  else
    return 0;
}

int 
reddit_ls (struct reddit_state_t *state, struct reddit_ls_options_t *options)
{
  int err = 0;
  if (options->stories)
    {
      char *id_str;
      char *prev = NULL;
      while ((id_str = argz_next (options->stories, options->stories_len, 
                                  prev)))
        {
          prev = id_str;
          struct reddit_story_t *story = 
            reddit_find_story_with_id (state, atoi (id_str));
          reddit_show_story (state, story);
        }
    }
  else
    reddit_reshow_stories (state);
  return err;
}
