/*  Copyright (C) 2011 Ben Asselstine

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU Library General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
  02110-1301, USA.
*/
#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include "xvasprintf.h"
#include "whats.h"
#include "gettext-more.h"
#include <argz.h>

char *const whats[] = 
{
  [REDDIT_WHATS_HOT] = "hot",
  [REDDIT_WHATS_NEW] = "new",
  [REDDIT_WHATS_CONTROVERSIAL] = "controversial",
  [REDDIT_WHATS_TOP] = "top",
  [REDDIT_WHATS_SAVED] = "saved",
  [REDDIT_WHATS_A_BATTLE] = NULL,
};

char *
reddit_whats_to_string (int w)
{
  return whats[w];
}

static struct argp_option options[] = 
{
    {0}
};

static char *generate_whats_list()
{
  char *argz = NULL;
  size_t argz_len = 0;
  if (argz_create (whats, &argz, &argz_len) == 0)
    {
      argz_stringify (argz, argz_len, ' ');
      return argz;
    }
  return NULL;
}

static void
whats_error (struct argp_state *state)
{
  struct reddit_whats_options_t *opt = NULL;
  if (state)
    opt = (struct reddit_whats_options_t*) state->input;
  char *s = generate_whats_list();
  argp_failure (state, 0, 0, _("You must specify one of: %s."), s);
  free (s);
  argp_state_help (state, state->err_stream, ARGP_HELP_SEE);
  if (opt)
    opt->parsing_errors = 1;
}

static error_t 
parse_opt (int key, char *arg, struct argp_state *state)
{
  struct reddit_whats_options_t *opt = NULL;
  if (state)
    opt = (struct reddit_whats_options_t*) state->input;
  switch (key)
    {
    case ARGP_KEY_ARG:
      if (opt->whats == -1 && !opt->parsing_errors)
        {
          char *ptr = arg;
          char *value = NULL;
          opt->whats = getsubopt (&ptr, whats, &value);
          if (opt->whats == -1 || value)
            {
              opt->whats = -1;
              whats_error (state);
            }
        }
      else if (!opt->parsing_errors)
        whats_error (state);
      break;
    case ARGP_KEY_NO_ARGS:
      if (state->next <= 1)
        whats_error (state);
      break;
    case ARGP_KEY_INIT:
      opt->whats = -1;
      opt->parsing_errors = 0;
      break;
    default:
      return ARGP_ERR_UNKNOWN;
    }
  return 0;
}

struct argp reddit_whats_argp = { options, parse_opt, "`hot' | `new' | `controversial' | `top' | `saved'",N_("Flips the tab on the list of stories.\v")};

int 
reddit_whats_parse_argp (struct reddit_state_t *state, int argc, char **argv, struct argp *argp)
{
  int err = 0;
  struct reddit_whats_options_t whats_options;
  whats_options.state = state;

  err = argp_parse (argp, argc, argv, ARGP_NO_EXIT,  0, &whats_options);
  if (!err && whats_options.whats != -1 && !whats_options.parsing_errors)
    return reddit_whats (state, &whats_options);
  else
    return 0;
}

int 
reddit_whats (struct reddit_state_t *state, struct reddit_whats_options_t *options)
{
  char *subreddit = NULL;
  int err = 0;
  if (options->whats == REDDIT_WHATS_SAVED)
    {
      if (state->logged_in == 0)
        {
          fprintf (state->out, _("You need to login to see your saved stories."
                                 "  Login using the `login' command.\n"));
          return 0;
        }
      subreddit = "";
    }
      
  free (state->page);
  state->page = NULL;
  int response = 0;
  err = reddit_load_and_show_stories (state, NULL, options->whats, subreddit, &response);
  if (err)
    {
      if (response == 200)
        fprintf (state->out, _("There doesn't seem to be anything here.\n"));
      else
        fprintf (state->out, _("Error: Couldn't load the `%s' page.  "
                               "(HTTP /1.1 %d err=%d)\n"), 
                 whats[options->whats],
                 response, err);
      err = 0;
    }
  return err;
}
