/*  Copyright (C) 2011 Ben Asselstine

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU Library General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
  02110-1301, USA.
*/
#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <glib.h>
#include "xvasprintf.h"
#ifdef HAVE_LIBREADLINE
#include <readline/readline.h>
#include <readline/history.h>
#endif
#include "submit.h"
#include "stories.h"
#include "gettext-more.h"

static struct argp_option options[] = 
{
    {"subreddit", 'r', "NAME", 0, N_("put the story in the given subreddit")},
    {"title", 't', "\"TEXT IN QUOTES\"", 0, 
      N_("specify the title of the story.")},
    {0}
};

static error_t 
parse_opt (int key, char *arg, struct argp_state *state)
{
  struct reddit_submit_options_t *opt = NULL;
  if (state)
    opt = (struct reddit_submit_options_t*) state->input;
  switch (key)
    {
    case 'r':
      opt->subreddit = arg;
      break;
    case 't':
      opt->title = arg;
      break;
    case ARGP_KEY_ARG:
        {
          if (opt->url == NULL && !opt->parsing_errors)
            {
              opt->url = argp_parse_link (state, arg, opt->state);
              if (opt->url == NULL)
                opt->parsing_errors = 1;
            }
          else if (!opt->parsing_errors)
            {
              argp_failure (state, 0, 0, 
                            _("That doesn't look like a link."));
              opt->parsing_errors = 1;
            }
        }
      break;
    case ARGP_KEY_NO_ARGS:
      if (state->next <= 1)
        {
          argp_failure (state, 0, 0, 
                        _("You must specify a link to submit."));
          argp_state_help (state, state->err_stream, ARGP_HELP_SEE);
          opt->parsing_errors = 1;
        }
      break;
    case ARGP_KEY_INIT:
      opt->url = NULL;
      opt->title = NULL;
      opt->subreddit = NULL;
      opt->parsing_errors = 0;
      break;
    default:
      return ARGP_ERR_UNKNOWN;
    }
  return 0;
}

char *
argp_parse_link (struct argp_state *state, char *link, struct reddit_state_t *r)
{
  if (g_regex_match_simple ("(http|ftp|https):\\/\\/"
                            "[\\w\\-_]+(\\.[\\w\\-_]+)+([\\w\\-\\.,@"
                            "?^=%&amp;:/~\\+#]*[\\w\\-\\@?^=%&amp;/~\\+#])?",
                            link, G_REGEX_CASELESS, 0))
    return link;
  argp_failure (state, 0, 0, _("The URL `%s' is malformed."), link);
  return NULL;
}

static char *
help_filter (int key, const char *text, void *input)
{
  struct reddit_submit_options_t *o = (struct reddit_submit_options_t*) input;
  if (key == ARGP_KEY_HELP_POST_DOC)
    {
      if (o && o->state)
        return xasprintf (text, 
                          o->state->subreddit ? o->state->subreddit : "main");
    }
  return (char *) text;
}

struct argp reddit_submit_argp = { options, parse_opt, "LINK",
  N_("Submit a link.\v"
     "The story will be submitted to the %s reddit by default."), 0, 
  help_filter };

int 
reddit_submit_parse_argp (struct reddit_state_t *state, int argc, char **argv, struct argp *argp)
{
  int err = 0;
  struct reddit_submit_options_t submit_options;
  submit_options.state = state;

  err = argp_parse (argp, argc, argv, ARGP_NO_EXIT,  0, &submit_options);
  if (!err && submit_options.url != NULL && !submit_options.parsing_errors)
    return reddit_submit (state, &submit_options);
  else
    return 0;
}

static int
reddit_submit_story (struct reddit_state_t *state, struct reddit_submit_options_t *options, int *resp, char **errmsg)
{
  int err = 0;
  char *post = 
    xasprintf ("uh=%s&kind=%s&url=%s&sr=%s&title=%s&id%%23newlink&r=%s"
               "&renderstyle=%s",
               state->modhash,
               "link",
               options->url,
               options->subreddit,
               options->title,
               options->subreddit,
               "html");
  char *msg = NULL;
  err = reddit_post (state, "submit", post, resp, &msg);
  free (post);
  if (strstr (msg, "http://" REDDIT_SITE "/") == NULL)
    {
      if (!err)
        err = -1;
      *errmsg = msg;
    }
  else
    free (msg);
  return err;
}

int 
reddit_submit (struct reddit_state_t *state, struct reddit_submit_options_t *options)
{
  int err = 0;
  if (state->logged_in == 0)
    {
      fprintf (state->out, _("You need to login to submit a story."
                             "  Login using the `login' command.\n"));
      return 0;
    }
  int malloced_title = 0;
  if (options->title == NULL)
    {
      //go get a title.
      fprintf (state->out, _("Type in your title and then press enter.\n"));
#ifdef HAVE_LIBREADLINE
      stifle_history (0);
      options->title = readline (_("Title: "));
      unstifle_history();
#else
      options->title = myreadline (_("Title: "));
#endif
      malloced_title = 1;
    }

  int response = 0;
  char *errmsg = NULL;
  err = reddit_submit_story (state, options, &response, &errmsg);

  if (!err)
    reddit_update_modhash (state);
  if (!err)
    fprintf (state->out, _("The story has been submitted.\n"));
  else
    {
      fprintf (state->out, _("Error: Couldn't submit the story.  "
                             "(HTTP /1.1 %d err=%d)\n"), response, err);
      if (errmsg)
        {
          fprintf (state->out, _("Maybe this gobbledegook will be useful "
                                 "in diagnosing what went wrong:\n"
                                 "%s\n"), errmsg);
          free (errmsg);
        }
    }
  if (malloced_title)
    free (options->title);

  return err;
}
