/*  Copyright (C) 2011 Ben Asselstine

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU Library General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
  02110-1301, USA.
*/
#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <curl/curl.h>
#include <unistd.h>
#include "xvasprintf.h"
#include "login.h"
#include "story.h"
#include "stories.h"
#include "gettext-more.h"
#include "getpass.h"
#include "xvasprintf.h"

static struct argp_option options[] = 
{
    {"password", 'p', "PASSWORD", OPTION_HIDDEN, N_("specify a password.")},
    {0}
};

static error_t 
parse_opt (int key, char *arg, struct argp_state *state)
{
  struct reddit_login_options_t *opt = NULL;
  if (state)
    opt = (struct reddit_login_options_t*) state->input;
  switch (key)
    {
    case 'p':
      opt->password = arg;
      break;
    case ARGP_KEY_ARG:
        {
          if (opt->username == NULL && !opt->parsing_errors)
            opt->username = arg;
          else if (!opt->parsing_errors)
            {
              argp_failure (state, 0, 0, 
                            _("That looks like more than one username.  "
                              "Try putting it all in quotes."));
              opt->parsing_errors = 1;
            }
        }
      break;
    case ARGP_KEY_NO_ARGS:
      if (state->next <= 1)
        {
          argp_failure (state, 0, 0, 
                        _("You must specify a username to login as."));
          argp_state_help (state, state->err_stream, ARGP_HELP_SEE);
          opt->parsing_errors = 1;
        }
      break;
    case ARGP_KEY_INIT:
      opt->username = NULL;
      opt->password = NULL;
      opt->parsing_errors = 0;
      break;
    default:
      return ARGP_ERR_UNKNOWN;
    }
  return 0;
}

struct argp reddit_login_argp = { options, parse_opt, "USERNAME",N_("Login to reddit.com.") };

int 
reddit_login_parse_argp (struct reddit_state_t *state, int argc, char **argv, struct argp *argp)
{
  int err = 0;
  struct reddit_login_options_t login_options;
  login_options.state = state;

  err = argp_parse (argp, argc, argv, ARGP_NO_EXIT,  0, &login_options);
  if (!err && login_options.username != NULL && !login_options.parsing_errors)
    return reddit_login (state, &login_options);
  else
    return 0;
}

static int
reddit_login_with_credentials (struct reddit_state_t *state, struct reddit_login_options_t *options, int *resp, char **errmsg)
{
  int err = 0;
  char *msg = NULL;
  char *post = xasprintf ("user=%s&passwd=%s&op=login&dest=%%3f",
                          options->username, options->password);
  err = reddit_post (state, "login", post, resp, &msg);
  free (post);
  CURLcode res;
  struct curl_slist *cookies;
  struct curl_slist *nc;
  int found = 0;
  int i;

  res = curl_easy_getinfo(state->curl, CURLINFO_COOKIELIST, &cookies);
  nc = cookies, i = 1;
  while (nc) 
    {
      if (strstr (nc->data, "reddit_session"))
        {
          found = 1;
          break;
        }
      nc = nc->next;
      i++;
    }
  curl_slist_free_all(cookies);
  if (!found)
    err = -2;
  free (msg);
  return err;
}


int 
reddit_login (struct reddit_state_t *state, struct reddit_login_options_t *options)
{
  int err = 0;
  char *p = NULL;
  if (state->logged_in && strcmp (state->username, options->username) == 0)
    {
      fprintf (state->out, _("You are already logged in.\n"));
      return 0;
    }

  fflush (stdout);
  if (options->password == NULL)
    p = getpass (_("Password: "));

  options->password = p;

  int response = 0;
  char *errmsg = NULL;
  err = reddit_login_with_credentials (state, options, &response, &errmsg);

  memset (options->password, 0, strlen (options->password));
  if (!err)
    {
      free (state->username);
      state->username = strdup (options->username);
      state->logged_in = 1;
      reddit_update_modhash (state);
      fprintf (state->out, _("Hello redditor.\n"));
    }
  else
    {
      fprintf (state->out, _("Error: Login unsuccessful.  " 
                             "(HTTP /1.1 %d err=%d\n"), response, err);
      if (errmsg)
        {
          fprintf (state->out, _("Maybe this gobbledegook will be useful "
                                 "in diagnosing what went wrong:\n"
                                 "%s\n"), errmsg);
          free (errmsg);
        }
    }
  return err;
}
