/*  Copyright (C) 2011 Ben Asselstine

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU Library General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
  02110-1301, USA.
*/
#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include "xvasprintf.h"
#include "upvote.h"
#include "story.h"
#include "stories.h"
#include "gettext-more.h"
#include "read-file.h"

static struct argp_option options[] = 
{
    {0}
};

static error_t 
parse_opt (int key, char *arg, struct argp_state *state)
{
  struct reddit_upvote_options_t *opt = NULL;
  if (state)
    opt = (struct reddit_upvote_options_t*) state->input;
  switch (key)
    {
    case ARGP_KEY_ARG:
        {
          if (opt->story_id == -1 && !opt->parsing_errors)
            {
              opt->story_id = argp_parse_story_id (state, arg, opt->state);
              if (opt->story_id == -1)
                opt->parsing_errors = 1;
            }
          else if (!opt->parsing_errors)
            {
              argp_failure (state, 0, 0, 
                            _("You may only upvote one story at a time."));
              opt->parsing_errors = 1;
            }
        }
      break;
    case ARGP_KEY_NO_ARGS:
      if (state->next <= 1)
        {
          argp_failure (state, 0, 0, 
                        _("You must specify a story number to upvote."));
          argp_state_help (state, state->err_stream, ARGP_HELP_SEE);
          opt->parsing_errors = 1;
        }
      break;
    case ARGP_KEY_INIT:
      opt->story_id = -1;
      opt->parsing_errors = 0;
      break;
    default:
      return ARGP_ERR_UNKNOWN;
    }
  return 0;
}

static char *
help_filter (int key, const char *text, void *input)
{
  struct reddit_upvote_options_t *o = (struct reddit_upvote_options_t*) input;
  return argp_parse_story_help_filter (key, text, o ? o->state : NULL);
}

struct argp reddit_upvote_argp = { options, parse_opt, "STORY-NUMBER",N_("Increases the number of people who liked a given story.  Tallies your vote in favour of a story.\v"), 0, help_filter};

int 
reddit_upvote_parse_argp (struct reddit_state_t *state, int argc, char **argv, struct argp *argp)
{
  int err = 0;
  struct reddit_upvote_options_t upvote_options;
  upvote_options.state = state;

  err = argp_parse (argp, argc, argv, ARGP_NO_EXIT,  0, &upvote_options);
  if (!err && upvote_options.story_id != -1 && !upvote_options.parsing_errors)
    return reddit_upvote (state, &upvote_options);
  else
    return 0;
}

static int 
reddit_upvote_story (struct reddit_state_t *state, struct reddit_upvote_options_t *options, struct reddit_story_t *story, int *resp)
{
  int err = 0;
  char tmp[sizeof(PACKAGE) + 13];
  snprintf (tmp, sizeof tmp, "/tmp/%s.XXXXXX", PACKAGE);

  int fd = mkstemp(tmp);
  close (fd);
  FILE *fileptr = fopen (tmp, "wb");
  char *url = xasprintf ("http://%s/api/vote", REDDIT_SITE);
  char *post = 
    xasprintf ("id=%s&dir=%d&vh=%s&uh=%s&renderstyle=%s",
               story->id,
               1,
               story->votehash,
               state->modhash,
               "html");
  curl_easy_reset (state->curl);
  curl_easy_setopt (state->curl, CURLOPT_URL, url);
  curl_easy_setopt (state->curl, CURLOPT_POSTFIELDS, post);
  curl_easy_setopt (state->curl, CURLOPT_POSTFIELDSIZE, (long)strlen (post));
  curl_easy_setopt (state->curl, CURLOPT_WRITEDATA, fileptr);
  curl_easy_perform(state->curl);
  free (url);
  free (post);
  fflush (fileptr);
  fclose (fileptr);
  long response = 0;
  curl_easy_getinfo (state->curl, CURLINFO_RESPONSE_CODE, &response);
  size_t data_len = 0;
  char *text = read_file (tmp, &data_len);
  if (text)
    {
      if (strcmp (text, "{}") != 0)
        err=-1;
      free (text);
    }
  remove (tmp);
  if (resp)
    *resp = response;
  return err;
}

int 
reddit_upvote (struct reddit_state_t *state, struct reddit_upvote_options_t *options)
{
  int err = 0;
  if (state->logged_in == 0)
    {
      fprintf (state->out, _("You need to login to vote on a story."
                             "  Login using the `login' command.\n"));
      return 0;
    }
  struct reddit_story_t *story = 
    reddit_find_story_with_id (state, options->story_id);
  int response = 0;
  if (story)
    err = reddit_upvote_story (state, options, story, &response);

  if (!err)
    reddit_update_modhash (state);
  if (!err)
    fprintf (state->out, _("The story has been upvoted.\n"));
  else
    fprintf (state->out, _("Error: Couldn't upvote the story "
                           "HTTP /1.1 %d err=%d)\n"), response, err);
  return err;
}
