/*  Copyright (C) 2011 Ben Asselstine

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU Library General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
  02110-1301, USA.
*/
#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <argp.h>
#include <glib.h>
#include <unistd.h>
#include "stories.h"
#include "reddit_priv.h"
#include "gettext-more.h"
#include "read-file.h"
#include "xvasprintf.h"

static void
reddit_get_modhash_from_chunk (struct reddit_state_t *state, char *data)
{
  gchar **fields;
  fields = g_regex_split_simple ("<input type=\"hidden\" name=\"uh\" value=\"",
                                 data, 0, 0);
  int num_fields = g_strv_length (fields);
  if (num_fields <= 1)
    {
      g_strfreev (fields);
      printf("crap1\n");
      return;
    }
  else
    {
      char *quote = strchr (fields[1], '\"');
      if (!quote)
        {
          g_strfreev (fields);
          return;
        }
      *quote = 0;
      free (state->modhash);
      state->modhash = strdup (fields[1]);
    }
  g_strfreev (fields);
  return;
}

static int
reddit_load_stories_from_data (struct reddit_state_t *state, char *data, size_t data_len, struct reddit_story_t **stories)
{
  gchar **fields = g_regex_split_simple ("<div class=\" thing id-", data, 0, 0);
  int num_stories =  g_strv_length (fields);
  if (num_stories > 1)
    {
      *stories = malloc (sizeof (struct reddit_story_t ) * (num_stories - 1));
      if (state->logged_in)
        reddit_get_modhash_from_chunk (state, fields[0]);
      if (*stories)
        {
          gchar **ptr = &fields[1];
          int i = 0;
          while (*ptr)
            {
              int err = 0, skip = 0;
              (*stories)[i] = reddit_load_story_from_chunk (state, *ptr, 
                                                            &err, &skip);
              if (err)
                return err;
              ptr++;
              if (!skip)
                i++;
            }
          num_stories = i;
        }
      else
        num_stories = 0;
    }
  else
    num_stories = 0;
  g_strfreev (fields);
  free (data);
  return num_stories;
}

static int 
reddit_load_stories_from_file (struct reddit_state_t *state, FILE *fileptr, struct reddit_story_t **stories)
{
  if (!fileptr)
    return 0;
  size_t data_len = 0;
  char *data = fread_file (fileptr, &data_len);
  fclose (fileptr);
  return reddit_load_stories_from_data (state, data, data_len, stories);
}

int 
reddit_load_stories (struct reddit_state_t *state, char *url, struct reddit_story_t **stories, int *resp)
{
  int err = 0;
  *stories = NULL;
  char tmp[sizeof(PACKAGE) + 13];
  snprintf (tmp, sizeof tmp, "/tmp/%s.XXXXXX", PACKAGE);
  int fd = mkstemp(tmp);
  close (fd);
  FILE *fileptr = fopen (tmp, "wb");
  curl_easy_reset (state->curl);
  curl_easy_setopt (state->curl, CURLOPT_URL, url);
  curl_easy_setopt (state->curl, CURLOPT_WRITEDATA, fileptr);
  curl_easy_perform(state->curl);
  fflush (fileptr);
  fsync (fileno (fileptr));
  fclose (fileptr);
  long response = 0;
  curl_easy_getinfo (state->curl, CURLINFO_RESPONSE_CODE, &response);

  if (response == 200)
    err = reddit_load_stories_from_file (state, fopen (tmp, "r"), stories);
  else
    err = -1;
  if (resp)
    *resp = response;
  remove (tmp);
  return err;
}

void
reddit_show_stories (struct reddit_state_t *state)
{
  reddit_line (state);
  int i = state->num_stories - 1;
  for (; i >= 0; i--)
    reddit_show_story (state, &state->stories[i]);
  reddit_line (state);
  return;
}

char *
htmltotext (struct reddit_state_t *state, char *html)
{
  char tmp[sizeof(PACKAGE) + 13];
  snprintf (tmp, sizeof tmp, "/tmp/%s.XXXXXX", PACKAGE);
  int fd = mkostemp(tmp, 0);
  close (fd);
  FILE *fileptr = fopen (tmp, "w");
  fprintf (fileptr, "%s", html);
  fclose (fileptr);
  char *cmd = xasprintf ("%s -dump %s 2>/dev/null", state->links_binary, tmp);
  fileptr = popen (cmd, "r");
  size_t data_len = 0;
  char *text = fread_file (fileptr, &data_len);
  remove (tmp);
  pclose (fileptr);
  return text;
}

void 
reddit_free_stories (struct reddit_story_t *stories, int num_stories)
{
  int i = 0;
  for (; i < num_stories; i++)
    reddit_free_story (&stories[i], 0);
  free (stories);
}
  
struct reddit_story_t *
reddit_find_story_with_id (struct reddit_state_t *state, int story_number)
{
  int i = 0;
  for (; i < state->num_stories; i++)
    {
      if (state->stories[i].story_number == story_number)
        return &state->stories[i];
    }
  return NULL;
}
