/*  Copyright (C) 2011 Ben Asselstine

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU Library General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
  02110-1301, USA.
*/
#include <config.h>
#include <stdlib.h>
#include <error.h>
#include <string.h>
#include <argp.h>
#include <argz.h>
#include <unistd.h>
#include "opts.h"
#include "gettext-more.h"
#include "reddit_priv.h"

#define FULL_VERSION PACKAGE_NAME " " PACKAGE_VERSION

const char *argp_program_version = FULL_VERSION;
const char *argp_program_bug_address = "<" PACKAGE_BUGREPORT ">";

#define OAO OPTION_ARG_OPTIONAL
#define OH OPTION_HIDDEN

struct arguments_t arguments;
static struct argp_option options[] = 
{
    { "quiet", OPT_QUIET, NULL, 0, N_("don't show the initial banner"), 0 },
    { "login", OPT_LOGIN, "USER", 0, N_("login as USER to " REDDIT_SITE ), 0 },
    { 0 }
};

static void
init_options (struct reddit_options_t *app)
{
  app->quiet = -1;
  return;
}

static error_t 
parse_opt (int key, char *arg, struct argp_state *state) 
{
  struct arguments_t *arguments = (struct arguments_t *) state->input;

  switch (key) 
    {
    case OPT_QUIET:
	arguments->reddit.quiet = 1;
      break;
    case OPT_LOGIN:
	arguments->reddit.username = arg;
      break;
    case ARGP_KEY_INIT:
      init_options (&arguments->reddit);
      break;
    case ARGP_KEY_ARG:
      break;
    case ARGP_KEY_END:
      break;
    default:
      return ARGP_ERR_UNKNOWN;
    }
  return 0;
}


static const struct argp_child child_parsers[] = 
{
    {0}
};

struct argp argp = { options, parse_opt, "", 
  N_("A bandwidth-saving client for reddit.com."), child_parsers, 0, PACKAGE }; 

static void 
set_default_options (struct reddit_options_t *app)
{
  if (app->quiet == -1)
    app->quiet = atoi (DEFAULT_QUIET_VALUE);
  return;
}

void 
parse_opts (int argc, char **argv, struct arguments_t *arguments)
{
  int retval;
  setenv ("ARGP_HELP_FMT", "no-dup-args-note", 1);
  retval = argp_parse (&argp, argc, argv, 0, 0, arguments); 
  if (retval < 0)
    argp_help (&argp, stdout, ARGP_HELP_EXIT_ERR|ARGP_HELP_SEE,PACKAGE_NAME);
  set_default_options (&arguments->reddit);

  argp_program_version = 0;
  argp_program_bug_address = 0;
  return;
}
