### Copyright (C) 2009 Manuel Ospina <ospina.manuel@gmail.com>

# This file is part of rayuela.
#
# rayuela is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# rayuela is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with rayuela; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

try:
    import pygtk
    pygtk.require('2.0')
except:
    pass
try:
    import gtk
    import gtk.glade
    import pango
except:
    print "GTK is not installed"
    sys.exit(1)

import model
import view

class Controller:
    
    def __init__(self, glade_file):

        self.model = model.Model()
        self.view = view.View(self, self.model, glade_file)
        self.new(None)

    def quit(self, obj):
        window = obj.get_toplevel()
        window.destroy()
        gtk.main_quit()

    def delete_event(self, obj, event): 
        return False

    def _get_page_(self): 
        notebook = self.view.widget_tree.get_widget("main_notebook")
        if not self.model.documents:
            textview = self.view.widget_tree.get_widget("textview")
        else:
            textview = self.view.add_textview()
            if not textview:
                view.error_dialog("Impossible to create a new page")
        current_page = notebook.get_current_page()
        #textbuffer = textview.get_buffer()
        textbuffer = gtk.TextBuffer(self.view.tag_table)
        textview.set_buffer(textbuffer)
        return (current_page, textbuffer)
    
    def _get_document_(self):
        notebook = self.view.widget_tree.get_widget("main_notebook")
        current_page = notebook.get_current_page()
        document = self.model.get_document_by_page(current_page)
        return document

    def new(self, obj):
        current_page, textbuffer = self._get_page_()
        self.model.open(current_page, textbuffer)
        self._add_synopsis_()

    def open(self, obj):
        filename = view.file_chooser('Open...', gtk.FILE_CHOOSER_ACTION_OPEN)
        current_page, textbuffer = self._get_page_()
        self.model.open(current_page, textbuffer, filename)

    def save(self, obj):
        document = self._get_document_()
        if document.filepath:
            self.model.save(document)
        else:
            self.save_as(obj)

    def save_as(self, obj):
        document = self._get_document_()
        filename = view.file_chooser('Save As...', gtk.FILE_CHOOSER_ACTION_SAVE)
        self.model.save(document, filename)

    def cut(self, obj): 
        document = self._get_document_()
        document.buffer.cut_clipboard(self.view.clipboard, True)

    def copy(self, obj):
        document = self._get_document_()
        document.buffer.copy_clipboard(self.view.clipboard)

    def paste(self, obj):
        document = self._get_document_()
        document.buffer.paste_clipboard(self.view.clipboard, None, True)

    def delete(self, obj): 
        self.cut(obj)
    
    def _add_synopsis_(self):
        document = self._get_document_()
        document.synopsis.id = 'synopsis'
        result = self.view.section_dialog(document.synopsis)
        if result == gtk.RESPONSE_OK:
            # tags:
            xml_start = "<title>"
            xml_end = "</title>"

            # Start tag
            cursor_mark = document.buffer.get_insert()
            cursor = document.buffer.get_iter_at_mark(cursor_mark)
            document.buffer.insert_with_tags_by_name(cursor, 
                                                     xml_start, 
                                                     "invisible", "center")
            # Add title:
            document.buffer.insert_with_tags_by_name(cursor, 
                                                     document.synopsis.title,
                                                     "bold", "uneditable")
            # End tag    
            document.buffer.insert_with_tags_by_name(cursor, 
                                                     xml_end, 
                                                     "invisible")
            # Break Line
            document.buffer.insert(cursor, "\n")

    def _add_section_(self):
        section = model.Section()
        result = self.view.section_dialog(section)
        if result == gtk.RESPONSE_OK:
            document = self._get_document_()
            document.append(section)
            # tags:
            xml_start = "<title>"
            xml_end = "</title>"

            # Add tag:
            cursor_mark = document.buffer.get_insert()
            cursor = document.buffer.get_iter_at_mark(cursor_mark)
            mark = document.buffer.create_mark(section.id, cursor, True)

            xml_tag = '<section id="%s">' % section.id
            document.buffer.insert_with_tags_by_name(cursor, 
                                                     xml_tag, 
                                                     "invisible")
            # Title start tag
            document.buffer.insert_with_tags_by_name(cursor, 
                                                     xml_start, 
                                                     "invisible")

            # Add title:
            document.buffer.insert_with_tags_by_name(cursor, 
                                                     section.title,
                                                     "bold", "uneditable")
            # Title end tag
            document.buffer.insert_with_tags_by_name(cursor, 
                                                     xml_end, 
                                                     "invisible")
                
            # line break
            document.buffer.insert(cursor, "\n")

            # Ask the model to notify the observers:
            self.model.notify()

    def add_section(self, obj): 
        self._add_section_()
    
    def new_character(self, obj): 
        profile = model.Character()
        result = self.view.character_dialog(profile)
        if result == gtk.RESPONSE_OK:
            document = self._get_document_()
            document.character.append(profile)
            # Ask the model to notify the observers:
            self.model.notify()
    
    def edit_character(self, obj):
        tree = self.view.widget_tree.get_widget('character_treeview')
        model, iter = tree.get_selection().get_selected()
        if iter == None: return
        page = model.get_value(iter, 1)
        id = model.get_value(iter, 2)
        document = self.model.get_document_by_page(page)
        profile = document.get_character_by_id(id)
        result = self.view.character_dialog(profile)
        if result == gtk.RESPONSE_OK:
            self.model.notify()

    def new_location(self, obj): 
        profile = model.Location()
        result = self.view.location_dialog(profile)
        if result == gtk.RESPONSE_OK:
            document = self._get_document_()
            document.location.append(profile)
            # Ask the model to notify the observers:
            self.model.notify()

    def edit_location(self, obj): 
        tree = self.view.widget_tree.get_widget('location_treeview')
        model, iter = tree.get_selection().get_selected()
        if iter == None: return
        page = model.get_value(iter, 1)
        id = model.get_value(iter, 2)
        document = self.model.get_document_by_page(page)
        profile = document.get_location_by_id(id)
        result = self.view.character_dialog(profile)
        if result == gtk.RESPONSE_OK:
            self.model.notify()
               
    def about(self, obj):
        txt = "Rayuela is a creative writing framework."
        view.info_dialog(txt) 

if __name__ == "__main__":
    from rayuela.glade import RAYUELA_GLADE_FILE 
    Controller(RAYUELA_GLADE_FILE)
    gtk.main()
