#!/usr/bin/python

#  Copyright (C) 2005-2010, Axis Communications AB, LUND, SWEDEN
#
#  This file is part of RAPP.
#
#  RAPP is free software: you can redistribute it and/or modify
#  it under the terms of the GNU Lesser General Public License as published
#  by the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#  You can use the comments under either the terms of the GNU Lesser General
#  Public License version 3 as published by the Free Software Foundation,
#  either version 3 of the License or (at your option) any later version, or
#  the GNU Free Documentation License version 1.3 or any later version
#  published by the Free Software Foundation; with no Invariant Sections, no
#  Front-Cover Texts, and no Back-Cover Texts.
#  A copy of the license is included in the documentation section entitled
#  "GNU Free Documentation License".
#
#  RAPP is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
#  GNU Lesser General Public License for more details.
#
#  You should have received a copy of the GNU Lesser General Public
#  License and a copy of the GNU Free Documentation License along
#  with RAPP. If not, see <http://www.gnu.org/licenses/>.


######################################################
#  RAPP Tuner performance analyzer.
######################################################

import sys, getopt, datetime, math

# The auto-generated tag string.
TAGSTRING = 'Auto-generated by RAPP Compute performance tuner on'

# The default data file name.
DATAFILE = 'tunedata.py'

# The output config file name.
CFGFILE = 'rapptune.h'

# The implementation name symbol table.
IMPLSYM = {'generic': 'GEN', 'swar': 'SWAR', 'simd': 'SIMD'}

# The output HTML plot report name.
PLOTFILE = 'tunereport.html'

# The bar plot color table.
PLOTCOLORS = {'generic': 'silver',
              'swar':    '#a8d6e5',
              'simd':    'steelblue'}

# The bar plot legend strings.
PLOTLEGEND = {'generic': 'Best generic',
              'swar'   : 'Best SWAR',
              'simd'   : 'Best SIMD'}


# Compose a key value.
def get_keyval(name, parm1, parm2):
    return str(name) + (name != '')*'-' + str(parm1) + '-' + str(parm2)


# Compute the average performance across all implementations,
# with constant function, width and height.
def average(data):

    # Initialize the average table
    avg = {};
    for suite in data:
        for case in suite['data']:
            key = get_keyval(case[0], suite['width'], suite['height'])
            avg[key] = [0.0, 0]

    # Compute the performance sum
    for suite in data:
        for case in suite['data']:
            key = get_keyval(case[0], suite['width'], suite['height'])
            val = case[1]
            if (val) > 0:
                avg[key][0] += val
                avg[key][1] += 1

    # Compute the average performance
    for k, v in avg.iteritems():
        v[0] /= float(max(v[1], 1))
    return avg


# Normalize the data with respect to the average performance
# for the corresponding function, width and height.
def normalize(data):
    avg  = average(data)

    for suite in data:
        for case in suite['data']:
            name = case[0]
            key  = get_keyval(name, suite['width'], suite['height'])

            if not avg[key][0]:
                sys.stderr.write('ERROR: ' + name + ' not present.\n')
                sys.exit(1)

            case[1] /= max(avg[key][0], 1)


# Combine the performance results of different test cases for
# an implementation into a single performance number.
def combine(data):

    # Initialize the combination table
    table = {}
    for case in data[0]['data']:
         table[case[0]] = {}

    # Initialize the combination table entries
    for suite in data:
        impl   = suite['impl']
        unroll = suite['unroll']
        for case in suite['data']:
            key = get_keyval('', impl, unroll)
            table[case[0]][key] = [impl, unroll, 1.0, 0]

    # Combine the scores
    for suite in data:
        impl   = suite['impl']
        unroll = suite['unroll']
        for case in suite['data']:
            if case[1] > 0:
                key  = get_keyval('', impl, unroll)
                table[case[0]][key][2] *= case[1]
                table[case[0]][key][3] += 1

    # Compute the composite scores as arithmetic mean value
    comb = {}
    for key, entry in table.iteritems():

        # Compute the count-root of the combined scores
        vec = []
        for k, variant in entry.iteritems():
            if variant[3] > 0:
                variant[2] = math.pow(variant[2], 1.0 / variant[3])
                vec += [variant]

        # Normalize w.r.t. the minimum score
        if vec:
            norm = min([v[2] for v in vec])
            for elt in vec:
                elt[2] /= norm

        # Store the result
        comb[key] = vec

    return comb


#  Order the different implementations by the performance number.
def order(comb):
    for key, val in comb.iteritems():
        val.sort(lambda x, y: cmp(y[2], x[2]))
    return comb


# Output the tuning result from the ordered result.
def output(result, data):
    print 'function' + 35*' ' + 'impl    unroll   gain'
    print 64*'='

    # Output the functions as ordered in the data set
    for test in data[0]['data']:
        name = test[0]
        val  = result[name]
        for case in val:
            if case[3] > 0:
                print '%-40s  ' % name,
                print '%-7s    %s    %5.2f' % (case[0], case[1], case[2])
                name = len(name)*' '
        print 64*'-'


# Compute the per-function throughput (pixels/second)
# for the best implementation, averaged over all test cases.
def throughput(data):
    # Initialize the score table
    score = {};
    for suite in data:
        for case in suite['data']:
            key = get_keyval(case[0], suite['impl'], suite['unroll'])
            score[key] = [0.0, 0]

    # Compute the performance sum
    for suite in data:
        for case in suite['data']:
            key = get_keyval(case[0], suite['impl'], suite['unroll'])
            val = case[1]
            if val > 0:
                score[key][0] += val
                score[key][1] += 1

    # Compute the average score
    for k, v in score.iteritems():
        v[0] /= float(max(v[1], 1))
    return score


# Generate the config file from the ordered result and the throughput score.
def config(fname, result, data, score, compilerversion):
    tag  = fname.replace('.', '_').upper()

    head = '/**\n' + \
           ' *  @file   ' + fname + '\n' + \
           ' *  @brief  RAPP Compute implementation tuning config.\n' + \
           ' *          ' + TAGSTRING + '\n' + \
           ' *          ' + datetime.datetime.now().ctime() + '\n' + \
           ' */\n\n' + \
           '#ifndef ' + tag + '\n' + \
           '#define ' + tag + '\n\n' + \
           '#include "rc_impl.h" /* Implementation names */\n\n' + \
           '#define rc_compiler_version ' + compilerversion + '\n\n'

    tail = '#endif /* ' + tag + ' */\n'

    defs = ''

    # List the functions as ordered in the data set
    for test in data[0]['data']:
        name = test[0]
        val  = result[name]
        if val:
            key    = get_keyval(name, val[0][0], val[0][1])
            impl   = '#define ' + name + '_IMPL'
            unroll = '#define ' + name + '_UNROLL'
            scr    = '#define ' + name + '_SCORE'
            defs += '%-60s RC_IMPL_%s\n' % (impl,   IMPLSYM[val[0][0]])
            defs += '%-60s %d\n'         % (unroll, val[0][1])
            defs += '%-60s %.2e\n\n'     % (scr, score[key][0])

    # Write config file
    out = open(fname, 'wb')
    out.write(head)
    out.write(defs)
    out.write(tail)
    out.close()


# Get the best score for a given implementation.
def get_best(scores, impl):
    buf = [0]
    for val in scores:
        if val[0] == impl:
            buf += [val[2]]
    buf.sort(reverse=True)
    return buf[0]


# Generate an HTML plot report.
def plot(fname, result, data, arch, zindex):
    html = """
<!DOCTYPE html PUBLIC "-//W3C//DTD HTML 4.01//EN"
                      "http://w3.org/TR/html4/strict.dtd">
<html>
  <head>
    <style>
      body, h1, td, th {
        font-family: georgia, verdana, geneva, arial, helvetica, sans-serif;
      }
      body {
        margin: 2em;
      }
      h1 {
        font-size: 18pt;
        text-align: center;
      }
      div.footer {
        font-style: italic;
        font-size: 8pt;
      }
      table.legend {
        font-size: 10pt;
        float: right;
        border: 1px solid black;
        margin: 0 0 1em 1em;
      }
      table.legend tr td div {
        width: 1em;
        border: 1px solid black;
      }
      table.legend tr td + td {
        width: auto;
      }
      table.plot {
        clear: both;
        margin: 2em 0 2em 0;
      }
      table.plot > tbody > tr > td {
        font: 10pt monospace;
      }
      table.plot td + td + td {
        text-align: right;
      }
      table.plot thead th {
        font-size: 12pt;
        font-weight: bolder;
        text-align: left;
      }
      table.plot * table tr td {
        font-size: 12pt;
        font-weight: bolder;
        text-align: center;
      }
      div.bg {
        border: 1px solid black;
      }
      div.bar {
        border-right: 1px solid black;
      }
    </style>
    <title>RAPP Compute tuning report</title>
  </head>
  <body>
  <h1>RAPP Compute Tuning Report</h1>
"""
    body  = '<table class="plot">\n' + \
            '<thead><th>Function</th>' + \
            '<th><table><tr><td>0%</td>' + \
            '<td style="width: 100%">Relative Execution Speed</td>' + \
            '<td>100%</td></tr></table></th>' + \
            '<th>Gain</th></thead><tbody>'

    exist = {'generic': False, 'swar': False, 'simd': False}

    # Plot the functions as ordered in the data set
    for test in data[0]['data']:
        name  = test[0]
        entry = result[name]
        if not entry:
            continue

        # Generate a list of the best implementations for this function
        buf = []
        for impl in ['generic', 'swar', 'simd']:
            val = get_best(entry, impl)
            if val > 0:
                buf += [[val, impl]]
                exist[impl] = True

        # Only plot result if there are at least two different implementations
        if len(buf) < 2:
            continue

        # Sort and normalize
        buf.sort()
        for k in range(len(buf) - 1):
            buf[k] += [buf[k][0] / buf[k + 1][0]]

        # Plot the function name
        body += '<tr><td>%s</td><td style="width: 100%%">\n' % name

        # Plot the best bar
        best = buf.pop()
        body += '<div class="bg"' + \
                'style="background-color: %s; width: 100%%">\n' % \
                PLOTCOLORS[best[1]]

        # Plot remaining bars
        buf.reverse()
        prev = best
        nbar = 1
        for case in buf:
            if zindex[case[1]] <= zindex[prev[1]]:
                body += '<div class="bar"' + \
                        'style="background-color: %s; width: %.2f%%">\n' % \
                        (PLOTCOLORS[case[1]], 100.0*case[2])
                nbar += 1
            prev = case

        # Finalize this function
        gain = '%.2f' % (best[0]/buf[len(buf) - 1][0])
        body += '<br>' + nbar*'</div>' + \
                '</td><td>%s</td></tr>\n' % gain

    # Close the bar plot table
    body += '</tbody></table><div class="footer">%s %s</div></body></html>' % \
            (TAGSTRING, datetime.datetime.now().ctime())

    # Add the legend table
    html += '<table class="legend">'
    for impl in ['generic', 'swar', 'simd']:
        if exist[impl]:
            html += '<tr><td><div style="background-color: %s">' % \
                    PLOTCOLORS[impl]
            html += '<br></div></td><td>%s</td></tr>\n' % \
                    PLOTLEGEND[impl]
    html += '</table>\n'

    # Add the architecture string
    html += '<em>' + arch + '</em>\n'

    # Add the plot body
    html += body

    # Write the html plot to file
    out = open(fname, 'wb')
    out.write(html)
    out.close()


# Print the usage string
def usage():
    print 'Usage: ' + sys.argv[0] + ' [OPTIONS]'
    print 'Options are: \n' + \
          '  -f, --file=DATAFILE    Use DATAFILE instead of default ' + \
          DATAFILE + '\n' + \
          '  -u, --unordered        Plot performance bars in ranking order\n'+\
          '                         rather than in the default fixed order\n'+\
          '                         generic, SWAR, SIMD.\n' + \
          '  -v, --verbose          Print the full ranking to stdout\n' + \
          '  -h, --help             Display this help\n'


# Main function
def main():
    fname   = DATAFILE
    zindex  = {'generic': 0, 'swar': 1, 'simd': 2}
    verbose = False

    # Get command line options
    try:
        opts, args = getopt.getopt(sys.argv[1:], 'f::uvh',
                                   ['file=', 'unordered', 'verbose', 'help'])
    except getopt.GetoptError:
        usage()
        sys.exit(1)

    # Handle options
    for opt, val in opts:
        if opt in ('-f', '--file'):
            fname = val
        if opt in ('-u', '--unordered'):
            zindex = {'generic': 0, 'swar': 0, 'simd': 0}
        if opt in ('-v', '--verbose'):
            verbose = True
        if opt in ('-h', '--help'):
            usage()
            sys.exit()

    # Open the data file
    try:
        infile = open(fname, 'rb')
    except:
        usage()
        sys.exit(1)

    try:
        # Parse the data file
        exec infile

        # Close the data file
        infile.close()

        # Compute the per-function average throughput (pix/sec)
        score = throughput(data)

        # Normalize the data
        normalize(data)

        # Combine measurements
        comb = combine(data)

        # Order by the performance number
        result = order(comb)

        # Generate the HTML data plot report
        plot(PLOTFILE, result, data, arch, zindex)

        # Generate the config file
        config(CFGFILE, result, data, score, compilerversion)

        # Display the result
        if verbose:
            output(result, data)

        print 'Report written to ' + PLOTFILE
        print 'Configuration written to ' + CFGFILE

    except:
        print 'Invalid data file.'
        usage()
        infile.close()
        sys.exit(1)

    infile.close()


main()
