-------------------------------------------------------------------
--           RAPID - RAPID ADA PORTABLE INTERFACE DESIGNER
--           TCL PEER FOR THE MCC TKI (ToolKit Interface) library
--           Copyright (C) 1999 Martin C. Carlisle.
--
-- RAPID is free software;  you can  redistribute it  and/or modify
-- it under terms of the  GNU General Public License as published
-- by the Free Software  Foundation;  either version 2,  or (at your
-- option) any later version.  RAPID is distributed in the hope that
-- it will be useful, but WITHOUT ANY WARRANTY;  without even the
-- implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
-- PURPOSE.  See the GNU General Public License for more details.
-- You should have  received  a copy of the GNU General Public License
-- distributed with RAPID; see file COPYING.  If not, write to the
-- Free Software Foundation,  59 Temple Place - Suite 330,  Boston,
-- MA 02111-1307, USA.
--
-- As a special exception, if other files instantiate generics from
-- this unit, or you link this unit with other files to produce an
-- executable, this unit does not by itself cause the resulting
-- executable to be covered by the GNU General Public License.
-- This exception does not however invalidate any other reasons
-- why the executable file might be covered by the GNU Public
-- License.  This exception does not apply to executables which
-- are GUI design tools, or that could act as a replacement
-- for RAPID.
------------------------------------------------------------------------------
with Text_IO;
with Ada.Unchecked_Conversion;
with mcc.text_entry_types;

package body mcc.tki.Widget.Text_Entry is
   Max_Text_Length : constant := 4096;
   Text            : String (1 .. Max_Text_Length);

   -----------
   -- Clear --
   -----------

   procedure Clear (Obj : in out Text_Entry) is
   begin
      peer.Eval (Obj.My_Peer.Name.all & " delete 0 end");
   end Clear;

   ------------
   -- Create --
   ------------

   procedure Create
     (Obj    : in out Text_Entry;
      Parent : in mcc.tki.Container.Container'Class;
      X      : in Integer;
      Y      : in Integer;
      Width  : in Natural;
      Height : in Natural)
   is
      type Parent_Access is access constant mcc.tki.Container.Container'Class;
      function Convert is new Ada.Unchecked_Conversion (
         Parent_Access,
         mcc.tki.Container.Container_Pointer);
   begin
      Obj.My_Peer := peer.Create_Peer (mcc.tki.Container.Get_Peer (Parent));
      peer.Eval ("entry " & Obj.My_Peer.Name.all);
      peer.Eval
        ("place " & Obj.My_Peer.Name.all &
         " -anchor nw" &
         " -x " & mcc.Img (X) &
         " -y " & mcc.Img (Y) &
         " -width " & mcc.Img (Width) &
         " -height " & mcc.Img (Height));
      Obj.Parent := Convert (Parent'Unchecked_Access);
   end Create;

   --------------
   -- Get_Text --
   --------------

   function Get_Text (Obj : in Text_Entry) return String is
      Last : Natural;
   begin
      peer.Eval ("set entry_text [ " & Obj.My_Peer.Name.all & " get ]");
      peer.Get_Value (Name => "entry_text", Result => Text, Last => Last);
      return Text (Text'First .. Last);
   end Get_Text;

   --------------
   -- Get_Text --
   --------------

   function Get_Text (Obj : in Text_Entry) return Integer is
   begin
      return Integer'Value (Get_Text (Obj));
   end Get_Text;

   function Get_Text (Obj : in Text_Entry) return Interfaces.Unsigned_32 is
   begin
      return Interfaces.Unsigned_32'Value (Get_Text (Obj));
   end Get_Text;

   --------------
   -- Get_Text --
   --------------

   function Get_Text (Obj : in Text_Entry) return Float is
   begin
      return Float'Value (Get_Text (Obj));
   end Get_Text;

   ---------------------
   -- Get_Text_Length --
   ---------------------

   function Get_Text_Length (Obj : in Text_Entry) return Natural is
      Result : String := Get_Text (Obj);
   begin
      return Result'Length;
   end Get_Text_Length;

   ---------------
   -- Highlight --
   ---------------

   procedure Highlight
     (Obj   : in out Text_Entry;
      Start : in Natural;
      Stop  : in Natural)
   is
   begin
      peer.Eval
        (Obj.My_Peer.Name.all & " selection range " &
         mcc.Img (Start) & ' ' & mcc.Img (Stop));
      peer.Eval ("focus " & Obj.My_Peer.Name.all & ASCII.LF & "update");
   end Highlight;

   procedure Highlight (Obj : in out Text_Entry) is
   begin
      peer.Eval (Obj.My_Peer.Name.all & " selection range 0 end");
      peer.Eval ("focus " & Obj.My_Peer.Name.all & ASCII.LF & "update");
   end Highlight;

   --------------
   -- Set_Text --
   --------------

   procedure Set_Text (Obj : in out Text_Entry; Text : in String) is
      use type peer.String_Pointer;
   begin
      if Obj.My_Peer.Name = null then
         Text_IO.Put_Line
           ("mcc.tki.Widget.Text_Entry.Set_Text(" &
            Text &
            "): Obj.My_Peer.Name is null");
         return;
      end if;
      Clear (Obj);
      peer.Eval
        (Obj.My_Peer.Name.all &
         " insert end """ &
         peer.Fix_Quotes (Text) &
         """");
   end Set_Text;

   ------------------------------------------------
   -- procedure Disable
   --
   -- prevent the user from interacting with the
   -- Text_Entry
   ------------------------------------------------
   procedure Disable (Obj : in out Text_Entry) is
   begin
      peer.Eval (Obj.My_Peer.Name.all & " configure -state disabled");
   end Disable;

   procedure Enable (Obj : in out Text_Entry) is
   begin
      peer.Eval (Obj.My_Peer.Name.all & " configure -state normal");
   end Enable;

   ------------------------------------------------
   -- procedure Set_Text
   --
   -- Set text for a text entry using
   -- image of integer (w/o leading spaces)
   ------------------------------------------------
   procedure Set_Text (Obj : in out Text_Entry; Text : in Integer) is
   begin
      Set_Text (Obj, mcc.Img (Text));
   end Set_Text;

   procedure Set_Text
     (Obj  : in out Text_Entry;
      Text : in Interfaces.Unsigned_32)
   is
   begin
      Set_Text (Obj, mcc.text_entry_types.Image (Text));
   end Set_Text;

   ------------------------------------------------
   -- procedure Set_Text
   --
   -- Set text for a text entry using
   -- image of float (w/o leading spaces)
   ------------------------------------------------
   procedure Set_Text (Obj : in out Text_Entry; Text : in Float) is
      Txt : constant String := Float'Image (Text);
   begin
      if Txt (1) = ' ' then
         -- set image of float w/o leading space
         Set_Text (Obj, Txt (2 .. Txt'Last));
      else
         Set_Text (Obj, Txt);
      end if;
   end Set_Text;

end mcc.tki.Widget.Text_Entry;
