-------------------------------------------------------------------
--           RAPID - RAPID ADA PORTABLE INTERFACE DESIGNER
--           TCL PEER FOR THE MCC TKI (ToolKit Interface) library
--           Copyright (C) 1999 Martin C. Carlisle.
--
-- RAPID is free software;  you can  redistribute it  and/or modify
-- it under terms of the  GNU General Public License as published
-- by the Free Software  Foundation;  either version 2,  or (at your
-- option) any later version.  RAPID is distributed in the hope that
-- it will be useful, but WITHOUT ANY WARRANTY;  without even the
-- implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
-- PURPOSE.  See the GNU General Public License for more details.
-- You should have  received  a copy of the GNU General Public License
-- distributed with RAPID; see file COPYING.  If not, write to the
-- Free Software Foundation,  59 Temple Place - Suite 330,  Boston,
-- MA 02111-1307, USA.
--
-- As a special exception, if other files instantiate generics from
-- this unit, or you link this unit with other files to produce an
-- executable, this unit does not by itself cause the resulting
-- executable to be covered by the GNU General Public License.
-- This exception does not however invalidate any other reasons
-- why the executable file might be covered by the GNU Public
-- License.  This exception does not apply to executables which
-- are GUI design tools, or that could act as a replacement
-- for RAPID.
------------------------------------------------------------------------------
with peer;
with Tcl;
with CArgv;
with Interfaces.C;
with My_Expanding_Array;
with Cargv_Helpers;
with Ada.Characters.Handling;
with Ada.Unchecked_Conversion;
package body mcc.tki.Widget.Scale is
   Change_Command : Tcl.Tcl_Command;

   Have_Change_Command : Boolean := False;

   ------------
   -- Create --
   ------------

   procedure Create
     (Obj         : in out Scale;
      Parent      : in mcc.tki.Container.Container'Class;
      X           : in Integer;
      Y           : in Integer;
      Width       : in Natural;
      Height      : in Natural;
      Min         : in Integer;
      Max         : in Integer;
      Mark_Every  : in Natural;
      Orientation : in Scale_Orientation := Horizontal;
      By          : in Natural           := 1)
   is
      type Parent_Access is access constant mcc.tki.Container.Container'Class;
      function Convert is new Ada.Unchecked_Conversion (
         Parent_Access,
         mcc.tki.Container.Container_Pointer);
   begin
      Obj.My_Peer := peer.Create_Peer (mcc.tki.Container.Get_Peer (Parent));
      peer.Eval
        ("scale " & Obj.My_Peer.Name.all &
         " -orient " &
         Ada.Characters.Handling.To_Lower
            (Scale_Orientation'Image (Orientation)) &
         " -from " & mcc.Img (Min) &
         " -to " & mcc.Img (Max) &
         " -tickinterval " & mcc.Img (Mark_Every) &
         " -resolution " & mcc.Img (By) &
         " -showvalue 0");
      peer.Eval
        ("place " & Obj.My_Peer.Name.all &
         " -anchor nw" &
         " -x " & mcc.Img (X) &
         " -y " & mcc.Img (Y) &
         " -width " & mcc.Img (Width) &
         " -height " & mcc.Img (Height));
      Obj.Parent := Convert (Parent'Unchecked_Access);
   end Create;

   ---------
   -- Set --
   ---------

   procedure Set (Obj : in out Scale; Location : in Integer) is
   begin
      peer.Eval (Obj.My_Peer.Name.all & " set " & mcc.Img (Location));
   end Set;

   ---------
   -- Get --
   ---------

   function Get (Obj : in Scale) return Integer is
      Return_Value : Integer;
   begin
      peer.Eval ("set scale_value [ " & Obj.My_Peer.Name.all & " get ]");
      peer.Get_Value (Name => "scale_value", Result => Return_Value);
      return Return_Value;
   end Get;

   --------------------------
   -- Tcl callback for
   -- Change events
   --------------------------
   function Change_Command_Function
     (ClientData : in Integer;
      Interp     : in Tcl.Tcl_Interp;
      Argc       : in Interfaces.C.int;
      Argv       : in CArgv.Chars_Ptr_Ptr)
      return       Interfaces.C.int;
   pragma Convention (C, Change_Command_Function);

   -- protocol for arguments will be
   -- 1st argument : lookup into Expanding_Array
   function Change_Command_Function
     (ClientData : in Integer;
      Interp     : in Tcl.Tcl_Interp;
      Argc       : in Interfaces.C.int;
      Argv       : in CArgv.Chars_Ptr_Ptr)
      return       Interfaces.C.int
   is
      Obj : Scale_Pointer;
   begin
      Obj :=
        Scale_Pointer (My_Expanding_Array.Retrieve
                          (Table    => My_Expanding_Array.Table,
                           Location => Cargv_Helpers.Argument (Argv, 1)));
      Obj.Change_Listener (Obj.all);
      return Tcl.TCL_OK;
   end Change_Command_Function;

   ------------------------
   -- Set_Scale_Listener --
   ------------------------

   procedure Set_Scale_Listener
     (Obj      : in Scale_Pointer;
      Listener : in Scale_Listener)
   is
   begin
      if Obj.My_Peer.Lookup = 0 then
         My_Expanding_Array.Insert
           (Table    => My_Expanding_Array.Table,
            Element  => Sized_Object_Pointer (Obj),
            Location => Obj.My_Peer.Lookup);
      end if;

      if not Have_Change_Command then
         Change_Command      :=
            peer.CreateCommands.Tcl_CreateCommand
              (peer.Get_Interp,
               "scalecommand",
               Change_Command_Function'Access,
               0,
               null);
         Have_Change_Command := True;
      end if;

      peer.Eval
        (Obj.My_Peer.Name.all & " configure -command " &
         "{scalecommand " & mcc.Img (Obj.My_Peer.Lookup) & "}");
      Obj.Change_Listener := Listener;
   end Set_Scale_Listener;

   function Get_Scale_Listener (Obj : in Scale'Class) return Scale_Listener is
   begin
      return Obj.Change_Listener;
   end Get_Scale_Listener;

end mcc.tki.Widget.Scale;
