---------------------------------------------------------------
--
--  RAPID - Rapid Ada Portable Interface Designer
--
--  GUI-WIDGET-STATIC.ADB
--  Description : GUI Widget Static Label
--
--  By: Martin Carlisle and Jonathan Busch
--
-- RAPID is free software; you can redistribute it and/or
-- modify it without restriction.  However, we ask that you
-- please retain the original author information, and clearly
-- indicate if it has been modified.
--
-- RAPID is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
--
-- As a special exception, if other files instantiate generics from
-- this unit, or you link this unit with other files to produce an
-- executable, this unit does not by itself cause the resulting
-- executable to be covered by the GNU General Public License.
-- This exception does not however invalidate any other reasons
-- why the executable file might be covered by the GNU Public
-- License.
--
-- Copyright (C) 1999, Martin C. Carlisle <carlislem@acm.org>
---------------------------------------------------------------
with Gui_Enum;
with File_Helpers;
with Widget_IO;
with mcc.Msg;
with label_dialog_window;
with Generate_Helpers;
with mcc.tki.Colors;

package body gui.Widget.Static is

   subtype Justification is mcc.tki.Widget.Label.Justification;

   use type mcc.tki.Widget.Widget_Pointer;
   use type mcc.tki.Colors.Optional_Color;

   -- reads information from file into Label,
   -- assumes keyword already read.
   procedure Read_Widget (Widget : in out Label) is
      Word : Word_Type;
      Last : Natural;
   begin
      Read_Widget (GUI_Widget (Widget));

      -- Get subclass of stuff in Widget record added in static ads
      Widget.Text              := Get_String;
      File_Helpers.Token_Index := File_Helpers.Token_Index + 1;

      --Check for justification & colors
      --  (if statement for RAPID Ver. 1.0 gui files compatibility)
      File_Helpers.Get_String (File_Helpers.Token_Index, Word, Last, False);
      if Last /= 0 then
         Widget.Justify := Justification'Value (Word (1 .. Last));
         File_Helpers.Token_Index := File_Helpers.Token_Index + 1;
         Widget_IO.Get_FG_BG_Color (Widget.Color.FG, Widget.Color.BG);
      else
         Widget.Justify  := mcc.tki.Widget.Label.Left;
         Widget.Color.FG := mcc.tki.Colors.Default;
         Widget.Color.BG := mcc.tki.Colors.Default;
      end if;
   end Read_Widget;

   -- Writes information to file from Label
   procedure Write_Widget (Widget : in Label) is
   begin
      File_Helpers.Put (Gui_Enum.Img (Gui_Enum.Label) & " ");
      Write_Widget (GUI_Widget (Widget));
      File_Helpers.Put_String (Widget.Text);
      File_Helpers.Put (" " & mcc.tki.Widget.Label.Just_Img (Widget.Justify));
      Widget_IO.Put_FG_BG_Color (Widget.Color.FG, Widget.Color.BG);
      File_Helpers.NL;
   end Write_Widget;

   -- wbw 6/6/99
   procedure Generate_Widget_Context_Clause (Widget : in Label) is
   begin
      Generate_Helpers.Generate_With ("mcc.tki.Widget.Label");
      Generate_Helpers.Generate_With ("mcc.tki.Colors");
   end Generate_Widget_Context_Clause;

   -- wbw 6/6/99
   procedure Generate_Widget_Declaration (Widget : in Label) is
   begin
      File_Helpers.P
        (Widget.Name.all & " : aliased mcc.tki.Widget.Label.Label;");
   end Generate_Widget_Declaration;

   -- wbw 5/10/99
   procedure Generate_Widget_Creation
     (Widget      : in Label;
      Window_Name : in String)
   is
      procedure P (Text : String) is
      begin
         File_Helpers.P (Text);
      end P;
   begin
      P ("mcc.tki.Widget.Label.Create");
      P ("  (Obj     => " & Widget.Name.all & ",");
      P ("   Parent  => " & Window_Name & ",");
      P ("   X       =>" & Integer'Image (Widget.x) & ",");
      P ("   Y       =>" & Integer'Image (Widget.y) & ",");
      P ("   Width   =>" & Integer'Image (Widget.Width) & ",");
      P ("   Height  =>" & Integer'Image (Widget.Height) & ",");
      P ("   Justify => mcc.tki.Widget.Label." &
         Justification'Image (Widget.Justify) & ",");
      P ("   Text    => " &
         """" &
         Generate_Helpers.Quote_String (Widget.Text.all) &
         """);");
      Generate_Helpers.Gen_Set_FG_BG_Color
        ("Label", Widget.Name.all, Widget.Color.FG, Widget.Color.BG);
   end Generate_Widget_Creation;

   -- display the widget to a window
   procedure Display_Widget
     (Widget    : in out Label;
      Container : in out mcc.tki.Container.Container'Class)
   is
   begin
      if Widget.The_Widget = null then
         Widget.The_Widget := new mcc.tki.Widget.Label.Label;
      end if;

      mcc.tki.Widget.Label.Create
        (Obj     => mcc.tki.Widget.Label.Label (Widget.The_Widget.all),
         Parent  => Container,
         X       => Widget.x,
         Y       => Widget.y,
         Width   => Widget.Width,
         Height  => Widget.Height,
         Text    => Widget.Text.all,
         Justify => Widget.Justify);
      if Widget.Color.FG /= mcc.tki.Colors.Default then
         mcc.tki.Widget.Set_Foreground_Color
           (Obj       => Widget.The_Widget.all,
            New_Color =>
              mcc.tki.Colors.Named_Color (Widget.Color.FG));
      end if;
      if Widget.Color.BG /= mcc.tki.Colors.Default then
         mcc.tki.Widget.Set_Background_Color
           (Obj       => Widget.The_Widget.all,
            New_Color =>
              mcc.tki.Colors.Named_Color (Widget.Color.BG));
      end if;
      Display_Widget (GUI_Widget (Widget), Container);
   exception
      when e : others =>
         mcc.Msg.Error (E, "Can't display " & Widget.Name.all);
   end Display_Widget;

   procedure Set_Properties (Widget : in out Label) is
   begin
      label_dialog_window.Generate_Window;
      Widget.Properties    := label_dialog_window.label_dialog_window'Access;
      Widget.Name_Entry    := label_dialog_window.entry1'Access;
      Widget.X_Entry       := label_dialog_window.entry2'Access;
      Widget.Y_Entry       := label_dialog_window.entry3'Access;
      Widget.Width_Entry   := label_dialog_window.entry4'Access;
      Widget.Height_Entry  := label_dialog_window.entry5'Access;
      Widget.Text_Entry    := label_dialog_window.entry6'Access;
      Widget.Justify_DD    := label_dialog_window.entry7'Access;
      Widget.Color.FG_DD   := label_dialog_window.fg_dd'Access;
      Widget.Color.BG_DD   := label_dialog_window.bg_dd'Access;

      Widget.Font_Label  := label_dialog_window.font_label'Access;
      Widget.Style_Label := label_dialog_window.font_style'Access;

      Set_Properties (GUI_Widget (Widget));
      if Widget.Text /= null then
         mcc.tki.Widget.Text_Entry.Set_Text
           (Obj  => Widget.Text_Entry.all,
            Text => Widget.Text.all);
      end if;
      mcc.tki.Widget.Dropdown.Select_Item
        (Obj    => Widget.Justify_DD.all,
         Number => Justification'Pos (Widget.Justify) + 1);
      mcc.tki.Colors.Fill_Colors (Widget.Color);
   end Set_Properties;

   procedure Apply_Properties (Widget : in out Label) is
   begin
      Apply_Properties (GUI_Widget (Widget));

      mcc.tki.Colors.Read_Colors (Widget.Color);
      declare
         Text : String :=
            mcc.tki.Widget.Text_Entry.Get_Text (Widget.Text_Entry.all);
      begin
         Widget.Text := new String'(Text);
      end;
      declare
         Justify : constant Natural :=
            mcc.tki.Widget.Dropdown.Get_Selected (Widget.Justify_DD.all);
      begin
         if Justify = 0 then
            Widget.Justify := mcc.tki.Widget.Label.Left;
         else
            Widget.Justify := Justification'Val (Justify - 1);
         end if;
      end;
   end Apply_Properties;

   procedure Check_Properties (Widget : in out Label; Ok : out Boolean) is
   begin
      Check_Properties (GUI_Widget (Widget), Ok);
      if Ok then
         Ok := (Widget.Text /= null) and then (Widget.Text.all /= "");
         if not Ok then
            mcc.tki.Widget.Text_Entry.Highlight (Widget.Text_Entry.all);
         end if;
      end if;

      if not Ok then
         mcc.tki.Bell;
      end if;
   end Check_Properties;

end Gui.Widget.Static;
