---------------------------------------------------------------
--
--  RAPID - Rapid Ada Portable Interface Designer
--
--  GUI-WIDGET-SCALE.ADB
--  Description : GUI Widget Scale
--
--  By: Martin Carlisle, Patrick Maes and W. Blair Watkinson II
--
-- RAPID is free software; you can redistribute it and/or
-- modify it without restriction.  However, we ask that you
-- please retain the original author information, and clearly
-- indicate if it has been modified.
--
-- RAPID is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
--
-- As a special exception, if other files instantiate generics from
-- this unit, or you link this unit with other files to produce an
-- executable, this unit does not by itself cause the resulting
-- executable to be covered by the GNU General Public License.
-- This exception does not however invalidate any other reasons
-- why the executable file might be covered by the GNU Public
-- License.
--
-- Copyright (C) 1999, Martin C. Carlisle <carlislem@acm.org>
---------------------------------------------------------------

with File_Helpers;
with Gui_Enum;
with Ada.Characters.Handling;
with Generate_Helpers;
with mcc.tki.Widget.Scale;
--with Ada.Exceptions;
--use Ada.Exceptions;

-- debugging
with mcc.Common_Dialogs;
with scale_dialog_Window;

package body gui.Widget.Scale is
   -- reads information from file into GUI_Widget,
   -- assumes keyword already read.

   procedure Read_Widget (Widget : in out Scale) is
   begin
      Read_Widget (GUI_Widget (Widget));

      Widget.Min               :=
         Natural'Value (File_Helpers.Token (File_Helpers.Token_Index).all);
      File_Helpers.Token_Index := File_Helpers.Token_Index + 1;

      Widget.Max               :=
         Natural'Value (File_Helpers.Token (File_Helpers.Token_Index).all);
      File_Helpers.Token_Index := File_Helpers.Token_Index + 1;

      Widget.By                :=
         Natural'Value (File_Helpers.Token (File_Helpers.Token_Index).all);
      File_Helpers.Token_Index := File_Helpers.Token_Index + 1;

      Widget.Mark_Every        :=
         Natural'Value (File_Helpers.Token (File_Helpers.Token_Index).all);
      File_Helpers.Token_Index := File_Helpers.Token_Index + 1;

      Widget.Orientation       :=
         mcc.tki.Widget.Scale.Scale_Orientation'Value
           (File_Helpers.Token (File_Helpers.Token_Index).all);
      File_Helpers.Token_Index := File_Helpers.Token_Index + 1;

   end Read_Widget;

   -- Writes information to file from GUI_Widget

   procedure Write_Widget (Widget : in Scale) is
   begin
      File_Helpers.Put (Gui_Enum.Img (Gui_Enum.Scale) & " ");
      Write_Widget (GUI_Widget (Widget));
      File_Helpers.Put (Natural'Image (Widget.Min));
      File_Helpers.Put (Natural'Image (Widget.Max));
      File_Helpers.Put (Natural'Image (Widget.By));
      File_Helpers.Put (Natural'Image (Widget.Mark_Every));
      File_Helpers.P
        (" " &
         mcc.tki.Widget.Scale.Scale_Orientation'Image (Widget.Orientation),
         Indent => False);
   end Write_Widget;

   procedure Generate_Widget_Context_Clause (Widget : in Scale) is
   begin
      Generate_Helpers.Generate_With ("mcc.tki.Widget.Scale");
   end Generate_Widget_Context_Clause;

   -- wbw 6/6/99
   procedure Generate_Widget_Declaration (Widget : in Scale) is
   begin
      File_Helpers.P
        (Widget.Name.all & " : aliased mcc.tki.Widget.Scale.Scale;");
   end Generate_Widget_Declaration;

   procedure Generate_Widget_Creation
     (Widget      : in Scale;
      Window_Name : in String)
   is
      procedure P (Text : String) is
      begin
         File_Helpers.P (Text);
      end P;
   begin
      P ("mcc.tki.Widget.Scale.Create");
      P ("  (Obj        => " & Widget.Name.all & ",");
      P ("   Parent     => " & Window_Name & ",");
      P ("   X          => " & mcc.Img (Widget.x) & ",");
      P ("   Y          => " & mcc.Img (Widget.y) & ",");
      P ("   Width      => " & mcc.Img (Widget.Width) & ",");
      P ("   Height     => " & mcc.Img (Widget.Height) & ",");
      P ("   Min        => " & mcc.Img (Widget.Min) & ",");
      P ("   Max        => " & mcc.Img (Widget.Max) & ",");
      P ("   Mark_Every => " & mcc.Img (Widget.Mark_Every) & ",");
      P ("   Orientation=> mcc.tki.Widget.Scale." &
         mcc.tki.Widget.Scale.Scale_Orientation'Image (Widget.Orientation) &
         ",");
      P ("   By         => " & mcc.Img (Widget.By) & ");");
   end Generate_Widget_Creation;

   -- display the widget to a window
   procedure Display_Widget
     (Widget    : in out Scale;
      Container : in out mcc.tki.Container.Container'Class)
   is
      use type mcc.tki.Widget.Widget_Pointer;
   begin
      if Widget.The_Widget = null then
         Widget.The_Widget := new mcc.tki.Widget.Scale.Scale;
      end if;

      mcc.tki.Widget.Scale.Create
        (Obj         => mcc.tki.Widget.Scale.Scale (Widget.The_Widget.all),
         Parent      => Container,
         X           => Widget.x,
         Y           => Widget.y,
         Width       => Widget.Width,
         Height      => Widget.Height,
         Min         => Widget.Min,
         Max         => Widget.Max,
         Mark_Every  => Widget.Mark_Every,
         Orientation => Widget.Orientation,
         By          => Widget.By);
      Display_Widget (GUI_Widget (Widget), Container);
   exception
      when others =>
         mcc.Common_Dialogs.Ok_Box ("Can't display: " & Widget.Name.all);
   end Display_Widget;

   procedure Set_Properties (Widget : in out Scale) is
      use mcc.tki.Widget.Scale;
   begin
      scale_dialog_Window.Generate_Window;
      Widget.Properties   := scale_dialog_Window.scale_dialog_Window'Access;
      Widget.Name_Entry   := scale_dialog_Window.entry1'Access;
      Widget.X_Entry      := scale_dialog_Window.entry2'Access;
      Widget.Y_Entry      := scale_dialog_Window.entry3'Access;
      Widget.Width_Entry  := scale_dialog_Window.entry4'Access;
      Widget.Height_Entry := scale_dialog_Window.entry5'Access;
      Widget.Min_Entry    := scale_dialog_Window.entry6'Access;
      Widget.Max_Entry    := scale_dialog_Window.entry7'Access;
      Widget.By_Entry     := scale_dialog_Window.entry8'Access;
      Widget.Mark_Entry   := scale_dialog_Window.entry9'Access;
      Widget.Orient_Group := scale_dialog_Window.Orientation'Access;
      Widget.Horiz_Radio  := scale_dialog_Window.horizontal'Access;
      Widget.Vert_Radio   := scale_dialog_Window.vertical'Access;

      Set_Properties (GUI_Widget (Widget));

      mcc.tki.Widget.Text_Entry.Set_Text
        (Obj  => Widget.Min_Entry.all,
         Text => Widget.Min);

      mcc.tki.Widget.Text_Entry.Set_Text
        (Obj  => Widget.Max_Entry.all,
         Text => Widget.Max);

      mcc.tki.Widget.Text_Entry.Set_Text
        (Obj  => Widget.By_Entry.all,
         Text => Widget.By);

      mcc.tki.Widget.Text_Entry.Set_Text
        (Obj  => Widget.Mark_Entry.all,
         Text => Widget.Mark_Every);

      if Widget.Orientation = Horizontal then
         mcc.tki.Widget.Button.Radio.Select_Radio (Widget.Horiz_Radio.all);
      else
         mcc.tki.Widget.Button.Radio.Select_Radio (Widget.Vert_Radio.all);
      end if;

   end Set_Properties;

   procedure Apply_Properties (Widget : in out Scale) is
      use type mcc.tki.Widget.Button.Radio.Radio_Pointer;
   begin
      Apply_Properties (GUI_Widget (Widget));

      begin
         Widget.Min :=
            mcc.tki.Widget.Text_Entry.Get_Text (Widget.Min_Entry.all);
      exception
         when others =>
            Widget.Min := Integer'Last;
      end;

      begin
         Widget.Max :=
            mcc.tki.Widget.Text_Entry.Get_Text (Widget.Max_Entry.all);
      exception
         when others =>
            Widget.Max := Integer'Last;
      end;

      begin
         Widget.By :=
            mcc.tki.Widget.Text_Entry.Get_Text (Widget.By_Entry.all);
      exception
         when others =>
            Widget.By := -99;
      end;

      begin
         Widget.Mark_Every :=
            mcc.tki.Widget.Text_Entry.Get_Text (Widget.Mark_Entry.all);
      exception
         when others =>
            Widget.Mark_Every := -99;
      end;

      if mcc.tki.Widget.Button.Radio.Get_Selected (Widget.Orient_Group.all) =
         Widget.Horiz_Radio
      then
         Widget.Orientation := mcc.tki.Widget.Scale.Horizontal;
      else
         Widget.Orientation := mcc.tki.Widget.Scale.Vertical;
      end if;
   end Apply_Properties;

   procedure Check_Properties (Widget : in out Scale; Ok : out Boolean) is
   begin
      Check_Properties (GUI_Widget (Widget), Ok);
      if Ok and then Widget.Min = Integer'Last then
         Ok := False;
         mcc.tki.Widget.Text_Entry.Highlight (Widget.Min_Entry.all);
      end if;

      if Ok and then Widget.Max = Integer'Last then
         Ok := False;
         mcc.tki.Widget.Text_Entry.Highlight (Widget.Max_Entry.all);
      end if;

      if Ok and then Widget.Mark_Every < 0 then
         Ok := False;
         mcc.tki.Widget.Text_Entry.Highlight (Widget.Mark_Entry.all);
      end if;

      if Ok and then Widget.By < 0 then
         Ok := False;
         mcc.tki.Widget.Text_Entry.Highlight (Widget.By_Entry.all);
      end if;

      if not Ok then
         mcc.tki.Bell;
      end if;
   end Check_Properties;

end Gui.Widget.Scale;
