---------------------------------------------------------------
--
--  RAPID - Rapid Ada Portable Interface Designer
--
--  GUI-WIDGET-BUTTON.ADB
--  Description : Root of GUI Widget Button Hierarchy
--
--  Copyright (C) 2003, Martin C. Carlisle <carlislem@acm.org>
--
-- RAPID is free software; you can redistribute it and/or
-- modify it without restriction.  However, we ask that you
-- please retain the original author information, and clearly
-- indicate if it has been modified.
--
-- RAPID is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
--
-- As a special exception, if other files instantiate generics from
-- this unit, or you link this unit with other files to produce an
-- executable, this unit does not by itself cause the resulting
-- executable to be covered by the GNU General Public License.
-- This exception does not however invalidate any other reasons
-- why the executable file might be covered by the GNU Public
-- License.
---------------------------------------------------------------
-- Change log:
-- 08/21/98 (mcc) : added test to remove ".gif" if specified
---------------------------------------------------------------
with Ada.Strings.Fixed;
with File_Helpers;
with Gui_Enum;
with picturebutton_dialog_window;
with textbutton_dialog_window;
with Generate_Helpers;
with state;
with Ada.Characters.Handling;     use Ada.Characters.Handling;

-- debugging
with mcc.Common_Dialogs;
with mcc.tki.Widget.Button.Text;
with mcc.tki.Widget.Button.Picture; use type mcc.tki.Widget.Widget_Pointer;

package body gui.Widget.Button is
   -- reads information from file into GUI_Widget,
   -- assumes keyword already read.

   -- for novice mode, update actions when window name changes.
   procedure Update_Action
     (Widget     : in out Button;
      Old_Window : String;
      New_Window : String)
   is
      First : constant Positive := Widget.Action'First;
   begin
      if Widget.Action'Length <= Old_Window'Length + 8 then
         return;
      end if;
      if To_Lower
            (Widget.Action (First .. First + Old_Window'Length + 7)) =
         To_Lower (Old_Window) & "_actions"
      then
         Widget.Action :=
           new String'
             (New_Window & "_actions" &
              Widget.Action (First + Old_Window'Length + 8 ..
                             Widget.Action'Last));
      end if;
   end Update_Action;

   procedure Read_Widget (Widget : in out Button) is
      Word : Word_Type;
      Last : Natural;
   begin -- Read_Widget
      Read_Widget (GUI_Widget (Widget));

      Widget.Action            := Get_String;
      File_Helpers.Token_Index := File_Helpers.Token_Index + 1;
   end Read_Widget;

   -- Writes information to file from GUI_Widget

   procedure Write_Widget (Widget : in Button) is

   begin -- Write_Widget
      Write_Widget (GUI_Widget (Widget));
      File_Helpers.Put_String (Widget.Action);
   end Write_Widget;

   -- wbw 6/6/99
   procedure Generate_Action_Context_Clause
     (Widget : in Button)
   is
   begin
      if Widget.Action /= null then
         Generate_Helpers.Generate_With_For_FQN (Widget.Action.all);
      end if;
   end Generate_Action_Context_Clause;

   -- wbw 5/10/99
   procedure Generate_Widget_Creation
     (Widget      : in Button;
      Window_Name : in String)
   is
   begin
      if Widget.Action = null then
         return;
      end if;
      File_Helpers.P ("mcc.tki.Widget.Button.Set_Push_Callback");
      File_Helpers.P ("  (Obj      => " & Widget.Name.all & "'Access,");
      File_Helpers.P ("   Callback => ", Newline => False);
      Generate_Helpers.Generate_Subprogram_Name (Widget.Action.all);
      File_Helpers.P ("'access);", Indent => False);
   end Generate_Widget_Creation;

   -- SAG 05.Sep.2000
   procedure Generate_Callback_Action (Widget : in Button) is
      Dot_Notation  : Natural := 0;
      Low_Case_Text : String  :=
         Ada.Characters.Handling.To_Lower (Widget.Action.all);
   begin
      Dot_Notation := Ada.Strings.Fixed.Index (Widget.Action.all, ".");

      if Low_Case_Text = "close_window"
        or else Low_Case_Text = "generate_read_action"
        or else Low_Case_Text = "generate_fill_action"
        or else Low_Case_Text = "generate_window"
        or else Low_Case_Text = "fill_window"
        or else Low_Case_Text = "generate_and_fill_window"
        or else Low_Case_Text = "read_window"
        or else Low_Case_Text = "ok"
        or else Dot_Notation > 0
      then
         return;
      end if;

      File_Helpers.Put ("   procedure ");
      Generate_Helpers.Generate_Subprogram_Name (Widget.Action.all);
      File_Helpers.Put (" (Obj : in out mcc.tki.Widget.Button.Button'Class)");
      File_Helpers.P (" is separate ;", Indent => False);
   end Generate_Callback_Action;

   procedure Set_Properties (Widget : in out Button) is
   begin
      Set_Properties (GUI_Widget (Widget));
      if state.Get_Current_Window.Novice_Mode then
         mcc.tki.Widget.Text_Entry.Set_Text
           (Obj  => Widget.Action_Entry.all,
            Text => "disabled for novice");
         mcc.tki.Widget.Text_Entry.Disable (Widget.Action_Entry.all);
      elsif Widget.Action /= null then
         mcc.tki.Widget.Text_Entry.Set_Text
           (Obj  => Widget.Action_Entry.all,
            Text => Widget.Action.all);
      end if;
   end Set_Properties;

   procedure Apply_Properties (Widget : in out Button) is
   begin
      Apply_Properties (GUI_Widget (Widget));
      declare
         Action : String :=
            mcc.tki.Widget.Text_Entry.Get_Text (Widget.Action_Entry.all);
      begin
         if Action /= "disabled for novice" then
            Widget.Action := new String'(Action);
         else
            Widget.Action :=
              new String'(state.Get_Current_Window.Window_Name.all &
                          "_Actions." &
                          Widget.Name.all &
                          "_Pushed");
         end if;
      end;
   end Apply_Properties;

   procedure Check_Properties (Widget : in out Button; Ok : out Boolean) is
   begin
      Check_Properties (GUI_Widget (Widget), Ok);
      if Ok then
         Ok := state.Get_Current_Window.Novice_Mode or
               ((Widget.Action /= null) and then (Widget.Action.all /= ""));
         if not Ok then
            mcc.tki.Widget.Text_Entry.Highlight (Widget.Action_Entry.all);
         end if;
      end if;

      if not Ok then
         mcc.tki.Bell;
      end if;
   end Check_Properties;

   -- reads information from file into GUI_Widget,
   -- assumes keyword already read.
   procedure Read_Widget (Widget : in out Text_Button) is
      Word : Word_Type;
      Last : Natural;
   begin -- Read_Widget
      Read_Widget (Button (Widget));
      Widget.Text              := Get_String;
      File_Helpers.Token_Index := File_Helpers.Token_Index + 1;
   end Read_Widget;

   -- Writes information to file from GUI_Widget

   procedure Write_Widget (Widget : in Text_Button) is
   begin -- Write_Widget
      File_Helpers.Put (Gui_Enum.Img (Gui_Enum.TextButton) & " ");
      Write_Widget (Button (Widget));
      File_Helpers.Put_String (Widget.Text);
      File_Helpers.P;
   end Write_Widget;

   procedure Generate_Action_Context_Clause (Widget : in Text_Button) is
   begin
      Generate_Action_Context_Clause (Button (Widget));
   end Generate_Action_Context_Clause;

   -- wbw 6/6/99
   procedure Generate_Widget_Context_Clause (Widget : in Text_Button) is
   begin
      Generate_Helpers.Generate_With ("mcc.tki.Widget.Button.Text");
   end Generate_Widget_Context_Clause;

   -- wbw 6/6/99
   procedure Generate_Widget_Declaration (Widget : in Text_Button) is
   begin
      File_Helpers.P
        (Widget.Name.all &
         " : aliased mcc.tki.Widget.Button.Text.Text_Button;");
   end Generate_Widget_Declaration;

   -- wbw 5/10/99
   procedure Generate_Widget_Creation
     (Widget      : in Text_Button;
      Window_Name : in String)
   is
      procedure P (Text : String) is
      begin
         File_Helpers.P (Text);
      end P;
   begin
      File_Helpers.Set_Indent (2);
      P ("mcc.tki.Widget.Button.Text.Create");
      P ("  (Obj    => " & Widget.Name.all & ",");
      P ("   Parent => " & Window_Name & ",");
      P ("   X      =>" & Integer'Image (Widget.x) & ",");
      P ("   Y      =>" & Integer'Image (Widget.y) & ",");
      P ("   Width  =>" & Integer'Image (Widget.Width) & ",");
      P ("   Height =>" & Integer'Image (Widget.Height) & ",");
      P ("   Text   => """ &
         Generate_Helpers.Quote_String (Text_Button (Widget).Text.all) &
         """);");
      Generate_Widget_Creation (Button (Widget), Window_Name);
   end Generate_Widget_Creation;

   -- display the widget to a window
   procedure Display_Widget
     (Widget    : in out Text_Button;
      Container : in out mcc.tki.Container.Container'Class)
   is
   begin
      if Widget.The_Widget = null then
         Widget.The_Widget := new mcc.tki.Widget.Button.Text.Text_Button;
      end if;

      mcc.tki.Widget.Button.Text.Create
        (Obj    =>
           mcc.tki.Widget.Button.Text.Text_Button (Widget.The_Widget.all),
         Parent => Container,
         X      => Widget.x,
         Y      => Widget.y,
         Width  => Widget.Width,
         Height => Widget.Height,
         Text   => Widget.Text.all);

      Display_Widget (GUI_Widget (Widget), Container);
   exception
      when others =>
         mcc.Common_Dialogs.Ok_Box ("Can't display: " & Widget.Name.all);
   end Display_Widget;

   procedure Set_Properties (Widget : in out Text_Button) is
   begin
      textbutton_dialog_window.Generate_Window;
      Widget.Properties   :=
        textbutton_dialog_window.textbutton_dialog_window'Access;
      Widget.Name_Entry   := textbutton_dialog_window.entry1'Access;
      Widget.X_Entry      := textbutton_dialog_window.entry2'Access;
      Widget.Y_Entry      := textbutton_dialog_window.entry3'Access;
      Widget.Width_Entry  := textbutton_dialog_window.entry4'Access;
      Widget.Height_Entry := textbutton_dialog_window.entry5'Access;
      Widget.Action_Entry := textbutton_dialog_window.entry6'Access;
      Widget.Text_Entry   := textbutton_dialog_window.entry7'Access;

      Widget.Font_Label  := textbutton_dialog_window.font_label'Access;
      Widget.Style_Label := textbutton_dialog_window.font_style'Access;

      Set_Properties (Button (Widget));
      if Widget.Text /= null then
         mcc.tki.Widget.Text_Entry.Set_Text
           (Obj  => Widget.Text_Entry.all,
            Text => Widget.Text.all);
      end if;
   end Set_Properties;

   procedure Apply_Properties (Widget : in out Text_Button) is
   begin
      Apply_Properties (Button (Widget));
      declare
         Text : String :=
            mcc.tki.Widget.Text_Entry.Get_Text (Widget.Text_Entry.all);
      begin
         Widget.Text := new String'(Text);
      end;
   end Apply_Properties;

   procedure Check_Properties
     (Widget : in out Text_Button;
      Ok     : out Boolean)
   is
   begin
      Check_Properties (Button (Widget), Ok);
      if Ok then
         Ok := (Widget.Text /= null) and then (Widget.Text.all /= "");
         if not Ok then
            mcc.tki.Widget.Text_Entry.Highlight (Widget.Text_Entry.all);
         end if;
      end if;

      if not Ok then
         mcc.tki.Bell;
      end if;
   end Check_Properties;

   -- reads information from file into GUI_Widget,
   -- assumes keyword already read.

   procedure Read_Widget (Widget : in out Picture_Button) is
      Word : Word_Type;
      Last : Natural;
   begin -- Read_Widget
      Read_Widget (Button (Widget));
      Widget.Picture           := Get_String;
      File_Helpers.Token_Index := File_Helpers.Token_Index + 1;
      File_Helpers.Get_String (File_Helpers.Token_Index, Word, Last);
      if Last >= Word'First then
         Widget.Tooltip           := new String'(Word (1 .. Last));
         File_Helpers.Token_Index := File_Helpers.Token_Index + 1;
      else
         Widget.Tooltip := null;
      end if;
   end Read_Widget;

   -- Writes information to file from GUI_Widget

   procedure Write_Widget (Widget : in Picture_Button) is
   begin -- Write_Widget
      File_Helpers.Put (Gui_Enum.Img (Gui_Enum.PictureButton) & " ");
      Write_Widget (Button (Widget));
      File_Helpers.Put_String (Widget.Picture);
      File_Helpers.Put_String (Widget.Tooltip);
      File_Helpers.P;
   end Write_Widget;

   procedure Generate_Action_Context_Clause (Widget : in Picture_Button) is
   begin
      Generate_Action_Context_Clause (Button (Widget));
   end Generate_Action_Context_Clause;

   -- wbw 6/6/99
   procedure Generate_Widget_Context_Clause (Widget : in Picture_Button) is
   begin
      Generate_Helpers.Generate_With ("mcc.tki.Widget.Button.Picture");
      Generate_Helpers.Generate_With ("mcc.tki.Image");
   end Generate_Widget_Context_Clause;

   -- wbw 6/6/99
   procedure Generate_Widget_Declaration (Widget : in Picture_Button) is
   begin
      File_Helpers.Set_Indent (1);
      File_Helpers.P
        (Widget.Name.all &
         " : aliased mcc.tki.Widget.Button.Picture.Picture_Button;");
      File_Helpers.P
        (Generate_Helpers.Undot_Name (Widget.Picture.all) &
         "_Image : aliased mcc.tki.Image.External_Image;");
   end Generate_Widget_Declaration;

   -- wbw 5/10/99
   procedure Generate_Widget_Creation
     (Widget      : in Picture_Button;
      Window_Name : in String)
   is
      procedure P (Text : String) is
      begin
         File_Helpers.P (Text);
      end P;
   begin
      File_Helpers.Set_Indent (2);
      P ("mcc.tki.Image.Create");
      P ("  (Obj  => " &
         Generate_Helpers.Undot_Name (Widget.Picture.all) &
         "_Image,");
      P ("   Name => " & """" & Widget.Picture.all & """);");
      P ("mcc.tki.Widget.Button.Picture.Create");
      P ("  (Obj    => " & Widget.Name.all & ",");
      P ("   Parent => " & Window_Name & ",");
      P ("   X      =>" & Integer'Image (Widget.x) & ",");
      P ("   Y      =>" & Integer'Image (Widget.y) & ",");
      P ("   Width  =>" & Integer'Image (Widget.Width) & ",");
      P ("   Height =>" & Integer'Image (Widget.Height) & ",");
      P ("   Image  => " &
         Generate_Helpers.Undot_Name (Widget.Picture.all) &
         "_Image);");
      if Widget.Tooltip /= null then
         P ("mcc.tki.Widget.Button.Picture.Set_Tooltip_Text");
         P ("  (Obj    => " & Widget.Name.all & ",");
         P ("   Text  => """ &
            Generate_Helpers.Quote_String (Widget.Tooltip.all) &
            """);");
      end if;
      Generate_Widget_Creation (Button (Widget), Window_Name);
   end Generate_Widget_Creation;

   -- display the widget to a window
   procedure Display_Widget
     (Widget    : in out Picture_Button;
      Container : in out mcc.tki.Container.Container'Class)
   is
      Excepted : Boolean := False;
   begin

      begin
         mcc.tki.Image.Create (Widget.Image, Widget.Picture.all);
      exception
         when others =>
            Excepted := True;
      end;

      if not Excepted then
         if Widget.The_Widget = null
           or else not (Widget.The_Widget.all in 
              mcc.tki.Widget.Button.Picture.Picture_Button'Class)
         then
            Widget.The_Widget :=
              new mcc.tki.Widget.Button.Picture.Picture_Button;
         end if;

         mcc.tki.Widget.Button.Picture.Create
           (Obj    =>
              mcc.tki.Widget.Button.Picture.Picture_Button (Widget.The_Widget.
all),
            Parent => Container,
            X      => Widget.x,
            Y      => Widget.y,
            Width  => Widget.Width,
            Height => Widget.Height,
            Image  => Widget.Image);
      else
         if Widget.The_Widget = null
           or else not (Widget.The_Widget.all in 
              mcc.tki.Widget.Button.Text.Text_Button'Class)
         then
            Widget.The_Widget := new mcc.tki.Widget.Button.Text.Text_Button;
         end if;

         mcc.tki.Widget.Button.Text.Create
           (Obj    =>
              mcc.tki.Widget.Button.Text.Text_Button (Widget.The_Widget.all),
            Parent => Container,
            X      => Widget.x,
            Y      => Widget.y,
            Width  => Widget.Width,
            Height => Widget.Height,
            Text   => Widget.Picture.all);
      end if;

      if Widget.Tooltip /= null then
         mcc.tki.Widget.Set_Tooltip_Text
           (Obj  => Widget.The_Widget.all,
            Text => Widget.Tooltip.all);
      end if;
      Display_Widget (GUI_Widget (Widget), Container);
   exception
      when others =>
         mcc.Common_Dialogs.Ok_Box ("Can't display: " & Widget.Name.all);
   end Display_Widget;

   procedure Set_Properties (Widget : in out Picture_Button) is
   begin
      picturebutton_dialog_window.Generate_Window;
      Widget.Properties    :=
        picturebutton_dialog_window.picturebutton_dialog_window'Access;
      Widget.Name_Entry    := picturebutton_dialog_window.entry1'Access;
      Widget.X_Entry       := picturebutton_dialog_window.entry2'Access;
      Widget.Y_Entry       := picturebutton_dialog_window.entry3'Access;
      Widget.Width_Entry   := picturebutton_dialog_window.entry4'Access;
      Widget.Height_Entry  := picturebutton_dialog_window.entry5'Access;
      Widget.Action_Entry  := picturebutton_dialog_window.entry6'Access;
      Widget.Picture_Entry := picturebutton_dialog_window.entry7'Access;
      Widget.Tip_Entry     := picturebutton_dialog_window.entry8'Access;

      Set_Properties (Button (Widget));
      if Widget.Picture /= null then
         mcc.tki.Widget.Text_Entry.Set_Text
           (Obj  => Widget.Picture_Entry.all,
            Text => Widget.Picture.all);
      end if;
      if Widget.Tooltip /= null then
         mcc.tki.Widget.Text_Entry.Set_Text
           (Obj  => Widget.Tip_Entry.all,
            Text => Widget.Tooltip.all);
      end if;
   end Set_Properties;

   procedure Apply_Properties (Widget : in out Picture_Button) is
   begin
      Apply_Properties (Button (Widget));
      declare
         Picture : String :=
            mcc.tki.Widget.Text_Entry.Get_Text (Widget.Picture_Entry.all);
      begin
         Widget.Picture := new String'(Picture);
      end;
      declare
         Tooltip : String :=
            mcc.tki.Widget.Text_Entry.Get_Text (Widget.Tip_Entry.all);
      begin
         if Tooltip'Length > 0 then
            Widget.Tooltip := new String'(Tooltip);
         else
            Widget.Tooltip := null;
         end if;
      end;
   end Apply_Properties;

   procedure Check_Properties
     (Widget : in out Picture_Button;
      Ok     : out Boolean)
   is
   begin
      Check_Properties (Button (Widget), Ok);
      if Ok then
         Ok := (Widget.Picture /= null) and then (Widget.Picture.all /= "");
         if not Ok then
            mcc.tki.Widget.Text_Entry.Highlight (Widget.Picture_Entry.all);
         end if;
      end if;

      if not Ok then
         mcc.tki.Bell;
      end if;
   end Check_Properties;

end Gui.Widget.Button;
