#!/usr/bin/perl
#
# rapid2glade.pl    RAPID GUI to GladeXML file converter
# Author/Copyright: (C) 2009, O. Kellogg <okellogg@users.sourceforge.net>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
#
# ChangeLog:
#
$Version = "20100402";
#            Added translation of PICTUREBUTTON.
#            NB: glade-2 expects to find the icon pixmaps in a
#            subdirectory "pixmaps" of the glade project dir.
#
# 20090130   Fixed translation of DROPDOWN (now produces GtkComboBox).
#
# 20090110   Simplified handling of menubar:
#            Inside the top level GtkWindow, there is now a GtkVBox:
#            * When there is no menubar the VBox has a single entry,
#              namely the GtkFixed container for the application widgets
#            * When there is a menubar the VBox has two entries, the first
#              holds the menubar and the second holds the GtkFixed container.
#
# 20090107   Added handling of CHECKBUTTON and RADIOBUTTON.
#            Still to do: LISTBOX.
#

$indentlevel = 0;

sub whitespace {
    return (' ' x (2 * $indentlevel));
}

sub dent {
    my $line = shift;
    print(whitespace . $line . "\n");
}

sub indent {
    dent shift;
    $indentlevel++;
}

sub dedent {
    $indentlevel--;
    dent shift;
}

# Array of tokens into which each line is split
@token = ();
# Flag for presence of menubar
$have_menubar = 0;
# Widget counter auxiliary variable
$widgetcnt = 0;

sub prologue {
    my($widget, $id) = @_;
    if ($id) {
        $id .= "_$widgetcnt";
        $widgetcnt++;
    } else {
        $id = $token[1];
    }
    indent('<child>');
    indent('<widget class="' . $widget . '" id="' . $id . '">');
    dent('<property name="width_request">' . $token[4] . '</property>');
    dent('<property name="height_request">' . $token[5] . '</property>');
    dent('<property name="visible">True</property>');
}

sub epilogue {
    dedent('</widget>');
    indent('<packing>');
    dent('<property name="x">' . $token[2] . '</property>');
    dent('<property name="y">' . $token[3] . '</property>');
    dedent('</packing>');
    dedent('</child>');
}

sub string2xml {
    my $name = shift;
    $name =~ s/^[^"]*"//;
    $name =~ s/"[^"]*$//;
    if ($name =~ /\&/) {
        $name =~ s/\&/&amp;/g;
    } else {
        $name =~ s/</&lt;/g;
        $name =~ s/>/&gt;/g;
    }
    $name =~ s/""/&quot;/g;
    return $name;
}

# Main program
@ARGV or die "supply input file name\n";
$inputfile = $ARGV[0];
open(IN, "<$inputfile") or die "cannot open file $inputfile\n";
dent '<?xml version="1.0"?>';
indent '<glade-interface>';
dent('<!-- generated by rapid2glade.pl version ' . $Version . ' -->');
dent '<!-- interface-requires gtk+ 2.8 -->';
dent '<!-- interface-naming-policy project-wide -->';
$line = <IN>;
chop $line;
@token = split /\s+/, $line;
$token[0] eq 'WINDOW' or die "expecting WINDOW in first line of input file\n";
# WINDOW "platform_specific_data_normal" "Platform Specific Data Normal"  973 499 
$token[1] =~ s/"//g;
indent('<widget class="GtkWindow" id="' . $token[1] . '">');
$name = $line;
$name =~ s/^WINDOW "[^"]+" "//;
$name =~ s/".*$//;
dent('<property name="title" translatable="yes">' . string2xml($name) . '</property>');
dent('<property name="type">GTK_WINDOW_TOPLEVEL</property>');
# dent('<property name="default_width">' . $width . '</property>');
# dent('<property name="default_height">' . $height . '</property>');
dent('<property name="resizable">True</property>');
indent '<child>';
indent('<widget class="GtkVBox" id="vbox0">');
dent('<property name="visible">True</property>');
dent('<property name="homogeneous">False</property>');
dent('<property name="spacing">0</property>');
$window_width = 0;   # will usually be overwritten by true width
$window_height = 0;  # not currently used - may be used in the future
# Try to find the window width in the WINDOW line
my $i;
for ($i = $#token; $i > 3; $i--) {
    if ($token[$i] =~ /^\d+$/ and $token[$i - 1] =~ /^\d+$/) {
        $window_width = $token[$i - 1];
        $window_height = $token[$i];
        last;
    }
}
while (($line = <IN>)) {
    chop $line;
    next if ($line =~ /^\s*$/);
    if ($line =~ /^\s*(TRUE|FALSE)$/) {
        # WINDOW novice-mode indicator is on separate line pre 3.3 file format
        next;
    }
    @token = split /\s+/, $line;
    if ($token[0] =~ /^\d+$/) {
        # WINDOW width and height are on separate line pre 3.3 file format
        $window_width = $token[0];
        $window_height = $token[1];
        next;
    }
    my $keyword = $token[0];
    last if ($keyword eq 'WIDGETS');
    if ($keyword eq 'MENUBAR') {
        $have_menubar = 1;
        indent('<child>');
        indent('<widget class="GtkMenuBar" id="menubar' . $widgetcnt . '">');
        $widgetcnt++;
        unless ($window_width) {  # should not happen...
            $window_width = 320;  # but if it does then give it some non-zero value
        }
        dent('<property name="visible">True</property>');
        dent('<property name="pack_direction">GTK_PACK_DIRECTION_LTR</property>');
        dent('<property name="child_pack_direction">GTK_PACK_DIRECTION_LTR</property>');
    } elsif ($keyword eq 'MENU') {
        # MENU "File" 1 "null"
        indent('<child>');
        indent('<widget class="GtkMenuItem" id="menuitem' . $widgetcnt . '">');
        $widgetcnt++;
        dent('<property name="visible">True</property>');
        my $name = string2xml($token[1]);
        dent('<property name="label" translatable="yes">' . $name . '</property>');
        # @todo    ITEM "Open" 1 "Ada.Item" Ctrl+o
        dedent('</widget>');
        dedent('</child>');
    }
}
if ($have_menubar) {
    dedent '</widget>';
    indent '<packing>';
    dent '<property name="padding">0</property>';
    dent '<property name="expand">False</property>';
    dent '<property name="fill">False</property>';
    dedent '</packing>';
    dedent '</child>';
}
close IN;                # end of menubar handling
open(IN, "<$inputfile"); # rewind file for widget handling
indent '<child>';
indent '<widget class="GtkFixed" id="fixed1">';
dent('<property name="visible">True</property>');
dent('<property name="border_width">4</property>');
dent('<property name="width_request">' . $window_width . '</property>');
dent('<property name="height_request">' . $window_height . '</property>');
while (($line = <IN>)) {
    chop $line;
    next if ($line =~ /^\s*$/);
    next if ($line =~ /^\s*(TRUE|FALSE)$/);
    @token = split /\s+/, $line;
    next if ($token[0] =~ /^\d+$/);
    my $keyword = $token[0];
    next if ($keyword =~ /^(WINDOW|MENUBAR|MENU|ITEM|WIDGETS|ENDOF)$/);
    if ($keyword eq 'LABEL') {
        # LABEL lb_id 239 10 350 32 "DisplayText" center black white 
        my $name = string2xml($line);
        my $mcc_justify = $token[$#token - 2];
        my $justify = 'GTK_JUSTIFY_LEFT';
        if ($mcc_justify eq 'center') {
            $justify = 'GTK_JUSTIFY_CENTER';
        } elsif ($mcc_justify eq 'right') {
            $justify = 'GTK_JUSTIFY_RIGHT';
        }
        prologue('GtkLabel');
        dent('<property name="justify">' . $justify . '</property>');
        dent('<property name="label" translatable="yes">' . $name . '</property>');
        epilogue;
    } elsif ($keyword eq 'TEXTENTRY') {
        # TEXTENTRY te_id 275 135 40 20 Ada.Item Ada.Type UNBOUNDED_STRING
        prologue('GtkEntry');
        dent('<property name="editable">True</property>');
        dent('<property name="has_frame">True</property>');
        epilogue;
    } elsif ($keyword eq 'TEXTBOX') {
        # TEXTBOX tb_id 14 42 565 237 FALSE FALSE default white  Ada.Item
        prologue('GtkTextView');
        dent('<property name="can_focus">True</property>');
        dent('<property name="editable">False</property>');
        epilogue;
    } elsif ($keyword eq 'TEXTBUTTON') {
        my $name = string2xml($token[7]);
        # TEXTBUTTON btn_id 320 135 16 21 "related_id" "displaytext"
        prologue('GtkButton');
        dent('<property name="label" translatable="yes">' . $name . '</property>');
        epilogue;
    } elsif ($keyword eq 'PICTUREBUTTON') {
        my $name = string2xml($token[7]);
        # PICTUREBUTTON openButton 27 0 27 27 "File_Menu.Open_Choice"  "open_gif.gif"  "Open"
        prologue('GtkButton');
        dent('<property name="label" translatable="yes">' . $name . '</property>');
        prologue('GtkImage', 'image');
        dent('<property name="pixbuf">' . string2xml($token[7]) . '</property>');
        dedent('</widget>');
        dedent('</child>');
        epilogue;
    } elsif ($keyword eq 'CHECKBUTTON') {
        # CHECKBUTTON vertical 220 35 210 20 "Vertical Scrollbar"
        prologue('GtkCheckButton');
        my $name = string2xml($line);
        dent('<property name="label" translatable="yes">' . $name . '</property>');
        dent('<property name="can_focus">True</property>');
        epilogue;
    } elsif ($keyword eq 'RADIOBUTTON') {
        # RADIOBUTTON horizontal 321 126 100 20 "Horizontal" "Orientation"
        prologue('GtkRadioButton');
        my $name = string2xml($token[6]);
        dent('<property name="label" translatable="yes">' . $name . '</property>');
        dent('<property name="can_focus">True</property>');
        epilogue;
    } elsif ($keyword eq 'DROPDOWN') {
        # DROPDOWN dd_id 223 85 180 22 16 black white Ada.Type Ada.Item
        prologue('GtkComboBox');
        dent('<property name="focus_on_click">True</property>');
        # dent('<property name="enable_arrow_keys">True</property>');
        epilogue;
    } else {
        warn "$keyword is not yet implemented\n";
    }
}
dedent('</widget>');  # end of GtkFixed
dedent('</child>');   # GtkFixed is hard coded as child of GtkVBox
dedent('</widget>');  # end of GtkVBox
dedent '</child>';    # GtkVBox is hard coded as child of GtkWindow
dedent('</widget>');  # end of GtkWindow
dedent('</glade-interface>');
dent('');  # end of output
close IN;

