/*
 * 	Extended stream input functions.
 *
 * 	Copyright (C) 2002, 2003  Dmitry Rutsky	<rutsky@school.ioffe.rssi.ru>
 * 	
 * 	This program is free software; you can redistribute it and/or modify it
 *	under the terms of the GNU General Public License as published
 * 	by the Free Software Foundation; either version 2 of the License, or
 * 	(at your option) any later version.
 *
 * 	This program is distributed in the hope that it will be useful,
 * 	but WITHOUT ANY WARRANTY; without even the implied warranty of
 * 	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * 	See the GNU General Public License for more details.
 *
 * 	You should have received a copy of the GNU General Public License
 * 	along with this program; if not, write to the Free Software Foundation,
 * 	Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "read_line.h"

/* ========	Buffer hadling stuff	======== */

static const int EXP_BASE = 16;	// The base of the exponential expansion of the
				// buffer size.

typedef char BufferElement;	// For future generalisation.  It has to be the
				// one of the basical type for normal
				// operation.

typedef struct
{
   BufferElement *buffer;
   int buffer_size, allocated_size;
}
Buffer;


static void buffer_init (Buffer *b)
{
   memset (b, 0, sizeof (Buffer));
}

//
//	Free the buffer dynamically allocated memory if any.
//	
static void buffer_cleanup (Buffer *b)
{
   if (b -> buffer)
      free (b -> buffer);
}

//
//	Expand buffer allocated size twice, or allocate the first chunk if the
//	buffer itself isn't allocated.
//
//	Return the new buffer size if it was successfully expanded, or `false' 
//	if not.
//	
static int buffer_expand (Buffer *b)
{
   if (!(b -> allocated_size))
   {
      (b -> buffer) = (BufferElement *) malloc 
	      (EXP_BASE * sizeof (BufferElement));

      if (!(b -> buffer))
	 return 0;

      (b -> allocated_size) = EXP_BASE;
   }
   else
   {
      BufferElement *new = (BufferElement *) realloc
	      ((b -> buffer), (b -> allocated_size) * 2);
      if (!new)
         return 0;

      (b -> buffer) = new;
      (b -> allocated_size) *= 2;
   }

   return (b -> allocated_size);
}

//
//	Add the element `e' to the end of the buffer string.
//
//	Return the new string size if the element was successfuly added, or
//	`false' if not.
//	
static int buffer_add (Buffer *b, BufferElement e)
{
   if ((b -> buffer_size) == (b -> allocated_size))
      if (!buffer_expand (b))
	 return 0;

   (b -> buffer) [b -> buffer_size] = e;
   (b -> buffer_size) ++;

   return (b -> buffer_size);
}

//
//	Return the buffer content.
//
static BufferElement *buffer_content (Buffer *b)
{
   return (b -> buffer);
}

/* ========	The input functions	======== */

//
//	Read the string from the stream till one of the `delimiters' character
//	is met and return it excluding the delimiter being dynamically allocated
//	dynamically allocated with the standard `malloc' function.  If there was
//	an error, value NULL is returned and standard `errno' variable holds the
//	error code for the fault.
//
//	NOTE:  this version is probably not the fastest, we rely on stdlib's
//	fgetc functions.
//
char *read_chunk (FILE *stream, const char *delimiters)
{
   Buffer b;
   int c = 1;

   buffer_init (&b);

   while (c)
   {
      c = fgetc (stream);
     
      if (c == EOF)
      {
	 if (b.buffer_size)
	    c = 0;
	 else
	 {
            buffer_cleanup (&b);

	    return NULL;
	 }
      }
      
      if (strchr (delimiters, c))
	 c = 0;
	 
      if (! buffer_add (&b, c))
      {
         buffer_cleanup (&b);

         return NULL;
      }
   }
   
   return buffer_content (&b);
}

inline char *read_line (FILE *stream)
{
   return read_chunk (stream, "\n\0");
}
