/*
 * 	Random Access Machine.
 * 	Emulation stuff.
 *
 * 	Copyright (C) 2002, 2003  Dmitry Rutsky	<rutsky@school.ioffe.rssi.ru>
 * 	
 * 	This program is free software; you can redistribute it and/or modify
 * 	it under the terms of the GNU General Public License as published by
 * 	the Free Software Foundation; either version 2 of the License, or
 * 	(at your option) any later version.
 *
 * 	This program is distributed in the hope that it will be useful,
 * 	but WITHOUT ANY WARRANTY; without even the implied warranty of
 * 	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * 	GNU General Public License for more details.
 *
 * 	You should have received a copy of the GNU General Public License
 * 	along with this program; if not, write to the 
 * 	Free Software Foundation, Inc.,
 * 	59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <stdarg.h>

#include "ram.h"
#include "read_line.h"

/* ========	Prompts control			======== */

static int prompts_enabled = 0;

void ram_enable_IO_prompts ()
{
   prompts_enabled = 1;
}

void ram_disable_IO_prompts ()
{
   prompts_enabled = 0;
}

/* ========	Various emulation stuff		======== */

RAM *ram_new_by_program (RAM_Program *program)
{
   RAM *machine;

   machine = ram_new ();
   (machine -> program) = program;
   (machine -> current_instruction) = 0;

   return machine;
}

// Do something about emulation error
void emulation_error (RAM *machine, const char *error)
{
   err_error ("Instruction %d: %s",
		      (machine -> current_instruction), error);
}

static mpz_t *get_parameter_ptr (RAM *machine, RAM_Instruction *i)
{
   switch (i -> parameter_type)
   {
   case RAM_CONSTANT:
      return &(i -> parameter);
   case RAM_POINTER:
      return ram_get_register ((machine -> memory), &(i -> parameter));
   case RAM_INDIRECT_POINTER:
      return ram_get_register_by_pointer ((machine -> memory),
		      &(i -> parameter));
   default:
      emulation_error (machine, "unknown parameter type");
   }

   return NULL;	// Never reached
}

static inline void io_prompt (RAM *machine, int read)
{
   if (prompts_enabled)
   {
      if (verbosity_is_enough (1))
	 printf ("%s at instruction %d: ", read ? "READ" : "WRITE",
			 (machine -> current_instruction));
      else
	 printf ("%s: ", read ? "READ" : "WRITE"); 
   }
}

/* ========	Time cost evaluation routines	======== */

static inline void add_cost (RAM *machine, unsigned int cost)
{
   mpz_add_ui ((machine -> time_consumed), (machine -> time_consumed), cost);
}

static inline void add_number_cost (RAM *machine, mpz_t *n)
{
   add_cost (machine, mpz_sizeinbase (*n, 2));

   if (mpz_sgn (*n) < 0)
      add_cost (machine, 1);
}

static inline void add_register_0_cost (RAM *machine)
{
   add_number_cost (machine, ram_get_register_0 (machine -> memory));
}

static void add_param_read_cost (RAM *machine, RAM_Instruction *i)
{
   switch (i -> parameter_type)
   {
   case RAM_INDIRECT_POINTER:
      add_number_cost (machine, 
		      ram_get_register_by_pointer ((machine -> memory),
		      &(i -> parameter)));
   case RAM_POINTER:
      add_number_cost (machine, ram_get_register ((machine -> memory),
			      &(i -> parameter)));
   case RAM_CONSTANT:
      add_number_cost (machine, &(i -> parameter));
      break;
   default:
      emulation_error (machine, "unknown parameter type");
   }
}

static void add_param_store_cost (RAM *machine, RAM_Instruction *i)
{
   switch (i -> parameter_type)
   {
   case RAM_INDIRECT_POINTER:
      add_number_cost (machine, ram_get_register ((machine -> memory),
			      &(i -> parameter)));
   case RAM_POINTER:
      add_number_cost (machine, &(i -> parameter));
      break;
   default:
      emulation_error (machine, "unknown or invalid parameter type");
   }

   add_register_0_cost (machine);
}

/* ========	Emulation routines	======== */

// The handler should return `true' if the program is still running.
typedef int (InstructionHandler)(RAM *, RAM_Instruction *);

static const InstructionHandler ram_read, ram_write, ram_load, ram_store,
		ram_add, ram_neg, ram_half, ram_jump, ram_jgtz, ram_halt;

// Order is important!
static const InstructionHandler *instruction [] =
{
   NULL,
   ram_read, ram_write,
   ram_load, ram_store,
   ram_add, ram_neg, ram_half,
   ram_jump, ram_jgtz,
   ram_halt
};

// Return true if machine is still running, false if stopped.
int ram_do_instruction (RAM *machine)
{
   RAM_Instruction *i;

   if (!ram_is_running (machine))
      return 0;

   i = (machine -> program -> instructions) [machine -> current_instruction];

   if ((instruction [i -> instruction]) (machine, i))
   {
      mpz_add_ui ((machine -> instructions_done),
		      (machine -> instructions_done), 1);
      return ram_is_running (machine);
   }

   return 0;
}

inline int ram_is_running (RAM *machine)
{
   if ((machine -> current_instruction) >= (machine -> program -> n))
      return 0;

   return ((machine -> program -> instructions)
	 [machine -> current_instruction] -> instruction) != RAM_HALT;
}

/* ========	Implementation of the instruction set	======== */

static int ram_read (RAM *machine, RAM_Instruction *i)
{
   char *line;
   
   io_prompt (machine, 1);
   
   line = read_line (machine -> input);
   if (! line)
   {
      emulation_error (machine, "couldn't get input string for READ");
      return 0;
   }
   
   if (! ram_evaluate_expression (NULL, line,
			   ram_get_register_0 (machine -> memory)))
   {
      emulation_error (machine, "READ input error");
      return 0;
   }

   free (line);

   (machine -> current_instruction) ++;

   add_register_0_cost (machine);

   return 1;
}

static int ram_write (RAM *machine, RAM_Instruction *i)
{
   io_prompt (machine, 0);
   
   if (! mpz_out_str ((machine -> output), 10,
		*(ram_get_register_0 (machine -> memory))))
   {
      emulation_error (machine, "WRITE output error");
      return 0;
   }
   fprintf ((machine -> output), "\n");

   (machine -> current_instruction) ++;

   add_register_0_cost (machine);

   return 1;
}

static int ram_load (RAM *machine, RAM_Instruction *i)
{
   mpz_t *n;

   n = get_parameter_ptr (machine, i);
   if (!n)
   {
      emulation_error (machine, "LOAD with bad parameter");
      return 0;
   }
      
   mpz_set (*(ram_get_register_0 (machine -> memory)), *n);

   (machine -> current_instruction) ++;

   add_param_read_cost (machine, i);

   return 1;
}

static int ram_store (RAM *machine, RAM_Instruction *i)
{
   mpz_t *n;

   n = get_parameter_ptr (machine, i);
   if (!n)
   {
      emulation_error (machine, "STORE with bad parameter");
      return 0;
   }
   
   mpz_set (*n, *(ram_get_register_0 (machine -> memory)));

   (machine -> current_instruction) ++;

   add_param_store_cost (machine, i);

   return 1;
}

static int ram_add (RAM *machine, RAM_Instruction *i)
{
   mpz_t *n, *reg_0 = ram_get_register_0 (machine -> memory);

   n = get_parameter_ptr (machine, i);
   if (!n)
   {
      emulation_error (machine, "ADD with bad parameter");
      return 0;
   }
      
   mpz_add (*reg_0, *reg_0, *n);

   (machine -> current_instruction) ++;

   add_register_0_cost (machine);

   return 1;
}

static int ram_neg (RAM *machine, RAM_Instruction *i)
{
   mpz_t *reg_0 = ram_get_register_0 (machine -> memory);
   
   mpz_neg (*reg_0, *reg_0);

   (machine -> current_instruction) ++;

   add_cost (machine, 1);

   return 1;
}

static int ram_half (RAM *machine, RAM_Instruction *i)
{
   mpz_t *reg_0 = ram_get_register_0 (machine -> memory);
   
   mpz_tdiv_q_2exp (*reg_0, *reg_0, 1);

   (machine -> current_instruction) ++;

   add_cost (machine, 2);

   return 1;
}

static int ram_jump (RAM *machine, RAM_Instruction *i)
{
   (machine -> current_instruction) = mpz_get_ui (i -> parameter) - 1;

   add_cost (machine, 1);

   return 1;
}

static int ram_jgtz (RAM *machine, RAM_Instruction *i)
{
   if (mpz_sgn (*ram_get_register_0 (machine -> memory)) > 0)
   {
      (machine -> current_instruction) =
	      mpz_get_ui (i -> parameter) - 1;

      add_cost (machine, 3);
   }
   else
   {
      (machine -> current_instruction) ++;

      add_cost (machine, 2);
   }

   return 1;
}

static int ram_halt (RAM *machine, RAM_Instruction *i)
{
   add_cost (machine, 1);

   return 0;
}
