/*
 * 	Random Access Machine.
 * 	Basic functions.
 *
 * 	Copyright (C) 2002, 2003  Dmitry Rutsky	<rutsky@school.ioffe.rssi.ru>
 * 	
 * 	This program is free software; you can redistribute it and/or modify
 * 	it under the terms of the GNU General Public License as published by
 * 	the Free Software Foundation; either version 2 of the License, or
 * 	(at your option) any later version.
 *
 * 	This program is distributed in the hope that it will be useful,
 * 	but WITHOUT ANY WARRANTY; without even the implied warranty of
 * 	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * 	GNU General Public License for more details.
 *
 * 	You should have received a copy of the GNU General Public License
 * 	along with this program; if not, write to the 
 * 	Free Software Foundation, Inc.,
 * 	59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <gmp.h>

#include "ram.h"

/* ========	Constructors and destructors	======== */

RAM_Instruction *ram_instruction_new ()
{
   RAM_Instruction *ri;

   ri = (RAM_Instruction *) malloc (sizeof (RAM_Instruction));
   if (! ri)
      err_fatal_perror ("malloc",
		      "RAM_Instruction memory allocation failure");

   (ri -> parameter_type) = RAM_NO_PARAMETER;

   return ri;
}

static void ram_instruction_clear (RAM_Instruction *ri)
{
   if ((ri -> parameter_type) != RAM_NO_PARAMETER)
      mpz_clear (ri -> parameter);
}

void ram_instruction_delete (RAM_Instruction *ri)
{
   ram_instruction_clear (ri);
   free (ri);
}

RAM_Program *ram_program_new ()
{
   RAM_Program *rp;

   rp = (RAM_Program *) calloc (1, sizeof (RAM_Program));
   if (!rp)
      err_fatal_perror ("calloc", "RAM_Program memory allocation failure");

   return rp;
}

void ram_program_clear (RAM_Program *rp)
{
   if (rp -> instructions)
   {
      int i;
      
      for (i = 0; i < (rp -> n); i ++)
	 if ((rp -> instructions) [i])
	    ram_instruction_clear ((rp -> instructions) [i]);
   }
}

void ram_program_delete (RAM_Program *rp)
{
   ram_program_clear (rp);
   free (rp);
}

RAM *ram_new ()
{
   RAM *rm;

   rm = (RAM *) calloc (1, sizeof (RAM));
   if (!rm)
      err_fatal_perror ("calloc", "RAM memory allocation failure");

   (rm -> program) = NULL;

   (rm -> input) = stdin;
   (rm -> output) = stdout;

   (rm -> memory) = ram_memory_new ();

   mpz_init (rm -> instructions_done);
   mpz_init (rm -> time_consumed);

   return rm;
}

void ram_clear (RAM *rm)
{
   if (!rm)
      return;
	
   if (rm -> program)
   {
      ram_program_delete (rm -> program);
      (rm -> program) = NULL;
   }

   if (rm -> memory)
      ram_memory_delete (rm -> memory);

   (rm -> memory) = NULL;

   mpz_set_ui ((rm -> instructions_done), 0);
   mpz_set_ui ((rm -> time_consumed), 0);
}

void ram_delete (RAM *rm)
{
   ram_clear (rm);

   mpz_clear (rm -> instructions_done);
   mpz_clear (rm -> time_consumed);
   
   free (rm);
}

/* =======	Some basic operation	======== */

// Reinitialize everything except program itself
void ram_reset (RAM *rm)
{
   ram_memory_reset (rm -> memory);
   
   mpz_set_ui ((rm -> instructions_done), 0);
   mpz_set_ui ((rm -> time_consumed), 0);

   rewind (rm -> input);
}
