/*
 * 	Error processing module.
 *
 * 	Copyright (C) 2003  Dmitry Rutsky	<rutsky@school.ioffe.rssi.ru>
 * 	
 * 	This program is free software; you can redistribute it and/or modify
 * 	it under the terms of the GNU General Public License as published by
 * 	the Free Software Foundation; either version 2 of the License, or
 * 	(at your option) any later version.
 *
 * 	This program is distributed in the hope that it will be useful,
 * 	but WITHOUT ANY WARRANTY; without even the implied warranty of
 * 	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * 	GNU General Public License for more details.
 *
 * 	You should have received a copy of the GNU General Public License
 * 	along with this program; if not, write to the 
 * 	Free Software Foundation, Inc.,
 * 	59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <ctype.h>
#include <limits.h>

#include "error.h"

const ErrorLevel ERROR_FATAL = -VERBOSITY_FATAL_ERRORS;
const ErrorLevel ERROR_ERROR = -VERBOSITY_ERRORS;
const ErrorLevel ERROR_IMPORTANT_WARNING = -VERBOSITY_IMPORTANT_WARNINGS;
const ErrorLevel ERROR_WARNING = -VERBOSITY_WARNINGS;
const ErrorLevel ERROR_DEBUG = -VERBOSITY_DEBUG;
const ErrorLevel ERROR_EXTENSIVE_DEBUG = -VERBOSITY_EXTENSIVE_DEBUG;

const ErrorLevel ERROR_NONE = INT_MAX;

const ErrorLevel ERROR_PROGRAMMING = -1234567;

const ErrorLevel ERROR_MESSAGE = -7654321;

ErrorLevel err_highest_met_severity = INT_MIN;

ErrorLevel err_update_highest_met_severity (const ErrorLevel severity)
{
   ErrorLevel old = err_highest_met_severity;

   if (severity != ERROR_PROGRAMMING && (old < severity))
      err_highest_met_severity = severity;

   return old;
}

static ErrorVSNPrintFHandler vsnprintf_handler = vsnprintf;
static ErrorSNPrintFHandler snprintf_handler = snprintf;

static const char *err_message_prefix (const ErrorLevel severity)
{
   if (severity == ERROR_FATAL)
      return "FATAL ERROR:  ";
   else if (severity == ERROR_ERROR)
      return "ERROR:  ";
   else if (severity == ERROR_IMPORTANT_WARNING)
      return "WARNING:  ";
   else if (severity == ERROR_WARNING)
      return "Warning:  ";
   else if (severity == ERROR_PROGRAMMING)
      return "Programming Error (FIXME):  ";
   else
      return "";
}

static int output_width = 80;

const int output_width_min = 10;

int err_set_output_width (int width)
{
   int old = output_width;

   if (width < output_width_min)
      err_programming (
	"tried to set message output width to value less than %d (%d)",
	output_width_min, output_width);

   output_width = width;

   return old;
}

static int tab_width = 8;

// Try to format string using `size' characters buffer and print it to stderr,
// return true on success.  `size' have to be at least `prefix' string length
// plus 1.
static int try_to_report (int size, FILE *output, const char *prefix,
		const char *format, va_list args)
{
   char *buf = alloca (size * sizeof (char));
   int prefix_length = strlen (prefix);

   if (!buf)
   {
      fprintf (stderr,
	"ERROR:  couldn't allocate memory (%d byte%s) while formatting error\n"
	"message string", size, (size > 1) ? "s" : "");
      abort ();
   }

   strcpy (buf, prefix);

   if (vsnprintf_handler (buf + prefix_length, size, format, args) >
		   (size - prefix_length - 1))
      return 0;

   {
      char *line = buf;

      if (!*line)
	 line = NULL;

      while (line)
      {
	 char *last_space = NULL, *s = line;
	 int count = 0, end = 0, wrapped = 0;

	 do
	 {
	    if (isspace (*s))
	       last_space = s;

	    if (*s == '\t')
	       count += tab_width;
	    else
	       count ++;

	    s ++;
	 }
	 while (*s && *s != '\n' && count < output_width);

	 if (count >= output_width && *s && *s != '\n' &&
			 !last_space) // Long word --- output as is
	 {
	    while (!isspace (*s) && *s != '\n' && *s)
	       s ++;

	    last_space = s;

	    count ++;
	    wrapped = 1;
	 }

	 if (count <= output_width && (*s == '\n' || !*s))
            last_space = s;
	 else
	    wrapped = 1;

	 if (!last_space)
	    last_space = s;

	 if (!*last_space)
	    end = 1;

	 *last_space = '\0';
	 fprintf (output, "%s\n", line);

	 line = end ? NULL : last_space + 1;
	 if (wrapped && line)
	    while (*line && isspace (*line))
	       line ++;

	 if (line)
	    if (!*line)
	       line = NULL;
      }
   }

   return 1;
}

static void default_error_handler (const ErrorLevel severity, const char *error,
								va_list args)
{
   fflush (stdout);

   if (verbosity_is_enough (-severity) || severity == ERROR_PROGRAMMING ||
		   		severity == ERROR_MESSAGE)
   {
      const char *prefix = err_message_prefix (severity);
      int size = strlen (prefix) + 2 * strlen (error); 	// Quite appropriate
      							// evaluation
      while (!try_to_report (size,
			  (severity == ERROR_MESSAGE) ? stdout : stderr,
			  prefix, error, args))
	 size *= 2;
   }

   if (severity >= ERROR_FATAL)
      exit (severity);

   err_update_highest_met_severity (severity);
}

ErrorHandler err_handler = default_error_handler;

ErrorHandler err_install_handler (ErrorHandler eh)
{
   ErrorHandler old = err_handler;

   err_handler = eh;

   return old;
}

inline void err_report (const ErrorLevel severity, const char *error, ...)
{
   va_list args;

   va_start (args, error);

   err_handler (severity, error, args);

   va_end (args);
}

static inline void err_report_perror_vargs (const ErrorLevel severity,
			const char *caller, const char *error, va_list args)
{
   char *e = strerror (errno);
   char *s = (char *) alloca (sizeof (char) * 
		   (strlen (error) + strlen (caller) + strlen (e) + 5));
   if (!s)
      err_report (ERROR_FATAL,
		      "err_report_perror_vargs:  cannot allocate memory for"
		      " the error message\nalloca: %s", strerror (errno));

   sprintf (s, "%s\n%s:  %s", error, caller, e);
   err_handler (severity, s, args);
}

inline void err_report_perror (const ErrorLevel severity, const char *caller, 
					const char *error, ...)
{
   va_list args;

   va_start (args, error);

   err_report_perror_vargs (severity, caller, error, args);

   va_end (args);
}

inline void err_error (const char *error, ...)
{
   va_list args;

   va_start (args, error);

   err_handler (ERROR_ERROR, error, args);

   va_end (args);
}

inline void err_programming (const char *error, ...)
{
   va_list args;

   va_start (args, error);

   err_handler (ERROR_PROGRAMMING, error, args);

   va_end (args);
}

inline void err_message (const char *error, ...)
{
   va_list args;

   va_start (args, error);

   err_handler (ERROR_MESSAGE, error, args);

   va_end (args);
}

inline void err_fatal (const char *error, ...)
{
   va_list args;

   va_start (args, error);

   err_handler (ERROR_FATAL, error, args);

   va_end (args);
}

inline void err_important_warning (const char *error, ...)
{
   va_list args;

   va_start (args, error);

   err_handler (ERROR_IMPORTANT_WARNING, error, args);

   va_end (args);
}

inline void err_warning (const char *error, ...)
{
   va_list args;

   va_start (args, error);

   err_handler (ERROR_WARNING, error, args);

   va_end (args);
}

inline void err_debug (const char *error, ...)
{
   va_list args;

   va_start (args, error);

   err_handler (ERROR_DEBUG, error, args);

   va_end (args);
}

inline void err_extensive_debug (const char *error, ...)
{
   va_list args;

   va_start (args, error);

   err_handler (ERROR_EXTENSIVE_DEBUG, error, args);

   va_end (args);
}


inline void err_error_perror (const char *caller, const char *error, ...)
{
   va_list args;

   va_start (args, error);

   err_report_perror_vargs (ERROR_ERROR, caller, error, args);

   va_end (args);
}

inline void err_warning_perror (const char *caller, const char *error, ...)
{
   va_list args;

   va_start (args, error);

   err_report_perror_vargs (ERROR_WARNING, caller, error, args);

   va_end (args);
}

inline void err_debug_perror (const char *caller, const char *error, ...)
{
   va_list args;

   va_start (args, error);

   err_report_perror_vargs (ERROR_DEBUG, caller, error, args);

   va_end (args);
}

inline void err_fatal_perror (const char *caller, const char *error, ...)
{
   va_list args;

   va_start (args, error);

   err_report_perror_vargs (ERROR_FATAL, caller, error, args);

   va_end (args);
}

ErrorSNPrintFHandler err_use_snprintf_handler (ErrorSNPrintFHandler new_handler)
{
   ErrorSNPrintFHandler old_handler = snprintf_handler;

   snprintf_handler = new_handler;

   return old_handler;
}

ErrorVSNPrintFHandler err_use_vsnprintf_handler
					(ErrorVSNPrintFHandler new_handler)
{
   ErrorVSNPrintFHandler old_handler = vsnprintf_handler;

   vsnprintf_handler = new_handler;

   return old_handler;
}
