/*
 * 	Random Access Machine.
 * 	Configuration file stuff.
 *
 * 	Copyright (C) 2003  Dmitry Rutsky	<rutsky@school.ioffe.rssi.ru>
 * 	
 * 	This program is free software; you can redistribute it and/or modify
 * 	it under the terms of the GNU General Public License as published by
 * 	the Free Software Foundation; either version 2 of the License, or
 * 	(at your option) any later version.
 *
 * 	This program is distributed in the hope that it will be useful,
 * 	but WITHOUT ANY WARRANTY; without even the implied warranty of
 * 	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * 	GNU General Public License for more details.
 *
 * 	You should have received a copy of the GNU General Public License
 * 	along with this program; if not, write to the 
 * 	Free Software Foundation, Inc.,
 * 	59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "configuration.h"
#include "error.h"

// This stuff is located in the lexical analyzer.
void yyrestart (FILE *);
char *conf_lexan ();


/* ========	Constructors and destructors	======== */

Conf_Option *conf_option_new ()
{
   Conf_Option *o = (Conf_Option *) calloc (1, sizeof (Conf_Option));
   if (!o)
      err_fatal_perror ("calloc",
	"couldn't allocate memory for `Conf_Option' structure");

   return o;
}

static inline Conf_Option *option_new_with (char *name, char *value)
{
   Conf_Option *o = conf_option_new ();
   
   (o -> name) = name;
   (o -> value) = value;

   return o;
}

void conf_option_delete (Conf_Option *o)
{
   if (o -> name)
      free (o -> name);

   if (o -> value)
      free (o -> value);

   free (o);
}

static inline void conf_option_delete_list (Conf_Option *c)
{
   while (c)
   {
      Conf_Option *next = (c -> next);
      conf_option_delete (c);
      c = next;
   }
}

void conf_delete (Conf_Option *configuration)
{
   conf_option_delete_list (configuration);
}

/* ========	Search and insertion	======== */

// Returns pointer to option _after_ which the record should be inserted.
// If it returns NULL, the record should be first in the list.
static inline Conf_Option *find_position (Conf_Option *configuration,
			char *name)
{
   while (configuration)
   {
      if (strcmp ((configuration -> name), name) > 0)
	 return (configuration -> previous);

      if (!(configuration -> next))
	 return configuration;

      configuration = (configuration -> next);
   }

   return NULL;
}

void conf_insert (Conf_Option **configuration, char *name, char *value)
{
   Conf_Option *o = find_position (*configuration, name);

   if (o)
      if (! strcmp ((o -> name), name))
      {
         err_important_warning ("option `%s' already presents "
		"in the configuration, replacing old value.");

         free (o -> name);
         free (o -> value);

	 (o -> name) = name;
	 (o -> value) = value;

	 return;
      }
   
   if (!o)
   {
      o = option_new_with (name, value);

      (o -> next) = *configuration;
      *configuration = o;
   }
   else
   {
      Conf_Option *p = option_new_with (name, value);

      (p -> next) = (o -> next);
      (p -> previous) = o;
      (o -> next) = p;
      if (o -> next)
         (o -> next -> previous) = p;
   }
}

/* ========	Configuration IO	======== */

static char *build_string (const char *s, int unquote)
{
   char *c = (char *) malloc ((strlen (s) + 1) * sizeof (char)), *p;
   const char *sp = s;

   p = c;

   if (!c)
      err_fatal_perror ("malloc",
	"couldn't allocate memory for option string (%d bytes)", strlen (s));

   if (unquote)
   {
      if (*s != '"')
	 err_programming (
		"conf_build_string_unquoted called on unquoted string `%s'", s);
      else
         s ++;
   }
   
   while (*s)
   {
      if (*s != '\\')
	 *p = *s;
      else
      {
         s ++;
	 switch (*s)
	 {
	 case 'n':
	    *p = '\n';
	    break;
	 case 't':
	 case '\t':
	    *p = '\t';
	    break;
	 case '\\':
	    *p = '\\';
	    break;
	 case ' ':
	    *p = ' ';
	    break;
	 case '"':
	    *p = '"';
	    break;
	 default:
	    err_programming ("conf_build_string:  unsupported \\-symbol `%c'",
			    *s);
	 }
      }

      p ++;
      s ++;
   }

   if (unquote)
   {
      if (p == c)
	 err_programming ("conf_build_string:  single quotation in option");
      else if (*(p - 1) != '"')
	 err_programming ("conf_build_string:  bad quotation: `%s'", sp);
      else
	 p --;
   }
   
   *p = '\0';

   {
      char *n = realloc (c, sizeof (char) * (p - c + 1));
      if (!n)
      {
	 err_error_perror ("realloc", "couldn't shrink option string");
	 return p;
      }
      else
	 return n;
   }
}

// Return newly allocated string converted from `s'.  `\'-symbols are
// translated.
char *conf_build_string (const char *s)
{
   return build_string (s, 0);
}

// Same as above, but delete quotation marks around `s'.
char *conf_build_unquoted_string (const char *s)
{
   return build_string (s, 1);
}

// Read configuration from file `f'.  If error occured, this function returns
// options that were read before the error.
Conf_Option *conf_read (FILE *f)
{
   char *option_str, *value_str;
   Conf_Option *configuration = NULL;
   
   yyrestart (f);

   while ((option_str = conf_lexan ()))
   {
      value_str = conf_lexan ();
      if (!value_str)
      {
	 err_error ("No value found for option `%s'.", option_str);

	 free (value_str);

	 return configuration;
      }

      conf_insert (&configuration, option_str, value_str);
   }
   
   return configuration;
}
