# functions.pl
# Copyright (C) 2007,2008 Robert Alex Marder
# This file is part of RAM-CMS
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 3
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
# or see <http://www.gnu.org/licenses/>.
#
# This file contains all the functions used by RAM-CMS.
# The RAM-CMS function library.
#
use strict;

package RAMCMS;

use File::Basename;

# Purpose: determine when the requested page was last updated, and format as string.
# Santax: str $updated_show = ramcms_last_update_time ( str $page )
# Where: 
# $updated_show is the last update time as a date string, or 'unknown'.
# $page is the page requested.
#
sub ramcms_last_update_time
{
	my $page=shift;
	# get the last updated time from the file of the page we want.
	# we check the file creation/change time and the file modified time and 
	# take the newer of the two. This way we have foolproof last update time.
	# the latest update time of the requested file gets put in [[updated]].
	
	# get the file modification time.
	#$filemtime=filemtime($page);
	my $filemtime=(stat($page))[9];
	# get the file creation time.
	#$filectime=filectime($page);
	my $filectime=(stat($page))[10];
	
	my $updated;
	# determine if file creation time or file modified time is newer (higher)
	if($filectime > $filemtime)
	{
		# file creation time is newer.
		$updated=$filectime;
	}
	else
	{
		# file modification time is newer.
		$updated=$filemtime;
	}
	
	my $updated_show='';
	
	# if for some reason we failed to figure out when we were last 
	# updated, fix $updated_show to be "unknown" as opposed 
	# to nothing.
	# if we did figure out what the last update was, convert 
	# $updated_show from a UNIX timestamp into something humans 
	# prefer to read.
	if(!defined($updated) or $updated eq '')
	{
		$updated_show='unknown';
	}
	else
	{
		# format example: 28 Sep 2007
		# TODO: make the date format configurable in config.php
		#$updated_show=date('d M Y',$updated);
		my($tmp,$mday,$mon,$year);
		($tmp,$tmp,$tmp,$mday,$mon,$year)=localtime($updated);
		$year+=1900;
		my @months=qw(Jan Feb Mar Apr May Jun Jul Aug Sep Oct Nov Dec);
		$updated_show="$mday $months[$mon] $year";
	}
	
	# return the last update time of the file, to be put into [[updated]].
	return $updated_show;
}

# Purpose: to output http headers based on a set of criteria
# Santax: null ramcms_send_http_headers ( bool $nocache )
# Where: 
# $nocache is if we should send anti-cache headers.
#
sub ramcms_send_http_headers
{
	my $nocache=shift; #arg 1
	# send some http headers.
	
	print "Content-Type: text/html\r\n";
	
	# please leave this here.
	print "X-CMS: RAM-CMS\r\n";
	
	if($nocache==1)
	{
		print "Cache-Control: no-store, no-cache, must-revalidate, post-check=0, pre-check=0\r\n";
		print "Expires: Mon, 26 Jul 1997 05:00:00 GMT\r\n";
		print "Pragma: no-cache\r\n";
	}
	else
	{
		# cache content for 1 hour.
		print "Cache-Control: max-age=3600, proxy-revalidate\r\n";
		# only browsers may cache this.
		print "Pragma: private\r\n";
	}
	print "\r\n";
}

# Purpose: to strip unwanted whitespace from $body so as to reduce bandwidth.
# Santax: str $body = ramcms_remove_body_whitespace ( str $body )
# Where: 
# $body is the $body of the requested page.
#
sub ramcms_remove_body_whitespace
{
	my $body=shift;
	# using strstr twice is faster than using one regex (I think).
	
	# if <PRE> or <pre> is found in $body then preserve whitespace.
	# otherwise remove all tabs and newlines.
	if(index($$body,'<PRE')==-1 && index($$body,'<pre')==-1 && index($$body,'<TEXTAREA')==-1 && index($$body,'<textarea')==-1)
	{
		# strip all newlines and tabs.
		$$body=str_replace("\t",'',str_replace("\n",'',str_replace("\r",'',$$body)));
	}
	#return $body;
}

# Purpose: to replace [[string]] items with specific $string equivalents.
# Santax: 
# Where: 
#
sub ramcms_page_template_parser
{
	my $page_template=shift;
	my $title=shift;
	my $site_name=shift;
	my $header=shift;
	my $updated_show=shift;
	my $body=shift;
	my $copyright=shift;
	my $curdir=shift;
	
	my $protocol='http://';
	if(defined($ENV{'HTTPS'}) and lc($ENV{'HTTPS'}) eq 'on')
	{
		$protocol='https://';
	}
	
	# parse the $page_template information.
	$$page_template=str_replace('[[baseurl]]',$protocol.$ENV{'SERVER_NAME'}.dirname($ENV{'SCRIPT_NAME'}).'/',$$page_template);
	# parse the $page_template information.
	#$$page_template=str_replace('[[body]]',$body,$$page_template);
	$$page_template =~ s/\[\[body\]\]/$body/g;
	#$$page_template=str_replace('[[copyright]]',$copyright,$$page_template);
	$$page_template =~ s/\[\[copyright\]\]/$copyright/g;
	#$$page_template=str_replace('[[header]]',$header,$$page_template);
	$$page_template =~ s/\[\[header\]\]/$header/g;
	
	$$page_template=str_replace('[[nav]]',file_get_contents($curdir.'nav.txt'),$$page_template);
	
	#$$page_template=str_replace('[[site]]',$site_name,$$page_template);
	$$page_template =~ s/\[\[site\]\]/$site_name/g;
	#$$page_template=str_replace('[[title]]',$title,$$page_template);
	$$page_template =~ s/\[\[title\]\]/$title/g;
	#$$page_template=str_replace('[[updated]]',$updated_show,$$page_template);
	$$page_template =~ s/\[\[updated\]\]/$updated_show/g;
	#$$page_template=str_replace('[[ver]]','0.2.0',$$page_template);
	$$page_template =~ s/\[\[ver\]\]/0.2.5/g;
	
	#return $page_template;
}

# Purpose: to parse BBCode tags in $body.
# Santax: str $body = ramcms_bbcode_parser ( str $body )
# Where: 
# $body is the $body of the requested page.
#
sub ramcms_bbcode_parser
{
	my $body=shift; #arg 1
	# TODO: make a better BBCode parser.
	
	# TODO: convert str_replace into perl regex's for speed gains/ram reduction.
	# passing a big variable like $body back and forth is very inefficient.
	# (it's still faster than PHP5 on mod_perl though :)
	# bold
	$$body=str_replace('[b]','<b>',$$body);
	$$body=str_replace('[/b]','</b>',$$body);
	
	# italics
	$$body=str_replace('[i]','<i>',$$body);
	$$body=str_replace('[/i]','</i>',$$body);
	
	# underline
	$$body=str_replace('[u]','<u>',$$body);
	$$body=str_replace('[/u]','</u>',$$body);
	
	# strikethrough
	$$body=str_replace('[s]','<strike>',$$body);
	$$body=str_replace('[/s]','</strike>',$$body);
	
	# teletype
	$$body=str_replace('[tt]','<tt>',$$body);
	$$body=str_replace('[/tt]','</tt>',$$body);
	
	# preformatted text
	$$body=str_replace('[pre]','<pre>',$$body);
	$$body=str_replace('[/pre]','</pre>',$$body);
	
	# subscript
	$$body=str_replace('[sub]','<sub>',$$body);
	$$body=str_replace('[/sub]','</sub>',$$body);
	
	# superscript
	$$body=str_replace('[sup]','<sup>',$$body);
	$$body=str_replace('[/sup]','</sup>',$$body);
	
	# big
	$$body=str_replace('[big]','<big>',$$body);
	$$body=str_replace('[/big]','</big>',$$body);
	
	# small
	$$body=str_replace('[small]','<small>',$$body);
	$$body=str_replace('[/small]','</small>',$$body);
	
	# horizontal rule
	$$body=str_replace('[hr]','<hr>',$$body);
	
	# center
	$$body=str_replace('[center]','<center>',$$body);
	$$body=str_replace('[/center]','</center>',$$body);
	# paragraphs
	$$body=str_replace('[p]','<p>',$$body);
	$$body=str_replace('[/p]','</p>',$$body);
	
	# line breaks
	$$body=str_replace('[br]','<br>',$$body);
	
	# list items
	$$body=str_replace('[li]','<li>',$$body);
	$$body=str_replace('[/li]','</li>',$$body);
	
	# unordered lists
	$$body=str_replace('[ul]','<ul>',$$body);
	$$body=str_replace('[/ul]','</ul>',$$body);
	
	# ordered lists
	$$body=str_replace('[ol]','<ol>',$$body);
	$$body=str_replace('[/ol]','</ol>',$$body);
	
	# headings
	$$body =~ s/\[h([1-6])\](.+?)\[\/h([1-6])\]/<h${1}>${2}<\/h${1}>/gs;
	
	### Complex BBCode ###
	
	# handles [link]page-name[/link]
	$$body =~ s/\[link\](.+?)\[\/link\]/<a href=".\/?page=${1}">${1}<\/a>/gs;
	
	# handles [link=page-name]Click Blah[/link]
	$$body =~ s/\[link=(.+?)\](.+?)\[\/link\]/<a href=".\/?page=${1}">${2}<\/a>/gs;
	
	# handles [url]http://example.com/blah..[/url]
	$$body =~ s/\[url\](.+?)\[\/url\]/<a href="${1}">${1}<\/a>/gs;
	
	# handles [url=http://example.com/blah..]Click Blah[/url]
	$$body =~ s/\[url=(.+?)\](.+?)\[\/url\]/<a href="${1}">${2}<\/a>/gs;
	
	# handles [color=colorname]text here[/color]
	$$body =~ s/\[color=(.+?)\](.+?)\[\/color]/<font color="${1}"\>${2}<\/font>/gs;
	
	# handles [img=imagefile]
	$$body =~ s/\[img=(.+?)\]/<img alt="" src="${1}">/gs;
	
	# handles [!-- this is a code comment --]
	$$body =~ s/\[!--(.+?)--]//gs;
	
	#return $body;
}

#################################################################################

# file_get_contents() for Perl.
sub file_get_contents
{
	my $filename=shift; #arg 1
	open(FH,$filename) || exit("Error Reading File: $filename");
	undef($filename);
	my @data=<FH>;
	close(FH);
	my $contents='';
	my $line='';
	foreach $line (@data)
	{
		$contents=$contents.$line;
	}
	return $contents;
}

# str_replace() for Perl.
sub str_replace
{
	my $search=shift; #arg  1
	my $replace=shift; #arg 2
	my $subject=shift; #arg 3
	if(!defined($subject)) { return -1; }
	my $count=shift;
	if(!defined($count)) { $count=-1; }
	
	my($i,$pos)=(0,0);
	while((my $idx=index($subject,$search,$pos))!=-1)
	{
		substr($subject,$idx,length($search))=$replace;
		$pos=$idx+length($replace);
		if($count>0 && ++$i>=$count) { last; }
	}
	return $subject;
}
1;
__END__
