/***************************************************************************/
/*   Copyright (C) 2006 by Uwe Mayer                                       */
/*   merkosh@hadiko.de                                                     */
/*                                                                         */
/*   This is free software; you can redistribute it and/or modify it       */
/*   under the terms of the GNU Lesser General Public License as           */
/*   published by the Free Software Foundation; either version 2.1 of      */
/*   the License, or (at your option) any later version.                   */
/*                                                                         */
/*   This software is distributed in the hope that it will be useful,      */
/*   but WITHOUT ANY WARRANTY; without even the implied warranty of        */
/*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU      */
/*   Lesser General Public License for more details.                       */
/*                                                                         */
/*   You should have received a copy of the GNU Lesser General Public      */
/*   License along with this software; if not, write to the                */
/*   Free Software Foundation, Inc.,                                       */
/*   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.         */
/***************************************************************************/

%module(docstring="") sndfile


%feature("autodoc", "1");



%header %{
#include "pysndfile.h"

#include <sndfile.h>
#include <Python.h>
#include <Numeric/arrayobject.h>
%}

%init {
  // Python Numeric must initialise callbacks
  import_array();
}

%include "sftypes.i"


//-- format constants ----------------------------------------------------------
enum {   
  /* Major formats. */

  SF_FORMAT_WAV,      /* Microsoft WAV format (little endian). */
  SF_FORMAT_AIFF,     /* Apple/SGI AIFF format (big endian). */
  SF_FORMAT_AU,       /* Sun/NeXT AU format (big endian). */
  SF_FORMAT_RAW,      /* RAW PCM data. */
  SF_FORMAT_PAF,      /* Ensoniq PARIS file format. */
  SF_FORMAT_SVX,      /* Amiga IFF / SVX8 / SV16 format. */
  SF_FORMAT_NIST,     /* Sphere NIST format. */
  SF_FORMAT_VOC,      /* VOC files. */
  SF_FORMAT_IRCAM,    /* Berkeley/IRCAM/CARL */
  SF_FORMAT_W64,      /* Sonic Foundry's 64 bit RIFF/WAV */
  SF_FORMAT_MAT4,     /* Matlab (tm) V4.2 / GNU Octave 2.0 */
  SF_FORMAT_MAT5,     /* Matlab (tm) V5.0 / GNU Octave 2.1 */
  SF_FORMAT_PVF,      /* Portable Voice Format */
  SF_FORMAT_XI,       /* Fasttracker 2 Extended Instrument */
  SF_FORMAT_HTK,      /* HMM Tool Kit format */
  SF_FORMAT_SDS,      /* Midi Sample Dump Standard */
  SF_FORMAT_AVR,      /* Audio Visual Research */
  SF_FORMAT_WAVEX,    /* MS WAVE with WAVEFORMATEX */
  SF_FORMAT_SD2,      /* Sound Designer 2 */
  SF_FORMAT_FLAC,     /* FLAC lossless file format */
  SF_FORMAT_CAF,      /* Core Audio File format */

  /* Subtypes from here on. */

  SF_FORMAT_PCM_S8,   /* Signed 8 bit data */
  SF_FORMAT_PCM_16,   /* Signed 16 bit data */
  SF_FORMAT_PCM_24,   /* Signed 24 bit data */
  SF_FORMAT_PCM_32,   /* Signed 32 bit data */

  SF_FORMAT_PCM_U8,   /* Unsigned 8 bit data (WAV and RAW only) */

  SF_FORMAT_FLOAT,    /* 32 bit float data */
  SF_FORMAT_DOUBLE,   /* 64 bit float data */

  SF_FORMAT_ULAW,     /* U-Law encoded. */
  SF_FORMAT_ALAW,     /* A-Law encoded. */
  SF_FORMAT_IMA_ADPCM,/* IMA ADPCM. */
  SF_FORMAT_MS_ADPCM, /* Microsoft ADPCM. */

  SF_FORMAT_GSM610,   /* GSM 6.10 encoding. */
  SF_FORMAT_VOX_ADPCM,/* Oki Dialogic ADPCM encoding. */

  SF_FORMAT_G721_32,  /* 32kbs G721 ADPCM encoding. */
  SF_FORMAT_G723_24,  /* 24kbs G723 ADPCM encoding. */
  SF_FORMAT_G723_40,  /* 40kbs G723 ADPCM encoding. */

  SF_FORMAT_DWVW_12,  /* 12 bit Delta Width Variable Word encoding. */
  SF_FORMAT_DWVW_16,  /* 16 bit Delta Width Variable Word encoding. */
  SF_FORMAT_DWVW_24,  /* 24 bit Delta Width Variable Word encoding. */
  SF_FORMAT_DWVW_N,   /* N bit Delta Width Variable Word encoding. */

  SF_FORMAT_DPCM_8,   /* 8 bit differential PCM (XI only) */
  SF_FORMAT_DPCM_16,  /* 16 bit differential PCM (XI only) */

  /* Endian-ness options. */

  SF_ENDIAN_FILE,     /* Default file endian-ness. */
  SF_ENDIAN_LITTLE,   /* Force little endian-ness. */
  SF_ENDIAN_BIG,      /* Force big endian-ness. */
  SF_ENDIAN_CPU,      /* Force CPU endian-ness. */

  SF_FORMAT_SUBMASK,
  SF_FORMAT_TYPEMASK,
  SF_FORMAT_ENDMASK
};


//-- open modes ----------------------------------------------------------------
enum {
  SFM_READ,			// open for reading
  SFM_WRITE,			// open for writing
  SFM_RDWR  			// open for read+write
};


// enum {
//   SF_FALSE,
//   SF_TRUE,
// };


//-- seek `whence` constants ---------------------------------------------------
enum {
  SEEK_SET,
  SEEK_CUR,
  SEEK_END
};


//-- error codes ---------------------------------------------------------------
enum {   
  SF_ERR_NO_ERROR,
  SF_ERR_UNRECOGNISED_FORMAT,
  SF_ERR_SYSTEM,
  SF_ERR_MALFORMED_FILE,
  SF_ERR_UNSUPPORTED_ENCODING
};


//-- types of string data in audio files ---------------------------------------
enum {   
  SF_STR_TITLE,
  SF_STR_COPYRIGHT,
  SF_STR_SOFTWARE,
  SF_STR_ARTIST,
  SF_STR_COMMENT,
  SF_STR_DATE
};


//-- sf_command types ----------------------------------------------------------
enum {	
  SFC_GET_LIB_VERSION,
  SFC_GET_LOG_INFO,
  SFC_GET_NORM_DOUBLE,
  SFC_GET_NORM_FLOAT,
  SFC_SET_NORM_DOUBLE,
  SFC_SET_NORM_FLOAT,
  SFC_SET_SCALE_FLOAT_INT_READ,
    
  SFC_GET_SIMPLE_FORMAT_COUNT,
  SFC_GET_SIMPLE_FORMAT,

  SFC_GET_FORMAT_INFO,

  SFC_GET_FORMAT_MAJOR_COUNT,
  SFC_GET_FORMAT_MAJOR,
  SFC_GET_FORMAT_SUBTYPE_COUNT,
  SFC_GET_FORMAT_SUBTYPE,

  SFC_CALC_SIGNAL_MAX,
  SFC_CALC_NORM_SIGNAL_MAX,
  SFC_CALC_MAX_ALL_CHANNELS,
  SFC_CALC_NORM_MAX_ALL_CHANNELS,

  SFC_SET_ADD_PEAK_CHUNK,

  SFC_UPDATE_HEADER_NOW,
  SFC_SET_UPDATE_HEADER_AUTO,

  SFC_FILE_TRUNCATE,

  SFC_SET_RAW_START_OFFSET,

  SFC_SET_DITHER_ON_WRITE,
  SFC_SET_DITHER_ON_READ,

  SFC_GET_DITHER_INFO_COUNT,
  SFC_GET_DITHER_INFO,

  SFC_GET_EMBED_FILE_INFO,

  SFC_SET_CLIPPING,
  SFC_GET_CLIPPING,

  SFC_GET_INSTRUMENT,
  SFC_SET_INSTRUMENT,

  SFC_GET_LOOP_INFO,

  /* Following commands for testing only. */
  SFC_TEST_IEEE_FLOAT_REPLACE

// deprecated, so not including
//   /*
//   ** SFC_SET_ADD_* values are deprecated and will disappear at some
//   ** time in the future. They are guaranteed to be here up to and
//   ** including version 1.0.8 to avoid breakage of existng software.
//   ** They currently do nothing and will continue to do nothing.
//   */
//   SFC_SET_ADD_DITHER_ON_WRITE,
//   SFC_SET_ADD_DITHER_ON_READ
};


//-- simple format descriptor --------------------------------------------------
//  
//
struct SF_FORMAT_INFO {   
  int format;
 
  // no set-wrapper for those
  %immutable;			
  const char *name;
  const char *extension;
};

%extend SF_FORMAT_INFO {
  SF_FORMAT_INFO(int format) {
    SF_FORMAT_INFO *v = new SF_FORMAT_INFO;
    v->format    = format;
    v->name      = 0;
    v->extension = 0;
    return v;
  }

}

%header {
  const char *SF_FORMAT_INFO_name_get(SF_FORMAT_INFO *info) {
    if (info->name != 0) return info->name;
    return "";
  }

  const char *SF_FORMAT_INFO_extension_get(SF_FORMAT_INFO *info) {
    if (info->extension != 0) return info->extension;
    return "";
  }
}

//-- file info -----------------------------------------------------------------
// 
//
struct SF_INFO {
  sf_count_t frames;
  int samplerate;
  int channels;
  int format;
  int sections;
  int seekable;
};

%extend SF_INFO {
  SF_INFO(int format=0, int channels=1, int samplerate=44100) {
    SF_INFO *v = new SF_INFO;
    v->format = format;
    v->channels = channels;
    v->samplerate = samplerate;
    return v;
  }
};


//-- embed file info -----------------------------------------------------------
//
//
struct SF_EMBED_FILE_INFO {
  sf_count_t offset;
  sf_count_t length;
};

%extend SF_EMBED_FILE_INFO {
  SF_EMBED_FILE_INFO(long long offset=0, long long length=0) {
    SF_EMBED_FILE_INFO *v = new SF_EMBED_FILE_INFO;
    v->offset = offset;
    v->length = length;
    return v;
  }
}


//-- functions -----------------------------------------------------------------
// 

//-- open
// strait forward this; nothing to do
SNDFILE* sf_open(const char *path, int mode, SF_INFO *sfinfo);

// pysf_open_fd removed the auto-close file descriptor option
%rename(sf_open_fd) pysf_open_fd;
SNDFILE* pysf_open_fd(int fd, int mode, SF_INFO *sfinfo);

bool sf_format_check(const SF_INFO *info);

int sf_close(SNDFILE *sndfile);
void sf_write_sync(SNDFILE *sndfile);

sf_count_t sf_read_short (SNDFILE *sndfile, short  *OUT, sf_count_t items);
sf_count_t sf_read_int   (SNDFILE *sndfile, int    *OUT, sf_count_t items);
sf_count_t sf_read_float (SNDFILE *sndfile, float  *OUT, sf_count_t items);
sf_count_t sf_read_double(SNDFILE *sndfile, double *OUT, sf_count_t items);

sf_count_t sf_readf_short (SNDFILE *sndfile, short  *OUT, sf_count_t frames);
sf_count_t sf_readf_int   (SNDFILE *sndfile, int    *OUT, sf_count_t frames);
sf_count_t sf_readf_float (SNDFILE *sndfile, float  *OUT, sf_count_t frames);
sf_count_t sf_readf_double(SNDFILE *sndfile, double *OUT, sf_count_t frames);

sf_count_t sf_read_raw(SNDFILE *sndfile, void *OUT, sf_count_t bytes);

sf_count_t sf_write_short (SNDFILE *sndfile, short  *IN, sf_count_t items);
sf_count_t sf_write_int   (SNDFILE *sndfile, int    *IN, sf_count_t items);
sf_count_t sf_write_float (SNDFILE *sndfile, float  *IN, sf_count_t items);
sf_count_t sf_write_double(SNDFILE *sndfile, double *IN, sf_count_t items);

sf_count_t sf_writef_short (SNDFILE *sndfile, short  *IN, sf_count_t frames);
sf_count_t sf_writef_int   (SNDFILE *sndfile, int    *IN, sf_count_t frames);
sf_count_t sf_writef_float (SNDFILE *sndfile, float  *IN, sf_count_t frames);
sf_count_t sf_writef_double(SNDFILE *sndfile, double *IN, sf_count_t frames);

sf_count_t sf_write_raw(SNDFILE *sndfile, void *IN, sf_count_t bytes);

sf_count_t sf_seek(SNDFILE *sndfile, sf_count_t frames, int whence);

int sf_error(SNDFILE *sndfile);
const char* sf_strerror(SNDFILE *sndfile);
const char* sf_error_number(int errnum);

const char* sf_get_string(SNDFILE *sndfile, int str_type);
int sf_set_string(SNDFILE *sndfile, int str_type, const char *str);

//-- command 
//
// this is a bit more challenging to wrap:
// 
// int sf_command(SNDFILE *sndfile, int CMD, void *data, int datasize);
//
// libsndfile passes 4 different (python-)parameter:
// - no parameter:
//   SFC_GET_LIB_VERSION
//   SFC_GET_LOG_INFO
//   SFC_CALC_SIGNAL_MAX
//   SFC_CALC_NORM_SIGNAL_MAX
//   SFC_GET_NORM_FLOAT
//   SFC_GET_NORM_DOUBLE
//   SFC_GET_CLIPPING
//   SFC_GET_SIMPLE_FORMAT_COUNT
//   SFC_GET_FORMAT_MAJOR_COUNT
//   SFC_GET_FORMAT_SUBTYPE_COUNT
//   SFC_UPDATE_HEADER_NOW
//
// - long long (wrapped as INTEGER)
//   SFC_CALC_MAX_ALL_CHANNELS
//   SFC_CALC_NORM_MAX_ALL_CHANNELS
//
//   SFC_FILE_TRUNCATE
//   SFC_SET_RAW_START_OFFSET
//   
// - boolean (wrapped as INTEGER)
//   SFC_SET_NORM_FLOAT
//   SFC_SET_NORM_DOUBLE
//   SFC_SET_SCALE_FLOAT_INT_READ
//   SFC_SET_ADD_PEAK_CHUNK
//   SFC_SET_UPDATE_HEADER_AUTO
//   SFC_SET_CLIPPING
//
// - SF_FORMAT_INFO
//   SFC_GET_FORMAT_INFO
//   SFC_GET_FORMAT_MAJOR
//   SFC_GET_FORMAT_SUBTYPE
//
// - SF_EMBED_FILE_INFO
//   SFC_GET_EMBED_FILE_INFO

%rename(sf_command) pysf_command;

PyObject *pysf_command(SNDFILE *sndfile, int cmd);
PyObject *pysf_command(SNDFILE *sndfile, int cmd, long long bool_longlong);
PyObject *pysf_command(SNDFILE *sndfile, int cmd, SF_FORMAT_INFO *info);
PyObject *pysf_command(SNDFILE *sndfile, int cmd, SF_EMBED_FILE_INFO *info);

//-- deprecated functions
// not supported:
// 
// int sf_perror(SNDFILE *sndfile);
// int sf_error_str(SNDFILE *sndfile, char *str, size_t len);

