/***************************************************************************/
/*   Copyright (C) 2006 by Uwe Mayer                                       */
/*   merkosh@hadiko.de                                                     */
/*                                                                         */
/*   This is free software; you can redistribute it and/or modify it       */
/*   under the terms of the GNU Lesser General Public License as           */
/*   published by the Free Software Foundation; either version 2.1 of      */
/*   the License, or (at your option) any later version.                   */
/*                                                                         */
/*   This software is distributed in the hope that it will be useful,      */
/*   but WITHOUT ANY WARRANTY; without even the implied warranty of        */
/*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU      */
/*   Lesser General Public License for more details.                       */
/*                                                                         */
/*   You should have received a copy of the GNU Lesser General Public      */
/*   License along with this software; if not, write to the                */
/*   Free Software Foundation, Inc.,                                       */
/*   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA.         */
/***************************************************************************/

%header %{
  #include <assert.h>
  
//-- SWIG preprocessor helper variables --------------------------------------
  // constants py_(TYPE)_t which can be constructed
  // from a basetype's name
  enum {
    py_short_t  = PyArray_SHORT,
    py_int_t    = PyArray_INT,
    py_float_t  = PyArray_FLOAT,
    py_double_t = PyArray_DOUBLE
  };


%}


//-- (TYPE *OUT, sf_count_t items) ---------------------------------------------
//
// Task: construct variable length data return types

// For known types (short, int, float, double) a memory area is
// malloc()ed and used with Numeric as foreign data storage.
// This works because an ownership flag can be set in the Numeric
// array such that Numeric will free() the allocated area, if collected.

%typemap(in) 
  (short  *OUT, sf_count_t) (short  *data), 
  (int    *OUT, sf_count_t) (int    *data),
  (float  *OUT, sf_count_t) (float  *data),
  (double *OUT, sf_count_t) (double *data)
{
  // get number of samples to read
  $2 = PyLong_AsLongLong($input);

  // malloc()ate memory area
  data = ($1_type)malloc($2*sizeof($*1_type));
  if (data == 0) {
    PyErr_SetString(PyExc_MemoryError, "could not allocate contiguous memory area");
    return 0;
  }
  $1 = data;
}

// For returning raw data a Python buffer is used instead of a
// Numeric array.
// The memory block must be allocate in form of a Python object
// rather than with malloc(), because no ownership can be transfered.
//
// Also the allocated buffer cannot exceed size of a C int.

%typemap(in)
  (void *OUT, sf_count_t) (PyObject *buffer, int length)
{
  // get number of bytes to read
  $2 = PyLong_AsLongLong($input);
  if ((int)$2 < 0) {
    PyErr_SetString(PyExc_ValueError, "cannot allocate buffer with more than 2**(sizeof(int)-1) bytes");
    return 0;
  }

  // create output buffer
  buffer = PyBuffer_New($2);
  if (buffer == 0) return 0;

  // extract writeable memory pointer from buffer
  if (PyObject_AsWriteBuffer(buffer, &$1, &length) != 0) 
    // on type-error pass on exception
    return 0;
  assert(length == $2);		// make sure what-you-get is what-you-ordered
}


//-- (TYPE *OUT) ---------------------------------------------------------------
//
// Task: resize and return variable length data

// The returned array size is set to the number of read samples, otherwhise
// the Python programmer had no way of knowing the end-of-file.
//
// To make the array release the malloc()ed memory an ownership flag is set.
// The malloc()ed area needs not to be resized as the array will only call
// free() on the memory pointer and free() does not depend on the amount of
// memory allocated.

%typemap(argout) 
  (short  *OUT),
  (int    *OUT),
  (float  *OUT),
  (double *OUT)
{
  // required output size
  // (not entirely sure if I should be accessing `result` here)
  int dims[1] = {result};

  // construct array with pointer to allocated memory
  PyObject *o = PyArray_FromDimsAndData(1, dims, py_$1_basetype_t, (char*)data$argnum);
  if (o == 0) return 0;

  // set ownership flag; omitting this causes severe memory leak
  ((PyArrayObject*)o)->flags |= 0x8;

  $result = o;
}

// Returns a Python read-write buffer object with the size of the number
// of bytes read from sf_read_raw().
//
// Resizing the buffer object seems not to be possible, therefore a new
// buffer-from-object is created which shares the same raw memory, but
// exposes only number of bytes read. The primary buffer is released afterwards.

%typemap(argout) 
  (void *OUT)
{
  if (result < length$argnum) {
    $result = PyBuffer_FromReadWriteObject(buffer$argnum, 0, result);
    Py_DECREF(buffer$argnum);	// free original buffer
  } else
    $result = buffer$argnum;
}


//-- (sf_count_t) --------------------------------------------------------------
//
// Task: wrap sf_count_t (C --> Python) and (Python --> C)

// As return type: `sf_count_t sf_tell(...)`

%typemap(out) (sf_count_t)
{
  $result = PyLong_FromLongLong($1);
  if ($result == 0) return 0;
}

// As parameter type: `void sf_seek(sf_count_t pos)`

%typemap(in) (sf_count_t)
{
  $1 = PyLong_AsLongLong($input);
  if ($1 == 0) return 0;
}


//-- (TYPE *IN) ----------------------------------------------------------------
//
// Task: wrap input buffer: `sf_write_TYPE(TYPE *IN, sf_count_t items)`

// Wraps a Python objects which implements the buffer interface to
// and extracts a pointer to the *contiguous data area* and the number
// of items (sizeof(TYPE)) to write.
//
// This is a little tricky to use: libsndfile requires scaled data,
// i.e. float data must be scaled [-1, ..., +1] and really be float,
// not double.
//
// Example:
//
// data0 = sf_read_short(f, 1024**2)  # read data in 2 byte format
// data1 = data0.astype(Float32)      # converts Int16 to Float32
// data2 = data1 / 2**15              # scales Int16 to [-1,+1]; result: Float64
// sf_write_double(f, data2)          # save Float64 data
//
// data3 = data2.astype(Float32)      # convert Float64 to Float32
// sf_write_float(f, data3)           # save Float32 data
//
// data4 = data3.tostring()           # create string object
// sf_write_float(f, data4)           # save Float32 from string
//
// sf_write_raw(f, data2)             # works on SF_FORMAT_DOUBLE
// sf_write_raw(f, data3)             # works on SF_FORMAT_FLOAT
// sf_write_raw(f, data4)             # works on SF_FORMAT_FLOAT

%typemap(in)
  (short  *IN, sf_count_t),
  (int    *IN, sf_count_t),
  (float  *IN, sf_count_t),
  (double *IN, sf_count_t)
{
  // extract read-buffer data area
  int length;
  if (PyObject_AsReadBuffer($input, (const void**)&$1, &length) != 0)
     return 0;

  // set number of items to write
  $2 = PyObject_Length($input);
}

%typemap(in)
  (void *IN, sf_count_t)
{
  // extract read-buffer data area
  int length;
  if (PyObject_AsReadBuffer($input, (const void**)&$1, &length) != 0)
     return 0;

  // set number of bytes to write
  $2 = length;
}


//-- (long long bool_longlong) -------------------------------------------------
//
// Task: wrap python boolean or long ints to C long long

// Some functions require SF_TRUE /SF_FALSE, which are integer constants,
// to be passed from python boolean values. Others require an integer,
// i.e. the number of channels to be passed. Thats why SWIGs default
// wrapper won't suffice.

%typemap(in)
  (long long bool_longlong)
{
  if (PyBool_Check($input)) 
    $1 = ($input == Py_False)? SF_FALSE : SF_TRUE;
  else 
    $1 = PyLong_AsLongLong($input);
}


