# test_hprofile.py -- unittests for hprofile module

# Copyright (c) 2005 Floris Bruynooghe

# All rights reserved.

# Permission is hereby granted, free of charge, to any person
# obtaining a copy of this software and associated documentation files
# (the "Software"), to deal in the Software without restriction,
# including without limitation the rights to use, copy, modify, merge,
# publish, distribute, and/or sell copies of the Software, and to
# permit persons to whom the Software is furnished to do so, provided
# that the above copyright notice(s) and this permission notice appear
# in all copies of the Software and that both the above copyright
# notice(s) and this permission notice appear in supporting
# documentation.

# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT OF THIRD PARTY RIGHTS. IN NO EVENT SHALL THE
# COPYRIGHT HOLDER OR HOLDERS INCLUDED IN THIS NOTICE BE LIABLE FOR
# ANY CLAIM, OR ANY SPECIAL INDIRECT OR CONSEQUENTIAL DAMAGES, OR ANY
# DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
# WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
# ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE
# OF THIS SOFTWARE.

# Except as contained in this notice, the name of a copyright holder
# shall not be used in advertising or otherwise to promote the sale,
# use or other dealings in this Software without prior written
# authorization of the copyright holder.


import unittest
import os.path
import hotshot
import time
import os
from test import test_support

import hprofile
import hstats


def root():
    """Simple function to profile."""
    pass


class ProfileConstructor(unittest.TestCase):
    def setUp(self):
        self.prof = hprofile.Profile()
        
    def test_profile_opened(self):
        # Is the profiling file created correctly?
        self.failUnless(os.path.exists(self.prof._fname))

    def test_profile_closed(self):
        # Is the profiling file removed after use?
        fname = self.prof._fname
        del self.prof
        self.failIf(os.path.exists(fname))

    def test_hotshot_loaded(self):
        # Is Profile._profiler the hotshot profiler?
        self.failUnless(isinstance(self.prof._profiler, hotshot.Profile))

    def test_bias_applied_arg(self):
        # Does the bias get saved into the profile file?
        p = hprofile.Profile(bias=0.5)
        p.runctx('root()', globals(), locals())
        stats = hstats.Stats(p._fname)
        info = stats.get_info()
        self.failUnless('hprofile-bias' in info.keys())
        self.failUnless(float(info['hprofile-bias'][0]))


class PrintStats(unittest.TestCase):
    def test_invalid_sort(self):
        # Invalid sort argument detected?
        prof = hprofile.Profile()
        self.failUnlessRaises(ValueError, prof.print_stats, sort=7)
        self.failUnlessRaises(ValueError, prof.print_stats, sort=['x', 'y'])


class DumpStats(unittest.TestCase):
    def setUp(self):
        self.prof = hprofile.Profile()
        self.prof.runctx('root()', globals(), locals())
        self.prof.dump_stats(test_support.TESTFN)

    def tearDown(self):
        test_support.unlink(test_support.TESTFN)

    def test_file_created(self):
        # Does the file get created?
        self.failUnless(os.path.exists(test_support.TESTFN))

    def test_file_compare(self):
        # Is the saved file the same as the file of the profiler?
        dumpf = file(test_support.TESTFN, mode='rb')
        proff = file(self.prof._fname, mode='rb')
        self.failUnlessEqual(dumpf.read(), proff.read())


class Run(unittest.TestCase):
    def test_return_value(self):
        # Profile.run() returns `self'?
        prof = hprofile.Profile()
        ret = prof.runctx('root()', globals(), locals())
        self.failUnless(isinstance(ret, type(prof)))


class RunCtx(unittest.TestCase):
    def test_return_value(self):
        # Profile.runctx() returns `self'?
        prof = hprofile.Profile()
        ret = prof.runctx('root()', globals(), locals())
        self.failUnless(isinstance(ret, type(prof)))


class RunCall(unittest.TestCase):
    def test_return_value(self):
        # Profile.runcall() returns call result?
        prof = hprofile.Profile()
        ret = prof.runcall(root)
        self.failUnlessEqual(ret, None)


class Calibrate(unittest.TestCase):
    def setUp(self):
        self.prof = hprofile.Profile()
        self.ret = self.prof.calibrate(100) # Keep it small to save time
        
    def test_return_value(self):
        # Profile.clibrate() returns a number?
        self.failUnless(isinstance(self.ret, float))

    def test_bias_positive(self):
        # The returnded bias is positive?
        self.failUnless(self.ret >= 0)

    def test_insufficient_calls(self):
        # What happens when to few calls are asked?
        self.failUnless(0 <= self.prof.calibrate(99))


class Stats(unittest.TestCase):
    def setUp(self):
        prof = hprofile.Profile()
        prof.runctx('root()', globals(), locals())
        self.stats = hprofile._Stats(prof._fname)

    def test_data_float(self):
        # Is does self._data contain floats instead of ints?
        data = self.stats._data
        times = []
        ctimes = []
        for key in data.keys():
            times.append(data[key][hstats._SDATA_TIME])
            ctimes.append(data[key][hstats._SDATA_CUMTIME])
        for time in times:
            self.failUnless(isinstance(time, float))
        for ctime in ctimes:
            self.failUnless(isinstance(ctime, float))


class Bias(unittest.TestCase):
    def test_bias_subtracted(self):
        prof = hprofile.Profile(bias=10)
        prof.runctx('root()', globals(), locals())
        stats = hprofile._Stats(prof._fname)
        data, dd = stats.get_data(weed=['name', 'call',
                                        'avgtime', 'avgcumtime'])
        for row in data:
            for item in row:
                self.failUnless(item < 0)


class Timer(unittest.TestCase):
    def test_timer_accepted(self):
        # Profile.__init__() accepts the timer function?
        self.failUnless(hprofile.Profile(timer=time.time))


class ModuleFunctions(unittest.TestCase):
    def tearDown(self):
        if os.access(test_support.TESTFN, os.F_OK):
            test_support.unlink(test_support.TESTFN)

    def test_run(self):
        # Since this depends on root() being in the globals we can
        # only do this when this is indeed so.
        if __name__ == "__main__":
            self.failUnlessEqual(None,
                                 hprofile.run('root()',
                                              filename=test_support.TESTFN))

    def test_runctx(self):
        self.failUnlessEqual(None,
                             hprofile.runctx('root()', globals(), locals(),
                                             filename=test_support.TESTFN))


def test_main():
    test_support.run_unittest(ProfileConstructor,
                              PrintStats,
                              DumpStats,
                              Run,
                              RunCtx,
                              RunCall,
                              Calibrate,
                              Stats,
                              Bias,
                              Timer,
                              ModuleFunctions)


if __name__ == '__main__':
    test_main()
