# test_hstats.py -- unittests for hstats module

# Copyright (c) 2005 Floris Bruynooghe

# All rights reserved.

# Permission is hereby granted, free of charge, to any person
# obtaining a copy of this software and associated documentation files
# (the "Software"), to deal in the Software without restriction,
# including without limitation the rights to use, copy, modify, merge,
# publish, distribute, and/or sell copies of the Software, and to
# permit persons to whom the Software is furnished to do so, provided
# that the above copyright notice(s) and this permission notice appear
# in all copies of the Software and that both the above copyright
# notice(s) and this permission notice appear in supporting
# documentation.

# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT OF THIRD PARTY RIGHTS. IN NO EVENT SHALL THE
# COPYRIGHT HOLDER OR HOLDERS INCLUDED IN THIS NOTICE BE LIABLE FOR
# ANY CLAIM, OR ANY SPECIAL INDIRECT OR CONSEQUENTIAL DAMAGES, OR ANY
# DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
# WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
# ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE
# OF THIS SOFTWARE.

# Except as contained in this notice, the name of a copyright holder
# shall not be used in advertising or otherwise to promote the sale,
# use or other dealings in this Software without prior written
# authorization of the copyright holder.


import unittest
import hotshot
import os
import os.path
import sys
import tempfile
import copy
import random
import exceptions
from test import test_support

import hstats


class Break(exceptions.Exception):
    """Stop the execution stack in a test."""
    pass


def root():
    """Root function call for test profile."""
    recursion(2)
    recursion(0)
    do_nothing()


def recursion(x):
    """Simple do-nothing recursive test function.
    
    x --  The number of recursions needed to do.
    """
    if x == 0:
        return
    else:
        recursion(x-1)


def do_nothing():
    """Simple do-nothing function"""
    pass


class StatsData(unittest.TestCase):
    # Tests about the Stats.__init__() code and correct data.
    def setUp(self):
        profiler = hotshot.Profile(test_support.TESTFN)
        profiler.run('root()')
        profiler.close()
        self.stats = hstats.Stats(test_support.TESTFN)
        self.data = self.stats._data

    def tearDown(self):
        test_support.unlink(test_support.TESTFN)
        
    def unused_test_dummy(self):
        # Dummy test to try out the setUp() code.
        print
        import pprint
        p = pprint.PrettyPrinter()
        p.pprint(self.data)
        self.assert_(True)
        test_support.forget('pprint')

    def test_exec_string_stripped(self):
        #The '<string>' name from exec() gets stripped correctly?
        for n in self.data.keys():
            if n == ('<string>', 1, '?'):
                self.fail("Not stipped!")

    def test_root_calls(self):
        # root() testfunction has correct no of calls?
        for n in self.data.keys():
            if 'root' in n:
                name = n
        self.assertEqual(self.data[name][hstats._SDATA_CALL], [1, 0])

    def test_recursion_calls(self):
        # recursion() testfunction has correct no of calls?
        for n in self.data.keys():
            if 'recursion' in n:
                name = n
        self.assertEqual(self.data[name][hstats._SDATA_CALL], [4, 2])

    def test_time_cumtime(self):
        # Time and cumulative time don't conflict?
        for d in self.data.values():
            time = d[hstats._SDATA_TIME]
            ctime = d[hstats._SDATA_CUMTIME]
            self.assert_(time <= ctime, "%d <= %d" % (time, ctime))

    def test_cumtimes(self):
        # Cumulative time of root() is larger then other cumulative times?
        subctimes = list()
        for n in self.data.keys():
            if 'root' in n:
                rctime = self.data[n][hstats._SDATA_CUMTIME]
            else:
                subctimes.append(self.data[n][hstats._SDATA_CUMTIME])
        for time in subctimes:
            self.assert_(rctime > time, "%d > %d" % (rctime, time))

    def test_cumtime_root(self):
        # Cumulative time of root() is correct?
        subtimes = list()
        for n in self.data.keys():
            if 'root' in n:
                rctime = self.data[n][hstats._SDATA_CUMTIME]
                subtimes.append(self.data[n][hstats._SDATA_TIME])
            else:
                subtimes.append(self.data[n][hstats._SDATA_TIME])
        self.assertEqual(rctime, sum(subtimes))

    def test_parent_root(self):
        # The parent of root() is recored properly?
        for name in self.data.keys():
            if 'root' in name:
                frame = name
        self.assertEqual(self.data[frame][hstats._SDATA_PARENTS],
                         {('<string>', 1, '?'): 1})

    def test_empty_profile(self):
        # Appropriate exception raised if profile file is empty?
        test_support.unlink(test_support.TESTFN)
        profiler = hotshot.Profile(test_support.TESTFN)
        profiler.run('pass')
        profiler.close()
        self.assertRaises(EOFError, hstats.Stats, test_support.TESTFN)

    def test_empty_profile_when_asked(self):
        # '<string>' is preserved when asked to?
        test_support.unlink(test_support.TESTFN)
        p = hotshot.Profile(test_support.TESTFN)
        p.run('pass')
        p.close()
        s = hstats.Stats(test_support.TESTFN, keep_exec=True)
        self.assert_(('<string>', 1, '?') in s._data.keys())


class Show(unittest.TestCase, hstats.Stats):
    def setUp(self):
        self._data = {('some/dir/name', 24, 'func1'): [[2, 0], 45, 45, {}],
                      ('some/other/dir', 57, 'func2'): [[1, 0], 34, 34, {}]}

    def print_data_wrapper(self, *args, **kw):
        """Wrapper used by the test below."""
        hstats.Stats.print_data(self, fd=self.tmpfile, *args, **kw)

    def test_default(self):
        # Default output works correctly?
        expected = " calls  tott  avgt  cumt avgct  name         \n" \
                   "     2    45    22    45    22  name:24:func1\n" \
                   "     1    34    34    34    34  dir:57:func2 \n"
        self.tmpfile = outputf = tempfile.TemporaryFile()
        bak_print_data = self.print_data
        self.print_data = self.print_data_wrapper
        self.show()
        self.print_data = bak_print_data
        outputf.flush()
        outputf.seek(0)
        output = outputf.read()
        exp = expected.split('\n')
        out = output.split('\n')
        for o, e in zip(out, exp):
            self.assertEqual(o, e)

    def dummy_sort_data(self, data, order=None):
        """Helper function for next test."""
        # This is (sort of) data duplication from Stats.show()
        self.assertEqual(order, [(1, 'd'), (2, 'd'), (0, 'd'), (5, 'a')])
        raise Break

    def test_sort_args(self):
        # Sorting args gets passed correctly to .sort_data()?
        bak = self.sort_data
        self.sort_data = self.dummy_sort_data
        try:
            self.show(sort=['time', 'avgtime', 'call', 'name'])
        except Break:
            pass
        self.sort_data = bak

    def dummy_get_data(self, weed=None):
        """Helper function for next test."""
        self.assertEquals(['dirs', 'name'], weed)
        raise Break

    def test_weed_args(self):
        # Weed args gets passed correctly to .get_data()?
        bak = self.get_data
        self.get_data = self.dummy_get_data
        try:
            self.show(weed=['dirs', 'name'])
        except Break:
            pass
        self.get_data = bak

    def dummy_print_data(self, data, headers=True, limit=20):
        # Helper function for next test.
        self.assertEquals(10, limit)

    def test_limit_arg(self):
        # Limit arg gets passed correctly to .print_data()?
        bak = self.print_data
        self.print_data = self.dummy_print_data
        self.show(limit=10)
        self.print_data = bak


class GetData(unittest.TestCase, hstats.Stats):
    def setUp(self):
        self._data = {('some/dir/name', 24, 'func1'):
                      [[2, 0], 45, 45, {}],
                      ('some/other/dir', 57, 'func2'):
                      [[1, 0], 34, 34, {('some/dir/name', 24, 'func1'): 1}]}

    def test_no_weed(self):
        # All data goes through unweeded?
        result = [[[2, 0], 45, 22, 45, 22, ('some/dir/name', 24, 'func1')],
                  [[1, 0], 34, 34, 34, 34, ('some/other/dir', 57, 'func2')]]
        dds = ['call', 'time', 'avgtime', 'cumtime', 'avgcumtime', 'name']
        data, dd = self.get_data()
        for drow, rrow in zip(data, result):
            self.assertEqual(len(drow), len(rrow))
            for rindex, info in enumerate(dds):
                dindex = dd.index(info)
                self.assertEqual(drow[dindex], rrow[rindex])

    def test_weed_dirs(self):
        # Directory path get's weeded correctly?
        result = [[[2, 0], 45, 22, 45, 22, ('name', 24, 'func1')],
                  [[1, 0], 34, 34, 34, 34, ('dir', 57, 'func2')]]
        dds = ['call', 'time', 'avgtime', 'cumtime', 'avgcumtime', 'name']
        data, dd = self.get_data(weed=['dirs'])
        for drow, rrow in zip(data, result):
            self.assertEqual(len(drow), len(rrow))
            for rindex, info in enumerate(dds):
                dindex = dd.index(info)
                self.assertEqual(drow[dindex], rrow[rindex])

    def test_weed_other(self):
        # Weed works as expected (excluding 'dirs')?
        result = [[[2, 0], 45, 22, 45, 22, ('some/dir/name', 24, 'func1')],
                  [[1, 0], 34, 34, 34, 34, ('some/other/dir', 57, 'func2')]]
        dds = ['call', 'time', 'avgtime', 'cumtime', 'avgcumtime', 'name']
        for thing in dds:
            data, dd = self.get_data(weed=[thing])
            for drow, rrow in zip(data, result):
                for rindex, info in enumerate(dds):
                    if info is not thing:
                        dindex = dd.index(info)
                        self.assertEqual(drow[dindex], rrow[rindex])

    def test_weed_combo(self):
        # Weeding more then one thing works correctly?
        result = [[[2, 0], 45, 22, 45],
                  [[1, 0], 34, 34, 34]]
        dds = ['call', 'time', 'avgtime', 'cumtime']
        data, dd = self.get_data(weed=['avgcumtime', 'name'])
        for drow, rrow in zip(data, result):
            self.assertEqual(len(drow), len(rrow))
            for rindex, info in enumerate(dds):
                dindex = dd.index(info)
                self.assertEqual(drow[dindex], rrow[rindex])

    def test_dd(self):
        # Data Description of returned is correct?
        dd_expected = ['call', 'time', 'avgtime',
                       'cumtime', 'avgcumtime', 'name']
        data, dd = self.get_data()
        self.assertEqual(dd, dd_expected)

    def test_invalid_arg(self):
        # Exception raised when invalid `weed' arg passed?
        self.assertRaises(ValueError, self.get_data, weed='dirs')

    def test_extra_parents(self):
        # When asked the parents data is also returned?
        dd_expected = ['call', 'time', 'avgtime',
                       'cumtime', 'avgcumtime', 'name', 'parents']
        data_expected = [[[2, 0], 45, 22, 45, 22,
                          ('some/dir/name', 24, 'func1'),
                          {}],
                         [[1, 0], 34, 34, 34, 34,
                          ('some/other/dir', 57, 'func2'),
                          {('some/dir/name', 24, 'func1'): 1}]]
        data_expected.sort()
        data, dd = self.get_data(extra=['parents'])
        data.sort()
        self.assertEqual(dd, dd_expected)
        self.assertEqual(data, data_expected)


class SortData(unittest.TestCase, hstats.Stats):
    def test_simple(self):
        # Stats.sort_data() sorts correctly?
        result = [[[2, 0], 45, 45, 22, 22, ('some/dir/a', 24, 'f')],
                  [[2, 0], 45, 45, 22, 22, ('some/dir/a', 28, 'g')],
                  [[1, 0], 34, 34, 34, 34, ('some/dir/b', 57, 'f')]]
        data = copy.copy(result)
        random.shuffle(data)
        sorted = self.sort_data(data, [(0, 'd'), (5, 'a')])
        self.assertEqual(sorted, result)


class PrintData(unittest.TestCase, hstats.Stats):
    def setUp(self):
        self.data = [[[2, 0], 45, 45, 22, 22, ('some/dir/a', 24, 'f')],
                     [[1, 0], 34, 34, 34, 34, ('some/dir/b', 57, 'g')]]
        self.dd = ['call', 'time', 'cumtime', 'avgtime', 'avgcumtime', 'name']
        self.fd = tempfile.TemporaryFile()
        
    def compare_string(self, string):
        """Compare a file and a string.

        fd - file discriptor of the file.

        string - string to compare contents with.
        """
        self.fd.flush()
        self.fd.seek(0)
        output = self.fd.read()
        self.assertEqual(output, string)

    def test_simple(self):
        # Stats.print_data() prints correct data?
        result = " calls  tott  avgt  cumt avgct  name           \n" \
                 "     2    45    22    45    22  some/dir/a:24:f\n" \
                 "     1    34    34    34    34  some/dir/b:57:g\n"
        self.print_data(self.data, self.dd, fd=self.fd)
        self.compare_string(result)

    def test_limit(self):
        # Limit argument to stats works correctly?
        result = " calls  tott  avgt  cumt avgct  name           \n" \
                 "     2    45    22    45    22  some/dir/a:24:f\n"
        self.print_data(self.data, self.dd, limit=1, fd=self.fd)
        self.compare_string(result)

    def test_limit_zero(self):
        # Limit argument of '0' treated as infinite?
        result = " calls  tott  avgt  cumt avgct  name           \n" \
                 "     2    45    22    45    22  some/dir/a:24:f\n" \
                 "     1    34    34    34    34  some/dir/b:57:g\n"
        self.print_data(self.data, self.dd, limit=0, fd=self.fd)
        self.compare_string(result)


class GetInfo(unittest.TestCase):
    def test_info_is_dict(self):
        # The info returned is a dictionarry?
        profiler = hotshot.Profile(test_support.TESTFN)
        profiler.run('root()')
        profiler.close()
        stats = hstats.Stats(test_support.TESTFN)
        info = stats.get_info()
        test_support.unlink(test_support.TESTFN)
        self.assert_(isinstance(info, dict))


def test_main():
    test_support.run_unittest(StatsData,
                              Show,
                              GetData,
                              SortData,
                              PrintData,
                              GetInfo)


if __name__ == '__main__':
    test_main()
