# hprofile.py -- wrapper around hotshot to emulate profile module

# Copyright (c) 2005 Floris Bruynooghe

# All rights reserved.

# Permission is hereby granted, free of charge, to any person
# obtaining a copy of this software and associated documentation files
# (the "Software"), to deal in the Software without restriction,
# including without limitation the rights to use, copy, modify, merge,
# publish, distribute, and/or sell copies of the Software, and to
# permit persons to whom the Software is furnished to do so, provided
# that the above copyright notice(s) and this permission notice appear
# in all copies of the Software and that both the above copyright
# notice(s) and this permission notice appear in supporting
# documentation.

# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT OF THIRD PARTY RIGHTS. IN NO EVENT SHALL THE
# COPYRIGHT HOLDER OR HOLDERS INCLUDED IN THIS NOTICE BE LIABLE FOR
# ANY CLAIM, OR ANY SPECIAL INDIRECT OR CONSEQUENTIAL DAMAGES, OR ANY
# DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS,
# WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
# ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE
# OF THIS SOFTWARE.

# Except as contained in this notice, the name of a copyright holder
# shall not be used in advertising or otherwise to promote the sale,
# use or other dealings in this Software without prior written
# authorization of the copyright holder.

"""hprofile - Class for profiling Python Code.

This module should behave exactly as the profile module.
"""


# Before importing hotshot we need to make sure our version of
# _hotshot appears before the system installed one.  This means we
# should move the sys.path entry containing "hprof" before the one
# containing "lib-dynload".
import sys
import os.path
for index, path in enumerate(sys.path):
    head, tail = os.path.split(path)
    if tail == "lib-dynload":
        dynload_index = index
    if tail == "hprof":
        hprof_path = path
        hprof_index = index
try:
    del sys.path[hprof_index]
except NameError:
    pass
else:
    sys.path.insert(dynload_index, hprof_path)
    del dynload_index
    del hprof_path
    del hprof_index


import hotshot
import tempfile
import os
import timeit

import hstats
import hpstats


__all__ = ['run', 'runctx', 'help', 'Profile']


def _calibration_root(n):
    """Root of calibration loop.
    
    n - The number of times _calibrate_sub() gets called
    """
    for i in xrange(n):
        _calibration_sub()


def _calibration_sub():
    """Function to call many times from _calibration_root()"""
    for i in xrange(100):       # want to spend _some_ time in here
        1 + 1


class Profile:
    bias = 0                            # calibration constant

    def __init__(self, timer=None, bias=None):
        fdno, self._fname = tempfile.mkstemp(prefix="hprofile")
        self._profiler = hotshot.Profile(self._fname)

        if bias != None:
            self.bias = bias = float(bias)
            if bias != 0:
                self._profiler.addinfo('hprofile-bias', repr(bias))
        elif self.bias != 0:
            self.bias = bias = float(self.bias)
            if bias != 0:
                self._profiler.addinfo('hprofile-bias', repr(bias))

        self._timer = None
        if timer != None:
            if not hasattr(self._profiler._prof, "settimer"):
                raise RuntimeError, "loaded _hotshot module does " \
                      "not support settimer()."
            self._profiler._prof.settimer(timer)
            self._timer = timer

    def __del__(self):
        os.unlink(self._fname)

    def print_stats(self, sort=-1):
        """Print out the statistics gathered.

        sort - Can be one of -1, 0, 1 or 2.  The results are then
          sorted by name, number of calls, time and cumulative time
          respectively.  Numbers are sorted descending, names
          ascending.
        """
        if sort not in [-1, 0, 1, 2]:
            raise ValueError, "invalid sort option: '%s'" % sort
        stats = hpstats.Stats(self._fname)
        stats.strip_dirs()
        stats.sort_stats(sort)
        stats.print_stats()
    
    def dump_stats(self, filename):
        """Write the profile data to a file we know how to load back.

        It is not guaranteed that the file will be readable between
        different versions of the profiler.
        """
        # This does not close the current profiler, instead it does
        # just take a snapshot of the current state.
        infile = file(self._fname, mode='rb')
        outfile = file(filename, mode='wb')
        outfile.write(infile.read())
        outfile.close()
        infile.close()
    
    def run(self, cmd):
        """Evaluate an exec() compatible string.

        The globals from the __main__ module are used as both the
        globals and locals for the script.
        """
        return self._profiler.run(cmd)
    
    def runctx(self, cmd, globals, locals):
        """Evaluate an exec-compatible string in a specific environment.

        The string is compiled before profiling begins.
        """
        return self._profiler.runctx(cmd, globals, locals)
    
    def runcall(self, func, *args, **kw):
        """Profile a single call of a callable.

        Additional positional and keyword arguments may be passed
        along; the result of the call is returned, and exceptions are
        allowed to propogate cleanly, while ensuring that profiling is
        disabled on the way out.
        """
        return self._profiler.runcall(func, *args, **kw)
    
    def calibrate(self, no_calls):
        """Calculate and returns calibration constant.

        This constant can then be set as the `bias' of the profiler.
        It represents the profiler overhead in every function call.
        This can be significant when the code has lots of small
        functions.

        m - The number of times to call a function.  The bigger the
          number the more accuracy.

        Verbose mode was not in the reference manual so has not been
        duplicated.
        """
        if no_calls < 100:
            no_calls = 100
        bias_bak = self.bias
        self.bias = 0
        
        global _CALIBRATION_NO
        _CALIBRATION_NO = no_calls
        stmt = '_calibration_root(_CALIBRATION_NO)'
        setup = 'from hprofile import _calibration_root, _calibration_sub,' \
                '_CALIBRATION_NO'
        if self._timer is not None:
            timer = timeit.Timer(stmt, setup, timer=self._timer)
        else:     # At least on UNIX this uses the same timer as _hotshot.
            timer = timeit.Timer(stmt, setup)
        times = timer.repeat(number=int(no_calls/100))
        time_noprof = min(times)/int(no_calls/100)
        
        p = Profile()
        p.runctx('_calibration_root(no_calls)', globals(), locals())
        p._profiler.close()
        stats = hstats.Stats(p._fname)
        data, dd = stats.get_data(weed=['dirs', 'call', 'time',
                                        'avgtime', 'avgcumtime'])
        for row in data:
            if '_calibration_root' in row[dd.index('name')]:
                time_prof = row[dd.index('cumtime')]/1000000.0

        mean = (time_prof - time_noprof)/(no_calls + 1)/2.0
        if mean < 0:
            mean = 0

        self.bias = bias_bak
        return mean
    
    # Following are the methods I regard as profile internals dispite
    # them not using a leading underscore.  I can't find or think of a
    # use for them.  They will not be implemented.  Contact me if you
    # think they should be there.

    def trace_dispatch(self, frame, event, arg):
        # Heavily optimized dispatch routine for os.times() timer.
        raise NotImplementedError, \
              "internal function, contact maintainer if use was genuine."

    def trace_dispatch_i(self, frame, event, arg):
        # Dispatch routine for best timer program (return = scalar,
        # fastest if an integer but float works too -- and
        # time.clock() relies on that).
        raise NotImplementedError, \
              "internal function, contact maintainer if use was genuine."

    def trace_dispatch_mac(self, frame, event, arg):
        # Dispatch routine for macintosh (timer returns time in ticks
        # of 1/60th second)
        raise NotImplementedError, \
              "internal function, contact maintainer if use was genuine."

    def trace_dispatch_l(self, frame, event, arg):
        # SLOW generic dispatch routine for timer returning lists of
        # numbers.
        raise NotImplementedError, \
              "internal function, contact maintainer if use was genuine."

    def trace_dispatch_exception(self, frame, t):
        raise NotImplementedError, \
              "internal function, contact maintainer if use was genuine."

    def trace_dispatch_call(self, frame, t):
        raise NotImplementedError, \
              "internal function, contact maintainer if use was genuine."

    def trace_disptach_c_call(self, frame, t):
        raise NotImplementedError, \
              "internal function, contact maintainer if use was genuine."

    def trace_dispatch_return(self, frame, t):
        raise NotImplementedError, \
              "internal function, contact maintainer if use was genuine."

    def set_cmd(self, cmd):
        raise NotImplementedError, \
              "internal function, contact maintainer if use was genuine."

    class fake_code:
        def __init__(self, filename, line, name):
            raise NotImplementedError, \
                  "internal function, contact maintainer if use was genuine."

        def __repr__(self):
            raise NotImplementedError, \
                  "internal function, contact maintainer if use was genuine."

    class fake_frame:
        def __init__(self, code, prior):
            raise NotImplementedError, \
                  "internal function, contact maintainer if use was genuine."

    def simulate_call(self, name):
        raise NotImplementedError, \
              "internal function, contact maintainer if use was genuine."

    def simulate_cmd_complete(self):
        raise NotImplementedError, \
              "internal function, contact maintainer if use was genuine."

    def create_stats(self, file):
        raise NotImplementedError, \
              "internal function, contact maintainer if use was genuine."

    def snapshot_stats(self):
        raise NotImplementedError, \
              "internal function, contact maintainer if use was genuine."


class _Stats(hstats.Stats):
    """Subclass for stats handling results in seconds.

    The hstats.Stats class does handle everything in hotshot's
    nanoseconds.  This subclass will convert the data into
    seconds like the old pstats did.
    """
    def __init__(self, filename):
        hstats.Stats.__init__(self, filename)
        bias = None
        if 'hprofile-bias' in self.get_info().keys():
            bias = float(self.get_info()['hprofile-bias'][0])
        for key in self._data.keys():
            # Convert to seconds
            self._data[key][hstats._SDATA_TIME] /= 1000000.0
            self._data[key][hstats._SDATA_CUMTIME] /= 100000.0
            if bias != None:
                self._data[key][hstats._SDATA_TIME] -= bias
                self._data[key][hstats._SDATA_CUMTIME] -= bias


def Stats(*args):
    """Legacy function -- useless."""
    print 'Report generating functions are in the "fpstats" module\a'


def help():
    """Legacy function -- depreciated."""
    print "Documentation for the profile module can be found "
    print "in the Python Library Reference, section 'The Python Profiler'."


def run(statement, filename=None, sort=-1):
    """Run statement under profiler optonally saving results in filename.

    This function takes a single argument that can be passed to the
    'exec' statement and an optional file name.  In all cases this
    routine attempts to 'exec' its first argument and gather profiling
    statistics from the execution.  If no file name is present, then
    this function automatically prints a simple profiling report,
    sorted by the standard name strings (file/line/function-name) that
    is presented in each line.
    """
    p = Profile()
    p.run(statement)

    if filename is not None:
        p.dump_stats(filename)
    else:
        return p.print_stats(sort)


def runctx(statement, globals_dict, locals_dict, filename=None):
    """Run statement under profiler, supplying globals and locals.

    statement and filename have the same semantics as profile.run.
    Optionally the result can be saved to a filename.
    """
    p = Profile()
    p.runctx(statement, globals_dict, locals_dict)

    if filename is not None:
        p.dump_stats(filename)
    else:
        return p.print_stats()


# Run the profiler on a file when invoked directly
if __name__ == "__main__":
    import sys
    import os.path
    from optparse import OptionParser

    usage = "usage: %prog [-o output_file] [-s sort] scriptfile [arg] ..."
    parser = OptionParser(usage=usage)
    parser.add_option('-o', '--outfile', dest="outfile",
                      help="Save stats to <outfile>", default=None)
    parser.add_option('-s', '--sort', dest="sort",
                      help="Sort order when printing to stdout, " \
                      "based on pstats.Stats class", default=-1)

    (options, args) = parser.parse_args()

    if len(args) < 1:
        parser.error("no script name supplied.")

    sys.path.insert(0, os.path.dirname(args[0]))
    run('execfile%s' % repr(tuple(args)))
        
