#
# NPI - Calculatrice en Notation Polonaise Inverse. 
# Copyright (C) 2005-2011 MiKael NAVARRO
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

"""NPI - Reverse Polish Notation Calculator. 
Copyright (C) 2005-2011 MiKael NAVARRO

Trigonometric commands.
"""

# Include directives
from npi_errors import TooFewArguments  # NPI errors
from npi_utils import func_name  # decorator: set func_name
from math import sin, cos, tan, asin, acos, atan, atan2


#
# Sin.
#
@func_name("sin") 
def npi_sin(stack):
  """Sine X (radian)

  >>> from stack import Stack
  >>> npi_sin(Stack([3.14]))  # doctest:+ELLIPSIS
  [0.0...]
  >>> npi_sin(Stack([]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: sin Error: Too Few Arguments
  """
  
  if len(stack) >= 1:
    stack.push(sin(stack.pop()))
  else:
    raise TooFewArguments(npi_sin)

  return stack


#
# Cos.
#
@func_name("cos") 
def npi_cos(stack):
  """Cosine X (radian)

  >>> from stack import Stack
  >>> npi_cos(Stack([3.14]))  # doctest:+ELLIPSIS
  [-0.99...]
  >>> npi_cos(Stack([]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: cos Error: Too Few Arguments
  """
  
  if len(stack) >= 1:
    stack.push(cos(stack.pop()))
  else:
    raise TooFewArguments(npi_cos)

  return stack


#
# Tan.
#
@func_name("tan") 
def npi_tan(stack):
  """Tangent X (radian)

  >>> from stack import Stack
  >>> npi_tan(Stack([3.14]))  # doctest:+ELLIPSIS
  [-0.0...]
  >>> npi_tan(Stack([]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: tan Error: Too Few Arguments
  """
  
  if len(stack) >= 1:
    stack.push(tan(stack.pop()))
  else:
    raise TooFewArguments(npi_tan)

  return stack


#
# Asin.
#
@func_name("asin") 
def npi_asin(stack):
  """Asine X (radian)

  >>> from stack import Stack
  >>> npi_asin(Stack([1]))  # doctest:+ELLIPSIS
  [1.57...]
  >>> npi_asin(Stack([]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: asin Error: Too Few Arguments
  """
  
  if len(stack) >= 1:
    stack.push(asin(stack.pop()))
  else:
    raise TooFewArguments(npi_asin)

  return stack


#
# Acos.
#
@func_name("acos") 
def npi_acos(stack):
  """Acosine X (radian)

  >>> from stack import Stack
  >>> npi_acos(Stack([0]))  # doctest:+ELLIPSIS
  [1.57...]
  >>> npi_acos(Stack([]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: acos Error: Too Few Arguments
  """
  
  if len(stack) >= 1:
    stack.push(acos(stack.pop()))
  else:
    raise TooFewArguments(npi_acos)

  return stack


#
# Atan.
#
@func_name("atan") 
def npi_atan(stack):
  """Atangent X (radian)

  >>> from stack import Stack
  >>> npi_atan(Stack([1]))  # doctest:+ELLIPSIS
  [0.78...]
  >>> npi_atan(Stack([]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: atan Error: Too Few Arguments
  """
  
  if len(stack) >= 1:
    stack.push(atan(stack.pop()))
  else:
    raise TooFewArguments(npi_atan)

  return stack


#
# Atan2 (= atan(Y/X)).
#
@func_name("atan2") 
def npi_atan2(stack):
  """Atangent Y/X (radian)

  >>> from stack import Stack
  >>> npi_atan2(Stack([2, 1]))  # doctest:+ELLIPSIS
  [1.10...]
  >>> npi_atan2(Stack([]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: atan2 Error: Too Few Arguments
  """
  
  if len(stack) >= 2:
    op2 = stack.pop()
    stack.push(atan2(stack.pop(), op2))
  else:
    raise TooFewArguments(npi_atan2)

  return stack


#
# Self-test.
#
def _test():
  import doctest
  print("Testing 'Trigonometric' commands...", end="")
  (failure_count, test_count) = doctest.testmod()
  if not failure_count: print("Okey")
  else: print("Ko!")


#
# External entry point.
#
if __name__ == "__main__":
  _test()
