#
# NPI - Calculatrice en Notation Polonaise Inverse. 
# Copyright (C) 2005-2012 MiKael NAVARRO
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#

"""NPI - Reverse Polish Notation Calculator. 
Copyright (C) 2005-2012 MiKael NAVARRO

Shadowrun RPG utilities.
"""

# Include directives
from npi_errors import TooFewArguments, BadArgumentValue  # NPI errors
from npi_utils import func_name  # decorator: set func_name
from random import randint


#
# Roll 1d6.
#
def _d6():
  """Roll 1d6"""
  
  return randint(1, 6)

@func_name("d6")
def npi_d6(stack):
  """Roll 1d6
    
  >>> from stack import Stack
  >>> npi_d6(Stack([]))  # doctest:+ELLIPSIS
  [...]
  """

  stack.push(_d6())
  
  return stack


#
# Roll nd6.
#
@func_name("nd6")
def npi_nd6(stack):
  """Roll nd6
  
  >>> from stack import Stack
  >>> npi_nd6(Stack([3]))  # doctest:+ELLIPSIS
  [...]
  >>> npi_nd6(Stack([]))
  Traceback (most recent call last):
  ...
  npi_errors.TooFewArguments: nd6 Error: Too Few Arguments
  """
    
  if len(stack) >= 1:
    for i in range(stack.pop()):
      stack.push(_d6())
  else:
    raise TooFewArguments(npi_nd6)
      
  return stack


#
# Shadowrun version specific functions.
#
SR_VERSION=4


if SR_VERSION <= 3:

  #
  # Roll 1d6 (with rule of six).
  #
  def _d6p():
    """Roll 1d6 (with rule of six)"""
    
    r = 0
    while (r % 6) == 0:
      r += randint(1, 6)

    return r

  @func_name("d6p")
  def npi_d6p(stack):
    """Roll 1d6 (with rule of six)
    
    >>> from stack import Stack
    >>> npi_d6p(Stack([]))  # doctest:+ELLIPSIS
    [...]
    """

    stack.push(_d6p())
    
    return stack


  #
  # Roll nd6 (with rule of six).
  #
  @func_name("nd6p")
  def npi_nd6p(stack):
    """Roll nd6 (with rule of six)
    
    >>> from stack import Stack
    >>> npi_nd6p(Stack([3]))  # doctest:+ELLIPSIS
    [...]
    >>> npi_nd6p(Stack([]))
    Traceback (most recent call last):
    ...
    npi_errors.TooFewArguments: nd6p Error: Too Few Arguments
    """
    
    if len(stack) >= 1:
      for i in range(stack.pop()):
        stack.push(_d6p())
    else:
      raise TooFewArguments(npi_nd6p)
      
    return stack


  #
  # Hits nd6>=sr (with rule of six).
  #
  @func_name("nd6srp")
  def npi_nd6srp(stack):
    """Hits nd6>=sr (with rule of six)
    
    >>> from stack import Stack
    >>> npi_nd6srp(Stack([10, 8]))  # doctest:+ELLIPSIS
    [...]
    >>> npi_nd6srp(Stack([]))
    Traceback (most recent call last):
    ...
    npi_errors.TooFewArguments: nd6srp Error: Too Few Arguments
    """

    if len(stack) >= 2:
      sr, n = stack.pop(), stack.pop()

      # Roll n dice
      res = [ _d6p() for i in range(n) ]
      if __debug__: print(res)

      # Count success
      hits = len( [ r for r in res if r >= sr ]  )
      
      stack.push(hits)
    else:
      raise TooFewArguments(npi_nd6sr)

    return stack


elif SR_VERSION == 4:

  #
  # Roll 1d6 (with edge).
  #
  def _d6e():
    """Roll 1d6 (with edge)"""
    
    res = [ randint(1,6), ]
    r6 = [ 6 for r in res if r == 6 ]
    while r6:
      tr = [ randint(1, 6) for i in r6 ]
      r6 = [ 6 for r in tr if r == 6 ]
      res.extend(tr)

    #if __debug__: print(res)
    return res

  @func_name("d6e")
  def npi_d6e(stack):
    """Roll 1d6 (with edge)
    
    >>> from stack import Stack
    >>> npi_d6e(Stack([]))  # doctest:+ELLIPSIS
    [...]
    """

    for r in _d6e():
      stack.push(r)
    
    return stack


  #
  # Roll nd6 (with edge).
  #
  @func_name("nd6e")
  def npi_nd6e(stack):
    """Roll nd6 (with rule of six)
    
    >>> from stack import Stack
    >>> npi_nd6e(Stack([3]))  # doctest:+ELLIPSIS
    [...]
    >>> npi_nd6e(Stack([]))
    Traceback (most recent call last):
    ...
    npi_errors.TooFewArguments: nd6e Error: Too Few Arguments
    """
    
    if len(stack) >= 1:
      for i in range(stack.pop()):
        for r in _d6e():
          stack.push(r)
    else:
      raise TooFewArguments(npi_nd6e)
      
    return stack


  #
  # Hits nd6>=5.
  #
  @func_name("nd6sr")
  def npi_nd6sr(stack):
    """Hits nd6>=5
    
    >>> from stack import Stack
    >>> npi_nd6sr(Stack([3]))  # doctest:+ELLIPSIS
    [...]
    >>> npi_nd6sr(Stack([]))
    Traceback (most recent call last):
    ...
    npi_errors.TooFewArguments: nd6sr Error: Too Few Arguments
    """

    if len(stack) >= 1:
      n = stack.pop()

      # Roll n dice
      res = [ _d6() for i in range(n) ]
      if __debug__: print(res)

      # Count success
      hits = len( [ r for r in res if r >= 5 ] )

      stack.push(hits)
    else:
      raise TooFewArguments(npi_nd6sr)

    return stack

  
  #
  # Hits nd6>=5 (with edge).
  #
  @func_name("nd6sre")
  def npi_nd6sre(stack):
    """Hits nd6>=5 (with edge)
    
    >>> from stack import Stack
    >>> npi_nd6sre(Stack([3]))  # doctest:+ELLIPSIS
    [...]
    >>> npi_nd6sre(Stack([]))
    Traceback (most recent call last):
    ...
    npi_errors.TooFewArguments: nd6sre Error: Too Few Arguments
    """

    if len(stack) >= 1:
      n = stack.pop()

      # Roll n dice
      res = [ _d6e() for i in range(n) ]
      res = sum(res, [])  # flatten list
      if __debug__: print(res)

      # Count success
      hits = len( [ r for r in res if r >= 5 ] )

      stack.push(hits)
    else:
      raise TooFewArguments(npi_nd6sre)

    return stack


#
# Self-test.
#
def _test():
  import doctest
  print("Testing 'SR' utilities...", end="")
  (failure_count, test_count) = doctest.testmod()
  if not failure_count: print("Okey")
  else: print("Ko!")


#
# External entry point.
#
if __name__ == "__main__":
  _test()
