#!/usr/bin/env python2.4
# -*- coding: iso-8859-1 -*-

#     This file is part of Pygrep. 
#     Please, read the "Copyright" and "LICENSE" variables for copyright advisement
#   (just below this line):

COPYRIGHT = "Copyright 2006 Miguel Angel Garcia Martinez <miguelangel.garcia@gmail.com>"

LICENSE = """
  Pygrep is free software; you can redistribute it and/or modify \
it under the terms of the GNU General Public License as published by \
the Free Software Foundation; either version 2 of the License, or \
(at your option) any later version.

  Pygrep is distributed in the hope that it will be useful, \
but WITHOUT ANY WARRANTY; without even the implied warranty of \
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the \
GNU General Public License for more details.

  You should have received a copy of the GNU General Public License \
along with Pygrep (maybe in file "COPYING"); if not, write to the Free Software \
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
"""

VERSION = "1.0.0"
###################################################################################

import gtk, gtk.glade
import sys, os, re, fnmatch
import threading
import gobject
import ConfigParser
import string



COL_CNT  = 1
COL_FILE = 0
COL_LINE = 1
COL_TEXT = 2
COL_MARKED_TEXT = 3

class Buffer:
    """ This class allows to use buffers of things, implemented as queues: First in, fist out """
    def __init__(self, n):
        self.limit = n
        self.data = []

    def add (self, obj):
        self.data.append(obj)
        if len(self.data) > self.limit:
            self.data.remove (0)

    def __str__(self):
        return str(self.data)

class Paths:
    """ Some generic paths useful for the application
    """
    @staticmethod
    def app_dir():
        return os.path.dirname(__file__)
    
    @staticmethod
    def locale_dir(*args):
        return os.path.join(Paths.app_dir(),"po", *args)
    
    @staticmethod
    def help_dir(*args):
        return os.path.join(Paths.app_dir(),"help", *args)
    
    @staticmethod
    def share_dir(*args):
        return os.path.join(Paths.app_dir(), *args)

    @staticmethod
    def glade_dir(*args):
        return os.path.join(Paths.app_dir(), "glade2", *args)

    @staticmethod
    def config_dir(*args):
        return os.path.expanduser(os.path.join('~', '.pygrep', *args))

class Utilities:
    @staticmethod
    def model_column2list(model, column):
        """
        Gets a model column data as a list
        """
        retval = []
        model.foreach(
            lambda m, p, i:
            retval.append(m.get_value(i, column))
            )
        return retval

    @staticmethod
    def file_regex2pyregex(regex):
        """
        Translates a file regular expresion (*,?) into a python regular expresion (.*,.).
        """
        if not regex:
            return None
        
        pat = "^"

        for i in regex:
            if pat != "^":
                pat += "|"
            pat += i + "$"

        pat = pat.replace(".", "\\.")
        pat = pat.replace("?", ".")
        pat = pat.replace("*", ".*")
        return pat


class Persistence:
    """ Saves and loads a Pygrep object """
    def __init__(self, pygrep):
        self.pygrep = pygrep
        
    def save_options(self):
        config = ConfigParser.ConfigParser()
        config.add_section ("metainfo")
        config.set("metainfo", "program", "Pygrep")
        config.set("metainfo", "version", VERSION)
        config.set("metainfo", "repository", "$Id: pygrep.py,v 1.14 2006/10/17 11:55:50 magmax Exp $")
        
        config.add_section ("directories")
        config.set("directories", "dirs", string.join(Utilities.model_column2list(self.pygrep.model_dirs, 0), ","))
        config.set("directories", "include", string.join(Utilities.model_column2list(self.pygrep.model_inclusion, 0), ","))
        config.set("directories", "exclude", string.join(Utilities.model_column2list(self.pygrep.model_exclusion, 0), ","))

        config.add_section ("miscelanea")
        entry = self.pygrep.xml.get_widget("entry_command")
        config.set("miscelanea", "command", entry.get_text())
        entry = self.pygrep.xml.get_widget("entry_size_limit")
        config.set("miscelanea", "file_size_limit", entry.get_text())
        check = self.pygrep.xml.get_widget("checkbutton_searchRE")
        config.set("miscelanea", "search_RE", check.get_active())
        check = self.pygrep.xml.get_widget("checkbutton_ignorecase")
        config.set("miscelanea", "ignore_case", check.get_active())
        check = self.pygrep.xml.get_widget("checkbutton_by_word")
        config.set("miscelanea", "search_by_word", check.get_active())

        filename = Paths.config_dir('options.cfg')
        fd = None
        try:
            fd = open(filename, 'w+')
        except:
            pass
        if not fd:
            try:
                os.mkdir (Paths.config_dir())
                fd = open(filename, 'w+')
            except:
                print "ERROR: configuration won't be saved"
                return
        config.write(fd)
        fd.close()

    def __load_dirs(self, items):
        for name, val in items:
            if name == "dirs":
                for d in val.split(','):
                    if d: self.pygrep.model_dirs.append([d])
            elif name == "include":
                for i in val.split(','):
                    if i: self.pygrep.model_inclusion.append([i])
            elif name == "exclude":
                for e in val.split(','):
                    if e: self.pygrep.model_exclusion.append([e])

    def __load_miscelanea(self, items):
        for name, val in items:
            if name == "command":
                entry = self.pygrep.xml.get_widget("entry_command")
                if val: entry.set_text(val)
            elif name == "file_size_limit":
                entry = self.pygrep.xml.get_widget("entry_size_limit")
                if val: entry.set_text(val)
            elif name == "search_RE":
                check = self.pygrep.xml.get_widget("checkbutton_searchRE")
                check.set_active(val == "True")
            elif name == "ignore_case":
                check = self.pygrep.xml.get_widget("checkbutton_ignorecase")
                check.set_active(val == "True")
            elif name == "search_by_word":
                check = self.pygrep.xml.get_widget("checkbutton_by_word")
                check.set_active(val == "True")
                
        
    def load_options(self):
        config = ConfigParser.ConfigParser()
        config.read (['defaults.cfg', 'options.cfg', Paths.config_dir('options.cfg')])
        #directories
        if config.has_section("directories"):
            self.__load_dirs(config.items("directories", True))
        #miscelanea
        if config.has_section("miscelanea"):
            self.__load_miscelanea(config.items("miscelanea"))


class Search(threading.Thread):
    """ Generic class to allow the file search. This is the engine of pygrep. """
    IGNORECASE = 1
    BYWORD = 2
    REGEXP = 4
    def __init__(self):
        super(Search, self).__init__()
        self.search_exp = None
        self.file_callback = self.print_file
        self.line_callback = self.print_line
        self.end_callback = self.print_end
        self.dir = "."
        self.end = False
        self.include = []
        self.exclude = []
        self.size_limit = 0
        self.flags = 0
        self.comp_exp = None
        
    def __del__(self):
        self.end_callback()
        
    def __compile(self):
        if self.comp_exp or not self.flags & self.REGEXP:
            return
        
        if self.flags & self.BYWORD:
            self.search_exp = "\b%s\b"%self.search_exp
        if self.flags & self.IGNORECASE:
            self.search_exp = "(?i)" + self.search_exp
        self.comp_exp = re.compile(self.search_exp)

    def print_file(self, filename, cnt, search_obj=None):
        """ default file_callback: prints the information about matched files. """
        print "%s:%s"%(filename, cnt)

    def print_line(self, filename, n, line, search_obj=None):
        """ default line_callback: prints the information about matched lines. """
        print "%s:%s:%s"%(filename, n, line)

    def print_end(self):
        """ default end_callback: prints at the end of the search"""
        print "ended"

    def __must_be_visited (self, pathname):
        """ checks if a pathname must be visited or not. """

        is_file = os.path.isfile(pathname)

        if self.size_limit > 0 and is_file:
            stat = os.stat(pathname)
            if stat.st_size > self.size_limit:
                return False

        for epat in self.exclude:
            if fnmatch.fnmatchcase(pathname, epat):
                return False

        # no patterns to include, all included; directories are not in inclusions.
        if not self.include or not is_file:
            return True

        for ipat in self.include:
            if fnmatch.fnmatchcase(pathname, ipat):
                return True

        return False


    def __process_line_as_re(self, filename, line, lineno):
        found =  re.findall(self.comp_exp, line)
        if found:
            if self.line_callback:
                self.line_callback(filename, lineno, line.rstrip(), self)
            return len(found)
        return 0

    def __process_line (self, filename, line, lineno):
        if self.flags & self.IGNORECASE:
            tr_line = line.upper()
            exp = self.search_exp.upper()
        else:
            tr_line = line
            exp = self.search_exp
        cnt = tr_line.count(exp)

        if cnt and self.line_callback:
            self.line_callback(filename, lineno, line.rstrip(), self)
        return cnt
            
    
    def process_file(self, filename):
        """ processes a file. """
        cnt = 0
        lineno = 0

        if not self.__must_be_visited(filename): return

        self.__compile()
        
        try:
            fd = open(filename)
        except:
            return

        for line in fd.readlines():
            if self.end: break
            lineno += 1
            if self.flags & self.REGEXP:
                cnt += self.__process_line_as_re(filename, line, lineno)
            else:
                cnt += self.__process_line(filename, line, lineno)

        if cnt and not self.end and self.file_callback:
            self.file_callback(filename, cnt, self)

        fd.close()
        
    def run(self):
        """ Main process. Walks over the files searching the string in EXP. """
        
        for dirpath, dirnames, filelist in os.walk(self.dir):
            removelist = []
            for d in dirnames:
                if not self.__must_be_visited(d):
                    removelist.append(d)
            for i in removelist:
                dirnames.remove(i)
            if self.end: break
            for filename in filelist:
                if self.end:break
                self.process_file(os.path.join(dirpath,filename))
        if self.end:
            self.__del__()

# MAIN
class Pygrep:
    def __init__(self):
        gtk.threads_init()
        self.xml = gtk.glade.XML(Paths.glade_dir('pygrep.glade'))
        self.xml.signal_autoconnect(self)

        self.model_dirs = None
        self.model_files = None
        self.model_matches = None

        self.__tab_search_init()
        self.__tab_dir_init()

        self.status_bar = self.xml.get_widget("statusbar1")

        self.threads = []

        self.load_options()

        self.threadsno = 0

        gtk.main()

    def notify (self, msg):
        self.status_bar.push(0, msg)
    
    def on_window1_destroy(self, object):
        self.stop_threads()
        self.destroy()



    def __add_file(self, filename, cnt, search_obj):
        self.model_files.append([filename, cnt])

    def add_file(self, filename, cnt, search_obj):
        gobject.idle_add(self.__add_file, filename, cnt, search_obj)

    @staticmethod
    def __add_marks(g=None, s=None):
        if g:
            return "<b><u>%s</u></b>"%g.group()
        if s:
            return "<b><u>%s</u></b>"%s
        return ""


    def __add_line(self, filename, lineno, text, search_obj=None):
        # url transformation
        for orig, target in [("&","&amp;"), ("<","&lt;"), (">","&gt;")]:
            text = text.replace(orig, target)

        if search_obj.flags & search_obj.REGEXP:
            self.model_matches.append([filename, lineno, re.sub(search_obj.comp_exp, self.__add_marks, text)])
        else:
            if search_obj.flags & search_obj.IGNORECASE:
                # well... I must find at the upper string and transform the original one.
                outline = text
                strup = outline.upper()
                rindex = len(outline) +1
                exp = search_obj.search_exp.upper()
                while 1:
                    rindex = strup.rfind(exp, 0, rindex)
                    if rindex == -1:
                        break
                    outline = outline[0:rindex] + \
                              self.__add_marks(s=outline[rindex:rindex+len(exp)]) + \
                              outline[len(exp) + rindex:]
            else:
                outline = text.replace(search_obj.search_exp,
                                       self.__add_marks(s=search_obj.search_exp))
            self.model_matches.append([filename, lineno, outline])



    def add_line(self, filename, lineno, line, search_obj=None):
        gobject.idle_add(self.__add_line, filename, lineno, line, search_obj)

    def endSearch(self):
        self.threadsno -= 1
        if self.threadsno == 0:
            gobject.idle_add(self.notify, "Search ended")

    def __create_search_object(self, exp, inclusion, exclusion):
        s = Search()
        s.search_exp = exp
        s.line_callback = self.add_line
        s.file_callback = self.add_file
        s.end_callback = self.endSearch
        s.include = inclusion
        s.exclude = exclusion
        entry = self.xml.get_widget("entry_size_limit")
        size_limit = entry.get_text()
        if size_limit:
            s.size_limit = int(size_limit) * 1024*1024 #Mb
        ignorecase = self.xml.get_widget("checkbutton_ignorecase")
        if ignorecase.get_active():
            s.flags = s.flags | s.IGNORECASE
        byword = self.xml.get_widget("checkbutton_by_word")
        if byword.get_active():
            s.flags = s.flags | s.BYWORD
        regexp = self.xml.get_widget("checkbutton_searchRE")
        if regexp.get_active():
            s.flags = s.flags | s.REGEXP
        return s

    def search_files(self, filelist, exp, inclusion, exclusion):
        notebook = self.xml.get_widget("notebook1")
        notebook.set_current_page(0)
        s = self.__create_search_object(exp, inclusion, exclusion)
        for filename in filelist:
            s.process_file(filename)
        
    #search algorithm
    def search_dir(self, dirname, exp, inclusion, exclusion):
        notebook = self.xml.get_widget("notebook1")
        notebook.set_current_page(0)
        s = self.__create_search_object(exp, inclusion, exclusion)
        s.dir = dirname
        self.threadsno += 1
        s.start()
        self.threads.append(s)

    def stop_threads(self):
        while len(self.threads):
            self.threads.pop().end = True

    def search(self, exp):
        self.stop_threads()
        self.notify ("Searching " + exp)
        #clean models
        self.model_matches.clear()
        self.model_files.clear()

        if len(exp)<3:
            self.notify("Too short string to search")
            return

        # Set patterns
        inclusion = None
        exclusion = None

        inclusion = Utilities.model_column2list(self.model_inclusion, COL_FILE)
        exclusion = Utilities.model_column2list(self.model_exclusion, COL_FILE)

        self.model_dirs.foreach(lambda m,p,i:
                                self.search_dir( m.get_value(i,0), exp,
                                                 inclusion, exclusion))

        self.notify ("Search Launched")
            
    # search tab
    def on_button_search_pressed (self, button):
        entry = self.xml.get_widget("entry_search_exp")
        self.search(entry.get_text())
    

    def on_button_search_again_pressed(self, button):
        entry = self.xml.get_widget("entry_search_exp")
        exp = entry.get_text()

        filelist = []

        self.model_files.foreach(lambda m,p,i:
                                 filelist.append( m.get_value(i,COL_FILE)))

        self.notify ("Searching Again " + exp)
        #clean models
        self.model_matches.clear()
        self.model_files.clear()
        #search

        inclusion = Utilities.model_column2list(self.model_inclusion, COL_FILE)
        exclusion = Utilities.model_column2list(self.model_exclusion, COL_FILE)

        self.search_files( filelist, exp,
                           inclusion, exclusion)
        self.notify ("Search Again Ended")

        
    def on_entry_search_exp_activate (self, entry):
        self.search(entry.get_text())

    def __filter_entries(self, model, iter, data=None):
        value = model.get_value(iter, COL_FILE)
        return value == self.selected_filename
        
    def __tab_search_init(self):
        #set matching files defaults
        renderer = gtk.CellRendererText()
        
        treeview = self.xml.get_widget('treeview_matched_files')
        
        self.model_files = gtk.ListStore(str, str)
        self.model_matches = gtk.ListStore(str, str, str)

        treeview.set_model(self.model_files)
        self.model_files.clear()
        
        column = gtk.TreeViewColumn("#Matches", renderer, text=COL_CNT)
        column.set_sort_column_id(1)
        treeview.append_column(column)
        column = gtk.TreeViewColumn("Filename", renderer, text=COL_FILE)
        column.set_sort_column_id(0)
        treeview.append_column(column)
        
        #set matching lines defaults
        treeview = self.xml.get_widget('treeview_matches_list')

        column = gtk.TreeViewColumn("File", renderer, text=COL_FILE)
        column.set_sort_column_id(0)
        column.set_visible(False)
        treeview.append_column(column)
        column = gtk.TreeViewColumn("Line", renderer, text=COL_LINE)
        column.set_sort_column_id(2)
        treeview.append_column(column)
        column = gtk.TreeViewColumn("Text", renderer, text=COL_TEXT)
        column.set_sort_column_id(3)
        column.set_visible(False)
        treeview.append_column(column)
        column = gtk.TreeViewColumn("Text", renderer, text=COL_MARKED_TEXT, markup=COL_TEXT)
        column.set_sort_column_id(1)
        treeview.append_column(column)

        # the model
        self.selected_filename = None
        self.treemodelfilter = self.model_matches.filter_new()
        self.treemodelfilter.set_visible_func(self.__filter_entries)

        treeview.set_model(self.treemodelfilter)


    def on_treeview_matched_files_cursor_changed(self, treeview):
        model, iter = treeview.get_selection().get_selected()
        self.selected_filename = model.get_value(iter, COL_FILE)
        self.treemodelfilter.refilter()



    def on_treeview_matches_list_cursor_changed (self, treeview):
        return
        model, iter = treeview.get_selection().get_selected()

    def __open_editor(self, filename, linenumber="1"):
        #get the command
        entry = self.xml.get_widget("entry_command")
        command = entry.get_text()

        #expand variables in command
        if "%F" in command:
            command = command.replace("%F", os.path.basename(filename))
        if "%P" in command:
            command = command.replace("%P", os.path.dirname(filename))
        if "%f" in command:
            command = command.replace("%f", filename)
        if "%l" in command:
            command = command.replace("%l", linenumber)
        self.notify("Launching " + command)

        if os.fork() == 0:
            sp_comm = command.split()
            os.execv (sp_comm[0], sp_comm) 

    def on_treeview_matched_files_row_activated(self, treeview, path, treeviewcolumn):
        model, iter = treeview.get_selection().get_selected()
        self.__open_editor(model.get_value(iter, COL_FILE))


    def on_treeview_matches_list_row_activated (self, treeview, path, treeviewcolumn):
        model, iter = treeview.get_selection().get_selected()
        self.__open_editor(model.get_value(iter, COL_FILE), model.get_value(iter, COL_LINE))

    # dir tab
    def __tab_dir_init(self):
        renderer = gtk.CellRendererText()
        treeview = self.xml.get_widget('dir_selection_tree')
        
        self.model_dirs = gtk.ListStore(str)
        treeview.set_model(self.model_dirs)

        column = gtk.TreeViewColumn("Files/directories to search", renderer, text=0)
        column.set_sort_column_id(0)
        treeview.append_column(column)

        # inclusions
        treeview = self.xml.get_widget('treeview_inclusion_patterns')
        self.model_inclusion = gtk.ListStore(str)
        treeview.set_model(self.model_inclusion)

        column = gtk.TreeViewColumn("Patterns to search in", renderer, text=0)
        column.set_sort_column_id(0)
        treeview.append_column(column)

        # inclusions
        treeview = self.xml.get_widget('treeview_exclusion_patterns')
        self.model_exclusion = gtk.ListStore(str)
        treeview.set_model(self.model_exclusion)

        column = gtk.TreeViewColumn("Patterns to ignore", renderer, text=0)
        column.set_sort_column_id(0)
        treeview.append_column(column)


        # exclusions

    def __sel_dir (self, button):
        entry = self.xml.get_widget("entry_directory")
        entry.set_text(self.filew.get_filename())
        self.filew.destroy()
        self.filew = None
        
    def on_button_open_file_clicked (self, button):
        self.filew = gtk.FileSelection("File/Dir selection")

        # Connect the ok_button to file_ok_sel method
        self.filew.ok_button.connect("clicked", self.__sel_dir)
    
        # Connect the cancel_button to destroy the widget
        self.filew.cancel_button.connect("clicked",
                                    lambda w: self.filew.destroy())

        self.filew.show()

    def on_button_add_dir_clicked (self, button):
        entry = self.xml.get_widget("entry_directory")
        self.model_dirs.append([entry.get_text()])

    def on_button_del_dir_clicked (self, button):
        treeview = self.xml.get_widget("dir_selection_tree")
        treeview.get_selection().selected_foreach(lambda model,path,iter: model.remove(iter))

    def on_button_inclusion_add_pressed (self, button):
        entry = self.xml.get_widget("entry_inclusion")
        self.model_inclusion.append([entry.get_text()])


    def on_button_inclusion_del_pressed (self, button):
        treeview = self.xml.get_widget("treeview_inclusion_patterns")
        treeview.get_selection().selected_foreach(lambda model,path,iter: model.remove(iter))

    def on_button_exclusion_add_pressed (self, button):
        entry = self.xml.get_widget("entry_exclusion")
        self.model_exclusion.append([entry.get_text()])


    def on_button_exclusion_del_pressed (self, button):
        treeview = self.xml.get_widget("treeview_exclusion_patterns")
        treeview.get_selection().selected_foreach(lambda model,path,iter: model.remove(iter))

    def on_entry_size_limit_changed(self, entry):
        text = entry.get_text()
        res = ""
        for i in text:
            if i in "0123456789":
                res += i
        if text != res:
            entry.set_text(res)
        
    def on_entry_size_limit_insert_text(self, entry, text, length, args):
        for i in text:
            if not text in range(10):
                return False

    #about tab
    def on_notebook1_switch_page(self, notebook, gpointer, tab_number):
        NOTIFIES = ["Searchs Pannel", "Directories Selection", "Option Switcher", "About Dialog"]
        if tab_number == 3:
            about = self.xml.get_widget("aboutdialog1")
            about.set_license(LICENSE)
            about.set_copyright(COPYRIGHT)
            about.set_name("Pygrep " + VERSION)
            about.show()
        self.notify (NOTIFIES[tab_number])

    def destroy(self):
        self.stop_threads()
        self.save_options()
        gtk.main_quit()

    def __del__(self):
        self.destroy()
        
    #persistence
    def save_options(self):
        p = Persistence(self)
        p.save_options()
        del(p)
        
    def load_options(self):
        p = Persistence(self)
        p.load_options()
        del(p)
if __name__ == "__main__":
    Pygrep()
