
"""Polygon meshes.

This module defines the PolySurface class, which can be used to describe
discrete geometrical models consisting of polygons.
"""
import numpy as np

from pyformex import utils
from pyformex import arraytools as at

from pyformex.coords import Coords
from pyformex.formex import Formex
from pyformex.varray import Varray
from pyformex.geometry import Geometry


__all__ = ['PolySurface']

##############################################################


class PolySurface(Geometry):
    """A PolySurface is a discrete geometrical model consisting of polygons.

    The PolySurface class is implemented in a similar manner as the
    :class:`Mesh` and :class`TriSurface` classes: the coordinates of
    all the vertices are collected in a single :class:`Coords` array,
    and the 'elements' (polygons) are defined using indices into that
    array. While the :class:`Mesh` and :class`TriSurface` classes store
    the elements in an :class:`Elems` object (requiring a fixed plexitude
    for all elements), the PolySurface class uses a :class:`Varray` so that
    the polygons can have a variable number of vertices.

    Parameters
    ----------
    coords: :class:`~coords.Coords` or other object.
        Usually, a 2-dim Coords object holding the coordinates of all the
        nodes used in the Mesh geometry.
        See details below for different initialization methods.
    elems: :class:`~connectivity.Connectivity` (nelems,nplex)
        A Connectivity object, defining the elements of the geometry
        by indices into the ``coords`` Coords array. All values in elems
        should be in the range 0 <= value < ncoords.
    prop: int :term:`array_like`, optional
        1-dim int array with non-negative element property numbers.
        If provided, :meth:`setProp` will be called to assign the
        specified properties.
    eltype: str or :class:`~elements.ElementType`, optional
        The element type of the geometric entities (elements).
        This is only needed if the element type has not yet been
        set in the ``elems`` Connectivity. See below.


    A Mesh object can be initialized in many different ways, depending on
    the values passed for the ``coords`` and ``elems`` arguments.

    - Coords, Connectivity: This is the most obvious case:
      ``coords`` is a 2-dim :class:`~coords.Coords` object holding
      the coordinates of all the nodes in the Mesh,
      and ``elems`` is a :class:`~connectivity.Connectivity` object
      describing the geometric elements by indices into the ``coords``.

    - Coords, : If A Coords is passed as first argument, but no ``elems``,
      the result is a Mesh of points, with plexitude 1. The Connectivity
      will be constructed automatically.

    - object with ``toMesh``, : As a convenience, if another object is
      provided that has a ``toMesh`` method and ``elems`` is not provided,
      the result of the ``toMesh`` method will be used to initialize
      both ``coords`` and ``elems``.

    - None: If neither ``coords`` nor ``elems`` are specified, but ``eltype``
      is, a unit sized single element Mesh of the specified
      :class:`~elements.ElementType` is created.

    - Specifying no parameters at all creates an empty Mesh, without any data.


    Setting the element type can also be done in different ways. If ``elems``
    is a Connectivity, it will normally already have a element type.
    If not, it can be done by passing it in the ``eltype`` parameter.
    In case you pass a simple array or list in the ``elems`` parameter,
    an element type is required.
    Finally, the user can specify an eltype to override the one in the
    Connectivity. It should however match the plexitude of the connectivity
    data.

    ``eltype`` should be one of the :class:`~elements.ElementType`
    instances or the name of such an instance.
    If required but not provided, the pyFormex default is used, which is
    based on the plexitude: 1 = point, 2 = line segment,
    3 = triangle, 4 or more is a polygon.


    A properly initialized Mesh has the following attributes:


    Attributes
    ----------
    coords: :class:`~coords.Coords` (ncoords,3)
        A 2-dim Coords object holding the coordinates of all the nodes used
        to describe the Mesh geometry.
    elems: :class:`~connectivity.Connectivity` (nelems,nplex)
        A Connectivity object, defining the elements of the geometry
        by indices into the :attr:`coords` Coords array. All values in elems
        should be in the range ``0 <= value < ncoords``.

        The Connectivity also stores the element type of the Mesh.
    prop: int array, optional
        Element property numbers. See :attr:`geometry.Geometry.prop`.
    attrib: :class:`~attributes.Attributes`
        An Attributes object. See :attr:`geometry.Geometry.attrib`.
    fields: OrderedDict
        The Fields defined on the Mesh. See :attr:`geometry.Geometry.fields`.

    Note
    ----
    The `coords`` attribute of a Mesh can hold points that are not used
    or needed to describe the Geometry. They do not influence the result
    of Mesh operations, but only use up some memory. If their number becomes
    large, you may want to free up that memory by calling the
    :meth:`compact` method. Also, before exporting a Mesh (e.g. to a
    numerical simulation program), you may want to compact the Mesh first.

    Examples
    --------
    Create a Mesh with four points and two triangle elements of type 'tri3'.

    >>> coords = Coords('0123')
    >>> elems = [[0,1,2], [0,2,3]]
    >>> M = Mesh(coords,elems,eltype='tri3')
    >>> print(M.report())
    Mesh: nnodes: 4, nelems: 2, nplex: 3, level: 2, eltype: tri3
      BBox: [0.  0.  0.], [1.  1.  0.]
      Size: [1.  1.  0.]
      Area: 1.0
      Coords: [[0.  0.  0.]
               [1.  0.  0.]
               [1.  1.  0.]
               [0.  1.  0.]]
      Elems: [[0 1 2]
              [0 2 3]]
    >>> M.nelems(), M.ncoords(), M.nplex(), M.level(), M.elName()
    (2, 4, 3, 2, 'tri3')

    And here is a line Mesh converted from of a Formex:

    >>> M1 = Formex('l:11').toMesh()
    >>> print(M1.report())
    Mesh: nnodes: 3, nelems: 2, nplex: 2, level: 1, eltype: line2
      BBox: [0.  0.  0.], [2.  0.  0.]
      Size: [2.  0.  0.]
      Length: 2.0
      Coords: [[0.  0.  0.]
               [1.  0.  0.]
               [2.  0.  0.]]
      Elems: [[0 1]
              [1 2]]

    Indexing returns the full coordinate set of the element(s):

    >>> M1[0]
    Coords([[0.,  0.,  0.],
            [1.,  0.,  0.]])

    The Mesh class inherits from :class:`Geometry` and therefore has
    all the coordinate transform methods defined there readily
    available:

    >>> M2 = M1.rotate(90)
    >>> print(M2.coords)
    [[0.  0.  0.]
     [0.  1.  0.]
     [0.  2.  0.]]

    """
    ###################################################################
    ## DEVELOPERS: ATTENTION
    ##
    ## The Mesh class is intended to be subclassable: TriSurface is an
    ## example of a class derived from Mesh.
    ## Therefore, all methods returning a Mesh and also operating correctly
    ## on a subclass, should use self.__class__ to return the proper class.
    ## The self.__class__ initiator should be called with the 'prop' and
    ## 'eltype' arguments, using keyword arguments, because only the first
    ## two arguments ('coords', 'elems') are guaranteed.
    ## See the copy() method for an example.
    ###################################################################


    # TODO: remove this when docstring for __getitem__ has been moved above
    _special_members_ = ['__getitem__']
    _exclude_members_ = ['matchLowerEntitiesMesh', 'matchFaces']

    fieldtypes = ['node', 'elemc', 'elemn']

    def __init__(self, coords=None, elems=None, prop=None):
        """
        Initialize a new Mesh.
        """
        Geometry.__init__(self)

        if coords is None or elems is None:
            # Create an empty PolySurface object
            self.coords = Coords()
            self.elems = Varray()
            self.prop = None
            return

        if not isinstance(coords, Coords):
            coords = Coords(coords)
        if coords.ndim != 2:
            raise ValueError(
                f"\nExpected 2D coordinate array, got {coords.ndim}")
        if not isinstance(elems, Varray):
            elems = Varray(elems)
        if elems.size > 0 and (
                elems.data.max() >= coords.shape[0] or elems.data.min() < 0):
            raise ValueError(
                "\nInvalid connectivity data: "
                "some node number(s) not in coords array "
                f"(min={elems.min()}, max={elems.max()}, "
                f"ncoords={coords.shape[0]}")

        self.coords = coords
        self.elems = elems
        self.setProp(prop)


    def _set_coords(self, coords):
        """
        Replace the current coords with new ones.

        Parameters
        ----------
        coords: Coords
             A Coords with same shape as self.coords.

        Returns
        -------
        Mesh
           A Mesh (or subclass) instance with same connectivity, eltype
           and properties as the current, but with possible changes in the
           coordinates of the nodes.
        """
        if isinstance(coords, Coords) and coords.shape == self.coords.shape:
            M = self.__class__(coords, self.elems, prop=self.prop)
            M.attrib(**self.attrib)
            return M
        else:
            raise ValueError(
                f"Invalid reinitialization of {self.__class__} coords")


    @property
    def eltype(self):
        """
        Return the element type of the Mesh.

        Returns
        -------
        :class:`elements.ElementType`
            The eltype attribute of the :attr:`elems` attribute.

        Examples
        --------
        >>> M = Mesh(eltype='tri3')
        >>> M.eltype
        Tri3
        >>> M.eltype = 'line3'
        >>> M.eltype
        Line3
        >>> print(M)
        Mesh: nnodes: 3, nelems: 1, nplex: 3, level: 1, eltype: line3
        BBox: [0.  0.  0.], [1.  1.  0.]
        Size: [1.  1.  0.]
        Length: 1.0

        One cannot set an element type with nonmatching plexitude:

        >>> M.eltype = 'quad4'
        Traceback (most recent call last):
        ...
        pyformex.exceptions.InvalidElementType: Data plexitude (3) != eltype plexitude (4)
        """
        return 'polygon'


    def elName(self):
        # TODO: deprecate this in favor of self.eltype.name?
        """
        Return the element name of the Mesh.

        Returns
        -------
        str
            The name of the ElementType of the Mesh.

        See Also
        --------
        elType: returns the ElementType instance

        Examples
        --------
        >>> Formex('4:0123').toMesh().elName()
        'quad4'
        """
        return 'polygon'


    def setNormals(self, normals=None):
        """
        Set/Remove the normals of the mesh.

        Parameters
        ----------
        normals: float :term:`array_like`
            A float array of shape (ncoords,3) or (nelems,nplex,3).
            If provided, this will set these normals for use in
            rendering, overriding the automatically computed ones.
            If None, this will clear any previously set user normals.

        """
        from pyformex import geomtools as gt
        if normals is None:
            pass
        elif utils.isString(normals):
            if normals == 'auto':
                normals = gt.polygonNormals(self.coords[self.elems])
            elif normals == 'avg':
                normals = gt.polygonAvgNormals(self.coords, self.elems,
                                               atnodes=False)
        else:
            normals = at.checkArray(normals, (self.nelems(), self.nplex(), 3), 'f')
        self.normals = normals


    def __getitem__(self, i):
        # This docstring is shown in the refman because __getitem__
        # is in _special_members_; it might be better to add this docstring
        # to the Mesh docstring.
        """
        Return element i of the Mesh.

        This allows addressing element i of Mesh M as M[i].

        Parameters
        ----------
        i: :term:`index`
            The index of the element(s) to return. This can be a single
            element number, a slice, or an array with a list of numbers.

        Returns
        -------
        Coords
            A Coords with a shape (nplex, 3), or if multiple elements are
            requested, a shape (nelements, nplex, 3), holding the
            coordinates of all points of the requested elements.

        Notes
        -----
        This is normally used in an expression as ``M[i]``, which will
        return the element i. Then ``M[i][j]`` will return the coordinates
        of node j of element i.
        """
        return self.coords[self.elems[i]]


    def level(self):
        """
        Return the level of the elements in the Mesh.

        Returns
        -------
        int
            The dimensionality of the elements: 0 (point), 1(line),
            2 (surface), 3 (volume).
        """
        return 2


    def nelems(self):
        """
        Return the number of elements in the Mesh. This is the first
        dimension of the :attr:`elems` array.
        """
        return self.elems.shape[0]


    def nplex(self):
        """
        Return the plexitude of the elements in the Mesh. This is the
        second dimension of the :attr:`elems` array.
        """
        return None


    def ncoords(self):
        """Return the number of nodes in the Mesh. This is the first
        dimension of the :attr:`~mesh.Mesh.coords` array.
        """
        return self.coords.shape[0]

    nnodes = ncoords
    npoints = ncoords


    def shape(self):
        """Return the shape of the :attr:`elems` array."""
        return self.elems.shape


    def plex(self):
        """Return the plexitude of all the elements"""
        return self.elems.lengths


    # def nedges(self):
    #     """
    #     Return the number of edges.

    #     Returns
    #     -------
    #     int
    #         The number of rows that would be returned by :meth:`getEdges`,
    #         without actually constructing the edges.

    #     Notes
    #     -----
    #     This is the total number of edges for all elements.
    #     Edges shared by multiple elements are counted multiple times.
    #     """
    #     return self.nelems() * self.eltype.nedges()


    def info(self):
        """
        Return short info about the Mesh.

        Returns
        -------
        str
            A string with info about the shape of the
            :attr:`~mesh.Mesh.coords` and :attr:`elems` attributes.
        """
        return "coords" + str(self.coords.shape) + "; elems" + str(self.elems.shape)


    def report(self, full=True):
        # TODO: We need an option here to let numpy print the full tables.
        """
        Create a report on the Mesh shape and size.

        The report always contains the number of nodes, number of elements,
        plexitude, dimensionality, element type, bbox and size.
        If full==True(default), it also contains the nodal coordinate
        list and element connectivity table. Because the latter can be rather
        bulky, they can be switched off.

        Note
        ----
        NumPy normally limits the printed output. You will have to change
        numpy settings to actually print the full arrays.
        """
        bb = self.bbox()
        s = (f"{self.__class__.__name__}: "
             f"nnodes: {self.ncoords()}, nelems: {self.nelems()}, "
             f"nplex: min {self.plex().min()}, max {self.plex().max()}, "
             f"eltype: {self.elName()}"
             f"\n  BBox: {bb[0]}, {bb[1]}"
             f"\n  Size: {bb[1]-bb[0]}")
        # if self.level() == 2:
        #     s += f"\n  Area: {self.area()}"

        if full:
            s += '\n' + at.stringar("  Coords: ", self.coords) + \
                 '\n' + at.stringar("  Elems: ", self.elems)
        return s


    def __str__(self):
        """
        Format a Mesh in a string.

        This creates a detailed string representation of a Mesh,
        containing the report() and the lists of nodes and elements.
        """
        return self.report(False)


    def shallowCopy(self, prop=None):
        """
        Return a shallow copy.

        Parameters
        ----------
        prop: int :term:`array_like`, optional
            1-dim int array with non-negative element property numbers.

        Returns
        -------

            A shallow copy of the Mesh, using the same data arrays
            for ``coords`` and ``elems``. If ``prop`` was provided,
            the new Mesh can have other property numbers.
            This is a convenient method to use the same Mesh
            with different property attributes.
        """
        if prop is None:
            prop = self.prop
        return self.__class__(self.coords, self.elems, prop=prop)


    # NB: It does not make sense putting compact=True here as default,
    # since _select is normally used via select, which has compact=False
    def _select(self, selected, compact=False):
        """Return a Mesh only holding the selected elements.

        This is the low level select method. The normal user interface
        is via the :meth:`select` method.
        """
        selected = at.checkArray1D(selected)
        M = self.__class__(self.coords, self.elems[selected])
        if self.prop is not None:
            M.setProp(self.prop[selected])
        # if compact:
        #     M = M.compact()
        return M


    ##########################################
    ## Allow drawing ##


    def actor(self, **kargs):

        if self.nelems() == 0:
            return None

        from pyformex.opengl.actors import Actor
        return Actor(self, **kargs)
