#
##
##  This file is part of pyFormex 2.2  (Wed Feb 10 14:59:35 CET 2021)
##  pyFormex is a tool for generating, manipulating and transforming 3D
##  geometrical models by sequences of mathematical operations.
##  Home page: http://pyformex.org
##  Project page:  http://savannah.nongnu.org/projects/pyformex/
##  Copyright 2004-2020 (C) Benedict Verhegghe (benedict.verhegghe@ugent.be)
##  Distributed under the GNU General Public License version 3 or later.
##
##  This program is free software: you can redistribute it and/or modify
##  it under the terms of the GNU General Public License as published by
##  the Free Software Foundation, either version 3 of the License, or
##  (at your option) any later version.
##
##  This program is distributed in the hope that it will be useful,
##  but WITHOUT ANY WARRANTY; without even the implied warranty of
##  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##  GNU General Public License for more details.
##
##  You should have received a copy of the GNU General Public License
##  along with this program.  If not, see http://www.gnu.org/licenses/.
##

# This is the only pyFormex module that is imported by the main script,
# so this is the place to put startup code

"""pyFormex command line options

This module defines the pyFormex command line options.
It is placed in a separate module so that it has very limited dependencies
and can be loaded very early in the startup procedure.
This allows options to be used to influence the further startup process
and the modules that are imported.
"""

import argparse

import pyformex as pf

###########################################################################
## parsing command line options
###############################


def createParser():
    """Create a parser for the pyFormex command line.

    Returns
    -------
    :class:`argparse.ArgumentParser`
        A parser for the pyFormex command line.
    """
    parser = argparse.ArgumentParser(
        prog=pf.__prog__,
        # usage = "%(prog)s [<options>] [ [ scriptname [scriptargs] ] ...]",
        description=pf.Description,
        epilog="More info on http://pyformex.org",
        # formatter = optparse.TitledHelpFormatter(),
        add_help=False,  # Because we do not want to exit here on help!
    )
    MO = parser.add_argument  # create a shorthand notation
    MO(
        '-h',
        "--help",
        action="store_true",
        dest="help",
        default=False,
        help="Show this help message and exit",
    )
    MO(
        "--version",
        action='version',
        version=pf.fullVersion(),
        help="Show program's version number and exit",
    )
    MO(
        '-v',
        "--verbose",
        action='store',
        type=int,
        default=2,
        help="Set the verbosity level (0..5). Default is 2.",
    )
    MO(
        "--gui",
        action="store_true",
        dest="gui",
        default=None,
        help="Start the GUI "
        "(this is the default when no scriptname argument is given)",
    )
    MO(
        "--nogui",
        action="store_false",
        dest="gui",
        default=None,
        help="Do not start the GUI "
        "(this is the default when a scriptname argument is given)",
    )
    MO(
        "--nocanvas",
        action="store_false",
        dest="canvas",
        default=True,
        help="Do not add an OpenGL canvas to the GUI "
        "(use for development purposes only!)",
    )
    MO(
        "--interactive",
        action="store_true",
        dest="interactive",
        default=False,
        help="Go into interactive mode after processing the command line "
        "parameters. This is implied by the --gui option.",
    )
    MO(
        "--uselib",
        action="store_true",
        dest="uselib",
        default=None,
        help="Use the pyFormex C lib if available. This is the default.",
    )
    MO(
        "--nouselib",
        action="store_false",
        dest="uselib",
        default=None,
        help="Do not use the pyFormex C-lib.",
    )
    MO(
        "--config",
        action="store",
        dest="config",
        default=None,
        help="Use file CONFIG for settings. This file is loaded in addition "
        "to the normal configuration files and overwrites their settings. "
        "Any changes will be saved to this file.",
    )
    MO(
        "--nodefaultconfig",
        action="store_true",
        dest="nodefaultconfig",
        default=False,
        help="Skip the default site and user config files. This option can "
        "only be used in conjunction with the --config option.",
    )
    MO(
        "--redirect",
        action="store_true",
        dest="redirect",
        default=None,
        help="Redirect standard output to the message board "
        "(ignored with --nogui)",
    )
    MO(
        "--noredirect",
        action="store_false",
        dest="redirect",
        help="Do not redirect standard output to the message board.",
    )
    MO(
        "--debug",
        action="store",
        dest="debug",
        default='',
        help="Display debugging information to sys.stdout. The value is "
        "a comma-separated list of (case-insensitive) debug items. "
        "Use option --debugitems to list  them. The special value 'all' "
        "can be used to switch on all debug info.",
    )
    MO(
        "--debuglevel",
        action="store",
        dest="debuglevel",
        type=int,
        default=0,
        help="Display debugging info to sys.stdout. "
        "The value is an int with the bits of the requested debug levels set. "
        "A value of -1 switches on all debug info. "
        "If this option is used, it overrides the --debug option.",
    )
    MO(
        "--debugitems",
        action="store_true",
        dest="debugitems",
        default=False,
        help="Show all available debug items. "
        "Each of these can be used with the -debug option.",
    )
    MO(
        "--mesa",
        action="store_true",
        dest="mesa",
        default=False,
        help="Force the use of software 3D rendering through the mesa libs. "
        "The default is to use hardware accelerated rendering whenever "
        "possible. This flag can be useful when running pyFormex remotely "
        "on another host. The hardware accelerated version will not work "
        "over remote X.",
    )
    MO(
        "--dri",
        action="store_true",
        dest="dri",
        default=None,
        help="Use Direct Rendering Infrastructure. "
        "By default, direct rendering will be used if available.",
    )
    MO(
        "--nodri",
        action="store_false",
        dest="dri",
        default=None,
        help="Do not use the Direct Rendering Infrastructure. "
        "This may be used to turn off the direc rendering, e.g. to allow "
        "better capturing of images and movies.",
    )
    MO(
        "--opengl",
        action="store",
        dest="opengl",
        default='2.0',
        help="Force the use of a specific OpenGL version. "
        "The version should be specified as a string 'a.b'. The default is 2.0",
    )
    MO(
        "--shader",
        action="store",
        dest="shader",
        default='',
        help="Force the use of an alternate GPU shader for the OpenGL "
        "rendering. If the default selected shader does not work well "
        "for your hardware, you can use this option to try one of the "
        "alternate shaders. See 'pyformex --detect' for a list of the "
        "available shaders.",
    )
    MO(
        "--nomultisample",
        action="store_false",
        dest="multisample",
        default=True,
        help="Switch off the use of multisample buffers in OpenGL.",
    )
    MO(
        "--vtk",
        action="store",
        dest="vtk",
        default='default',
        help="Specify which version of python-vtk to use in vtk_itf plugin. "
        "The value can be one of: 'standard', 'light', or 'default'. "
        "'standard' is the version as distributed from python-vtk. "
        "'light' is the trimmed vtk6 version as distributed with pyFormex.",
    )
    MO(
        "--testcamera",
        action="store_true",
        dest="testcamera",
        default=False,
        help="Print camera settings whenever they change.",
    )
    MO(
        "--memtrack",
        action="store_true",
        dest="memtrack",
        default=False,
        help="Track memory for leaks. This is only for developers.",
    )
    MO(
        "--fastnurbs",
        action="store_true",
        dest="fastnurbs",
        default=False,
        help="Test C library nurbs drawing: only for developers!",
    )
    MO(
        "--pathlib",
        action="store_true",
        dest="pathlib",
        default=False,
        help="Use the Python3 pathlib library. "
        "Use only for development and testing purposes.",
    )
    MO(
        "--bindings",
        action="store",
        dest="bindings",
        default=None,
        help="Override the configuration setting for the Qt5/4 bindings. "
        "Available bindings are 'pyside2' or 'pyqt5' for Qt5 and "
        "'pyside' or 'pyqt4' for Qt4. "
        "A value 'any' may be given to let pyFormex find out which "
        "bindings are available and use one of these.",
    )
    MO(
        "--experimental",
        action="store_true",
        dest="experimental",
        default=False,
        help="Allow the pyformex/experimental modules to be loaded. "
        "Beware: use only if you know what you are doing!",
    )
    MO(
        "--listfiles",
        action="store_true",
        dest="listfiles",
        default=False,
        help="List the pyFormex Python source files and exit.",
    )
    MO(
        "--listmodules",
        action="store",
        dest="listmodules",
        default=None,
        metavar='PKG',
        nargs='*',
        help="List the Python modules in the specified pyFormex subpackage "
        "and exit. Specify 'core' to just list the modules in the pyFormex "
        "top level. Specify 'all' to list all modules. The default is to "
        "list the modules in core, lib, plugins, gui, opengl.",
    )
    MO(
        "--search",
        action="store_true",
        dest="search",
        default=False,
        help="Search the pyformex source for a specified pattern and exit. "
        "This can optionally be followed by -- followed by options for the "
        "grep command and/or '-a' to search all files in the extended search "
        "path. The final argument is the pattern to search. '-e' before the "
        "pattern will interprete this as an extended regular expression. '-l' "
        "option only lists the names of the matching files.",
    )
    MO(
        "--remove",
        action="store_true",
        dest="remove",
        default=False,
        help="Remove the pyFormex installation and exit. This option only "
        "works when pyFormex was installed from a tarball release using the "
        "supplied install procedure. If you install from a distribution "
        "package (e.g. Debian), you should use your distribution's package "
        "tools to remove pyFormex. If you run pyFormex directly from the git "
        "source, you should just remove the whole cloned source tree.",
    )
    MO(
        "--whereami",
        action="store_true",
        dest="whereami",
        default=False,
        help="Show where the pyformex package is installed and exit.",
    )
    MO(
        "--detect",
        action="store_true",
        dest="detect",
        default=False,
        help="Show detected helper software and exit.",
    )
    MO(
        "--doctest",
        action="store",
        dest="doctest",
        default=None,
        metavar='MODULE',
        nargs='*',
        help="Run the docstring tests for the specified pyFormex modules "
        "and exit. MODULE name is specified in Python syntax, relative "
        "to the pyformex package (e.g. coords, plugins.curve).",
    )
    MO(
        "--pytest",
        action="store",
        dest="pytest",
        default=None,
        metavar='MODULE',
        nargs='*',
        help="Run the pytest tests for the specified pyFormex modules "
        "and exit. MODULE name is specified in Python syntax, relative "
        "to the pyformex package (e.g. coords, plugins.curve).",
    )
    MO(
        "--docmodule",
        action="store",
        dest="docmodule",
        default=None,
        metavar='MODULE',
        nargs='*',
        help="Print the autogenerated documentation for module MODULE "
        "and exit. This is mostly useful during the generation of the "
        "pyFormex reference manual, as the produced result still needs "
        "to be run through the Sphinx documentation generator. MODULE "
        "is the name of a pyFormex module (Python syntax).",
    )
    MO(
        '-c',
        "--script",
        action="store",
        dest="script",
        default=None,
        metavar='SCRIPT',
        help="A pyFormex script to be executed at startup. It is executed "
        "before any specified script files. This is mostly used in --nogui "
        "mode, when the script to execute is very short.",
    )
    MO(
        "args",
        action="store",
        nargs='*',
        metavar='FILE',
        help="pyFormex script files to be executed on startup. The files "
        "should have a .py extension. Their contents will be executed as "
        "a pyFormex script. While mostly used with the --nogui option, "
        "this will also work in GUI mode.",
    )

    return parser


def parseOptions(args):
    """Parse command line arguments

    The arguments of the pyFormex command line can be splitted in
    options and remaining arguments. This function will split the
    options from the other arguments and store them in the variable
    pf.options for access throughout pyFormex. The remaining arguments
    are stored in pf.options.args

    Parameters
    ----------
    args: list of str
        A list of command line arguments for the pyformex command

    Returns
    -------
    bool
        True if the parsing was successful.
    """
    pf.logger.info("Parse the command line options")
    pf.parser = createParser()

    # Store the function to print the help
    pf.print_help = pf.parser.print_help

    # Parse the arguments: set options; options.args contains remaining arguments
    pf.options = pf.parser.parse_args(args)

    # Set debug level
    if pf.options.verbose >= 5:
        pf.options.debuglevel = -1
    if pf.options.debug and not pf.options.debuglevel:
        pf.options.debuglevel = pf.debugLevel(pf.options.debug.split(','))

    # Check for invalid options
    if pf.options.nodefaultconfig and not pf.options.config:
        print("\nInvalid options: --nodefaultconfig but no --config option\n"
              "Do pyformex --help for help on options.\n")
        return False

    pf.debug("Done parsing options")
    pf.debug("Options: %s" % pf.options, pf.DEBUG.ALL)
    return True


# End
