#
##
##  This file is part of pyFormex 1.0.7  (Mon Jun 17 12:20:39 CEST 2019)
##  pyFormex is a tool for generating, manipulating and transforming 3D
##  geometrical models by sequences of mathematical operations.
##  Home page: http://pyformex.org
##  Project page:  http://savannah.nongnu.org/projects/pyformex/
##  Copyright 2004-2019 (C) Benedict Verhegghe (benedict.verhegghe@ugent.be)
##  Distributed under the GNU General Public License version 3 or later.
##
##  This program is free software: you can redistribute it and/or modify
##  it under the terms of the GNU General Public License as published by
##  the Free Software Foundation, either version 3 of the License, or
##  (at your option) any later version.
##
##  This program is distributed in the hope that it will be useful,
##  but WITHOUT ANY WARRANTY; without even the implied warranty of
##  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##  GNU General Public License for more details.
##
##  You should have received a copy of the GNU General Public License
##  along with this program.  If not, see http://www.gnu.org/licenses/.
##
#
##  This program is distributed in the hope that it will be useful,
##  but WITHOUT ANY WARRANTY; without even the implied warranty of
##  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##  GNU General Public License for more details.
##
##  You should have received a copy of the GNU General Public License
##  along with this program.  If not, see http://www.gnu.org/licenses/.
##

"""
Finite element meshes in pyFormex.

This module defines the Mesh class, which can be used to describe discrete
geometrical models like those used in Finite Element models.
It also contains some useful functions to create such models.
"""
from __future__ import absolute_import, division, print_function

import numpy as np

import pyformex as pf
from pyformex import utils
from pyformex import arraytools as at

from pyformex.coords import *
from pyformex.formex import Formex
from pyformex.connectivity import Connectivity
from pyformex.elements import ElementType, Elems
from pyformex.geometry import Geometry


##############################################################

# TODO: replace Connectivity with Elems wherever needed

class Mesh(Geometry):
    """
    A Mesh is a discrete geometrical model defined by nodes and elements.

    The Mesh class is one of the two basic geometrical models in pyFormex,
    the other one being the :class:`Formex`. Both classes have a lot in
    common: they represent a collection of geometrical entities of the same
    type (e.g., lines, or triangles, ...). The geometrical entities are
    also called 'elements', and the number of elements in the Mesh is
    :meth:`nelems`. The :term:`plexitude` (the number of points in an
    element) of a Mesh is found from :meth:`nplex`. Each point has
    ``ndim=3`` coordinates. While in a :class:`Formex` all these points
    are stored in an array with shape (nelems, nplex, 3), the :class:`Mesh`
    stores the information in two arrays: the coordinates of all the points
    are gathered in a single twodimensional array with shape (ncoords,3).
    The individual geometrical elements are then described by indices into
    that array: we call that the connectivity, with shape (nelems, nplex).

    This model has some advantages over the :class:`Formex` data model:

    - a more compact storage, because coordinates of coinciding points
      require only be stored once (and we usually call the points
      :term:`node` s);
    - the single storage of coinciding points represents the notion
      of connections between elements (a :class:`Formex` to the contrary
      is always a loose collection of elements);
    - connectivity related algorithms are generally faster;
    - the connectivity info also allows easy identification of geometric
      subentities (entities of a lower :term:`level`, like the border
      lines of a surface).

    The downside is that geometry generating and replicating algorithms are
    often far more complex and possibly slower.

    In pyFormex we therefore mostly use the Formex data model when creating,
    copying and replicating geometry, but when we come to the point of needing
    connectivity related algorithms or exporting the geometry to file (and to
    other programs), a Mesh data model usually becomes more appropriate.
    A :class:`Formex can be converted into a Mesh with the :meth:`Formex.toMesh`
    method, while the :meth:`Mesh.toFormex` method performs the inverse
    conversion.


    Parameters
    ----------
    coords: :class:`~coords.Coords` or other object.
        Usually, a 2-dim Coords object holding the coordinates of all the
        nodes used in the Mesh geometry.
        See details below for different initialization methods.
    elems: :class:`~connectivity.Connectivity` (nelems,nplex)
        A Connectivity object, defining the elements of the geometry
        by indices into the ``coords`` Coords array. All values in elems
        should be in the range 0 <= value < ncoords.
    prop: int :term:`array_like`, optional
        1-dim int array with non-negative element property numbers.
        If provided, :meth:`setProp` will be called to assign the
        specified properties.
    eltype: str or :class:`~elements.ElementType`, optional
        The element type of the geometric entities (elements).
        This is only needed if the element type has not yet been
        set in the ``elems`` Connectivity. See below.


    A Mesh object can be initialized in many different ways, depending on
    the values passed for the ``coords`` and ``elems`` arguments.

    - Coords, Connectivity: This is the most obvious case:
      ``coords`` is a 2-dim :class:`~coords.Coords` object holding
      the coordinates of all the nodes in the Mesh,
      and ``elems`` is a :class:`~connectivity.Connectivity` object
      describing the geometric elements by indices into the ``coords``.

    - Coords, : If A Coords is passed as first argument, but no ``elems``,
      the result is a Mesh of points, with plexitude 1. The Connectivity
      will be constructed automatically.

    - object with ``toMesh``, : As a convenience, if another object is
      provided that has a ``toMesh`` method and ``elems`` is not provided,
      the result of the ``toMesh`` method will be used to initialize
      both ``coords`` and ``elems``.

    - None: If neither ``coords`` nor ``elems`` are specified, but ``eltype``
      is, a unit sized single element Mesh of the specified
      :class:`~elements.ElementType` is created.

    - Specifying no parameters at all creates an empty Mesh, without any data.


    Setting the element type can also be done in different ways. If ``elems``
    is a Connectivity, it will normally already have a element type.
    If not, it can be done by passing it in the ``eltype`` parameter.
    In case you pass a simple array or list in the ``elems`` parameter,
    an element type is required.
    Finally, the user can specify an eltype to override the one in the
    Connectivity. It should however match the plexitude of the connectivity
    data.

    ``eltype`` should be one of the :class:`~elements.ElementType`
    instances or the name of such an instance.
    If required but not provided, the pyFormex default is used, which is
    based on the plexitude: 1 = point, 2 = line segment,
    3 = triangle, 4 or more is a polygon.


    A properly initialized Mesh has the following attributes:


    Attributes
    ----------
    coords: :class:`~coords.Coords` (ncoords,3)
        A 2-dim Coords object holding the coordinates of all the nodes used
        to describe the Mesh geometry.
    elems: :class:`~connectivity.Connectivity` (nelems,nplex)
        A Connectivity object, defining the elements of the geometry
        by indices into the :attr:`coords` Coords array. All values in elems
        should be in the range ``0 <= value < ncoords``.

        The Connectivity also stores the element type of the Mesh.
    prop: int array, optional
        Element property numbers. See :attr:`geometry.Geometry.prop`.
    attrib: :class:`~attributes.Attributes`
        An Attributes object. See :attr:`geometry.Geometry.attrib`.
    fields: OrderedDict
        The Fields defined on the Mesh. See :attr:`geometry.Geometry.fields`.

    Note
    ----
    The `coords`` attribute of a Mesh can hold points that are not used
    or needed to describe the Geometry. They do not influence the result
    of Mesh operations, but only use up some memory. If their number becomes
    large, you may want to free up that memory by calling the
    :meth:`compact` method. Also, before exporting a Mesh (e.g. to a
    numerical simulation program), you may want to compact the Mesh first.

    Examples
    --------
    Create a Mesh with four points and two triangle elements of type 'tri3'.

    >>> coords = Coords('0123')
    >>> elems = [[0,1,2], [0,2,3]]
    >>> M = Mesh(coords,elems,eltype='tri3')
    >>> print(M.report())
    Mesh: nnodes: 4, nelems: 2, nplex: 3, level: 2, eltype: tri3
      BBox: [ 0.  0.  0.], [ 1.  1.  0.]
      Size: [ 1.  1.  0.]
      Area: 1.0
      Coords: [[ 0.  0.  0.]
               [ 1.  0.  0.]
               [ 1.  1.  0.]
               [ 0.  1.  0.]]
      Elems: [[0 1 2]
              [0 2 3]]
    >>> M.nelems(), M.ncoords(), M.nplex(), M.level(), M.elName()
    (2, 4, 3, 2, 'tri3')

    And here is a line Mesh converted from of a Formex:

    >>> M1 = Formex('l:11').toMesh()
    >>> print(M1.report())
    Mesh: nnodes: 3, nelems: 2, nplex: 2, level: 1, eltype: line2
      BBox: [ 0.  0.  0.], [ 2.  0.  0.]
      Size: [ 2.  0.  0.]
      Length: 2.0
      Coords: [[ 0.  0.  0.]
               [ 1.  0.  0.]
               [ 2.  0.  0.]]
      Elems: [[0 1]
              [1 2]]

    Indexing returns the full coordinate set of the element(s):

    >>> M1[0]
    Coords([[ 0.,  0.,  0.],
            [ 1.,  0.,  0.]])

    The Mesh class inherits from :class:`Geometry` and therefore has
    all the coordinate transform methods defined there readily
    available:

    >>> M2 = M1.rotate(90)
    >>> print(M2.coords)
    [[ 0.  0.  0.]
     [ 0.  1.  0.]
     [ 0.  2.  0.]]

    """
    ###################################################################
    ## DEVELOPERS: ATTENTION
    ##
    ## The Mesh class is intended to be subclassable: TriSurface is an
    ## example of a class derived from Mesh.
    ## Therefore, all methods returning a Mesh and also operating correctly
    ## on a subclass, should use self.__class__ to return the proper class.
    ## The self.__class__ initiator should be called with the 'prop' and
    ## 'eltype' arguments, using keyword arguments, because only the first
    ## two arguments ('coords', 'elems') are guaranteed.
    ## See the copy() method for an example.
    ###################################################################


    # TODO: remove this when docstring for __getitem__ has been moved above
    _special_members_ = ['__getitem__']
    _exclude_members_ = ['matchLowerEntitiesMesh', 'matchFaces']

    fieldtypes = ['node', 'elemc', 'elemn']


    def __init__(self, coords=None, elems=None, prop=None, eltype=None):
        """
        Initialize a new Mesh.
        """
        Geometry.__init__(self)
        self.coords = self.elems = self.prop = None
        self.ndim = -1
        self.nodes = self.edges = self.faces = self.cells = None
        self.elem_edges = self.eadj = None
        self.conn = self.econn = self.fconn = None

        if coords is None:
            if eltype is None:
                # Create an empty Mesh object
                return

            else:
                # Create unit Mesh of specified type
                el = ElementType.get(eltype)
                coords = el.vertices
                elems = el.getElement()

        if elems is None:
            # A single object was specified instead of (coords,elems) pair
            try:
                # initialize from a single object
                if isinstance(coords, Coords):
                    M = Mesh(coords, arange(coords.ncoords()))
                elif isinstance(coords, Mesh):
                    M = coords
                else:
                    M = coords.toMesh()
                coords, elems = M.coords, M.elems
            except:
                raise ValueError("No `elems` specified and the first argument can not be converted to a Mesh.")

        try:
            self.coords = Coords(coords)
            if self.coords.ndim != 2:
                raise ValueError("\nExpected 2D coordinate array, got %s" % self.coords.ndim)
            self.elems = Connectivity(elems,eltype=eltype)
            if self.elems.size > 0 and (
                self.elems.max() >= self.coords.shape[0] or
                self.elems.min() < 0):
                raise ValueError("\nInvalid connectivity data: some node number(s) not in coords array (min=%s, max=%s, ncoords=%s)" % (self.elems.min(), self.elems.max(), self.coords.shape[0]))
        except:
            raise

        self.eltype = eltype  # This sanitizes the eltype
        self.setProp(prop)


    def _set_coords(self, coords):
        """
        Replace the current coords with new ones.

        Parameters
        ----------
        coords: Coords
             A Coords with same shape as self.coords.

        Returns
        -------
        Mesh
           A Mesh (or subclass) instance with same connectivity, eltype
           and properties as the current, but with possible changes in the
           coordinates of the nodes.
        """
        if isinstance(coords, Coords) and coords.shape == self.coords.shape:
            M = self.__class__(coords, self.elems, prop=self.prop, eltype=self.elType())
            M.attrib(**self.attrib)
            return M
        else:
            raise ValueError("Invalid reinitialization of %s coords" % self.__class__)


    @property
    def eltype(self):
        """
        Return the element type of the Mesh.

        Returns
        -------
        :class:`elements.ElementType`
            The eltype attribute of the :attr:`elems` attribute.

        Examples
        --------
        >>> M = Mesh(eltype='tri3')
        >>> M.eltype
        Tri3
        >>> M.eltype = 'line3'
        >>> M.eltype
        Line3
        >>> print(M)
        Mesh: nnodes: 3, nelems: 1, nplex: 3, level: 1, eltype: line3
        BBox: [ 0.  0.  0.], [ 1.  1.  0.]
        Size: [ 1.  1.  0.]
        Length: 1.0

        One cannot set an element type with nonmatching plexitude:

        >>> M.eltype = 'quad4'
        >>> M.eltype
        'plex3'

        """
        return self.elems.eltype


    @eltype.setter
    def eltype(self,eltype):
        """
        Set the eltype from a character string.

        Parameters
        ----------
        eltype: str or :class:`~elements.ElementType`, optional
            The element type to be set in the ``elems`` Connectivity.
            It is either one of the ElementType instances defined in
            elements.py, or the name of such an instance.
            The plexitude of the ElementType should match the plexitude
            of the Mesh.

        Note
        ----
        Setting the eltype sanitizes the eltype stored in the elems attribute
        and promotes the elems attribute to Elems class.
        """
        if eltype is None and hasattr(self.elems, 'eltype'):
            eltype = self.elems.eltype
        try:
            self.elems = Elems(self.elems,eltype)
        except:
            # We need this in intermediary states of a Mesh.convert,
            # where element types are used that are not (yet) defined.
            self.elems.eltype = 'plex%d' % self.nplex()


    def setEltype(self, eltype=None):
        """
        Set the eltype from a character string.

        Parameters
        ----------
        eltype: str or :class:`~elements.ElementType`, optional
            The element type to be set in the ``elems`` Connectivity.
            It is either one of the ElementType instances defined in
            elements.py, or the name of such an instance.
            The plexitude of the ElementType should match the plexitude
            of the Mesh.

        Returns
        -------
        Mesh
            The Mesh itself with possibly changed eltype.

        Examples
        --------
        >>> Mesh(eltype='tri3').setEltype('line3').eltype
        Line3
        """
        self.eltype = eltype
        return self

    # TODO: deprecate this in favor of self.eltype
    def elType(self):
        """
        Return the element type of the Mesh.

        Returns
        -------
        :class:`~elements.ElementType`
            The ElementType of the Mesh.

        See Also
        --------
        elName: returns the name of the ElementType.

        Examples
        --------
        >>> Formex('4:0123').toMesh().elType()
        Quad4
        """
        return self.elems.eltype


    # TODO: deprecate this in favor of self.eltype.name
    def elName(self):
        """
        Return the element name of the Mesh.

        Returns
        -------
        str
            The name of the ElementType of the Mesh.

        See Also
        --------
        elType: returns the ElementType instance

        Examples
        --------
        >>> Formex('4:0123').toMesh().elName()
        'quad4'
        """
        return self.elType().name()


    def setNormals(self, normals=None):
        """
        Set/Remove the normals of the mesh.

        Parameters
        ----------
        normals: float :term:`array_like`
            A float array of shape (ncoords,3) or (nelems,nplex,3).
            If provided, this will set these normals for use in
            rendering, overriding the automatically computed ones.
            If None, this will clear any previously set user normals.

        """
        from pyformex import geomtools as gt
        if normals is None:
            pass
        elif pf.isString(normals):
            if normals == 'auto':
                normals = gt.polygonNormals(self.coords[self.elems])
            elif normals == 'avg':
                normals = gt.averageNormals(self.coords, self.elems)
        else:
            normals = checkArray(normals, (self.nelems(), self.nplex(), 3), 'f')
        self.normals = normals


    # This docstring is shown in the refman because __getitem__
    # is in _special_members_; it might be better to add this docstring
    # to the Mesh docstring.
    def __getitem__(self, i):
        """
        Return element i of the Mesh.

        This allows addressing element i of Mesh M as M[i].

        Parameters
        ----------
        i: :term:`index`
            The index of the element(s) to return. This can be a single
            element number, a slice, or an array with a list of numbers.

        Returns
        -------
        Coords
            A Coords with a shape (nplex, 3), or if multiple elements are
            requested, a shape (nelements, nplex, 3), holding the
            coordinates of all points of the requested elements.

        Notes
        -----
        This is normally used in an expression as ``M[i]``, which will
        return the element i. Then ``M[i][j]`` will return the coordinates
        of node j of element i.
        """
        return self.coords[self.elems[i]]


    def __setstate__(self, state):
        """
        Set the object from serialized state.

        This allows to read back old pyFormex Project files where the Mesh
        class did not set element type yet.
        """
        elems = state['elems']
        if 'eltype' in state:
            if state['eltype'] is not None:
                # We acknowledge this eltype, even if it is also stored
                # in elems. This makes the restore also work for older projects
                # where eltype was not in elems.
                elems.eltype = ElementType.get(state['eltype'])
            # Do not store the eltype in the Mesh anymore
            del state['eltype']
        else:
            # No eltype in Mesh
            if hasattr(elems, 'eltype'):
                # eltype in elems: leave as it is
                pass
            else:
                # Try to set elems eltype from plexitude
                try:
                    elems.eltype = ElementType.get(nplex=elems.nplex())
                except:
                    raise ValueError("I can not restore a Mesh without eltype")
        self.__dict__.update(state)


    def ndim(self):
        """
        Returns the dimensionality of the global coordinate space.
        Currently, this always returns 3.
        """
        return 3


    def level(self):
        """
        Return the level of the elements in the Mesh.

        Returns
        -------
        int
            The dimensionality of the elements: 0 (point), 1(line),
            2 (surface), 3 (volume).
        """
        return self.elType().ndim


    def nelems(self):
        """
        Return the number of elements in the Mesh. This is the first
        dimension of the :attr:`elems` array.
        """
        return self.elems.shape[0]


    def nplex(self):
        """
        Return the plexitude of the elements in the Mesh. This is the
        second dimension of the :attr:`elems` array.
        """
        return self.elems.shape[1]


    def ncoords(self):
        """
        Return the number of nodes in the Mesh. This is the first
        dimension of the :attr:`~mesh.Mesh.coords` array.
        """
        return self.coords.shape[0]

    nnodes = ncoords
    npoints = ncoords


    def shape(self):
        """
        Return the shape of the :attr:`elems` array.
        """
        return self.elems.shape


    def nedges(self):
        """
        Return the number of edges.

        Returns
        -------
        int
            The number of rows that would be returned by :meth:`getEdges`,
            without actually constructing the edges.

        Notes
        -----
        This is the total number of edges for all elements.
        Edges shared by multiple elements are counted multiple times.
        """
        try:
            return self.nelems() * self.elType().nedges()
        except:
            return 0


    def info(self):
        """
        Return short info about the Mesh.

        Returns
        -------
        str
            A string with info about the shape of the
            :attr:`~mesh.Mesh.coords` and :attr:`elems` attributes.
        """
        return "coords" + str(self.coords.shape) + "; elems" + str(self.elems.shape)


    #TODO: We need an option here to let numpy print the full tables.
    def report(self, full=True):
        """
        Create a report on the Mesh shape and size.

        The report always contains the number of nodes, number of elements,
        plexitude, dimensionality, element type, bbox and size.
        If full==True(default), it also contains the nodal coordinate
        list and element connectivity table. Because the latter can be rather
        bulky, they can be switched off.

        Note
        ----
        NumPy normally limits the printed output. You will have to change
        numpy settings to actually print the full arrays.
        """
        bb = self.bbox()
        s = """Mesh: nnodes: %s, nelems: %s, nplex: %s, level: %s, eltype: %s
  BBox: %s, %s
  Size: %s""" % (self.ncoords(), self.nelems(), self.nplex(), self.level(), self.elName(), bb[0], bb[1], bb[1]-bb[0])
        if self.level() == 1:
            s += "\n  Length: %s" % self.length()
        elif self.level() == 2:
            s += "\n  Area: %s" % self.area()
        elif self.level() == 3:
            s += "\n  Volume: %s" % self.volume()

        if full:
            s += '\n' + at.stringar("  Coords: ", self.coords) + \
                 '\n' + at.stringar("  Elems: ", self.elems)
        return s


    def __str__(self):
        """
        Format a Mesh in a string.

        This creates a detailed string representation of a Mesh,
        containing the report() and the lists of nodes and elements.
        """
        return self.report(False)


    def shallowCopy(self, prop=None):
        """
        Return a shallow copy.

        Parameters
        ----------
        prop: int :term:`array_like`, optional
            1-dim int array with non-negative element property numbers.

        Returns
        -------
        Mesh
            A shallow copy of the Mesh, using the same data arrays
            for ``coords`` and ``elems``. If ``prop`` was provided,
            the new Mesh can have other property numbers.
            This is a convenient method to use the same Mesh
            with different property attributes.
        """
        if prop is None:
            prop = self.prop
        return self.__class__(self.coords, self.elems, prop=prop)


    def toFormex(self):
        """
        Convert a Mesh to a Formex.

        Returns
        -------
        Formex
            A Formex equivalent with the calling Mesh. The Formex inherits
            the element property numbers and eltype from the Mesh.
            Drawing attributes and Fields are not transfered though.

        Examples
        --------
        >>> M = Mesh([[0,0,0],[1,0,0]],[[0,1],[1,0]],eltype='line2')
        >>> M.toFormex()
        Formex([[[ 0.,  0.,  0.],
                 [ 1.,  0.,  0.]],
        <BLANKLINE>
                [[ 1.,  0.,  0.],
                 [ 0.,  0.,  0.]]])
        """
        return Formex(self.coords[self.elems], self.prop, self.elName())


    def toMesh(self):
        """
        Convert to a Mesh.

        Returns
        -------
        Mesh
            The Mesh itself. This is provided as a convenience for use
            in functions that need to work on different Geometry types.
        """
        return self


    def toSurface(self):
        """
        Convert a Mesh to a TriSurface.

        Only Meshes of level 2 (surface) and 3 (volume) can be converted to a
        TriSurface. For a level 3 Mesh, the border Mesh is taken first.
        A level 2 Mesh is converted to element type 'tri3' and then to a
        TriSurface.

        Returns
        -------
        :class:`TriSurface`
            A TriSurface corresponding with the input Mesh. If that has
            eltype 'tri3', the resulting TriSurface is fully equivalent.
            Otherwise, a triangular approximation is returned.

        Raises
        ------
        ValueError
            If the Mesh can not be converted to a TriSurface.

        """
        from pyformex.trisurface import TriSurface
        if self.level() == 3:
            obj = self.getBorderMesh()
        elif self.level() == 2:
            obj = self
        else:
            raise ValueError("Can not convert a Mesh of level %s to a Surface" % self.level())

        obj = obj.convert('tri3')
        return TriSurface(obj)


    def toCurve(self, connect=False):
        """
        Convert a Mesh to a Curve.

        If the element type is one of 'line*' types, the Mesh is converted
        to a Curve. The type of the returned Curve is dependent on the
        element type of the Mesh:

        - 'line2': :class:`PolyLine`,
        - 'line3': :class:`BezierSpline` (degree 2),
        - 'line4': :class:`BezierSpline` (degree 3)

        If connect is False, this is equivalent to ::

          self.toFormex().toCurve()

        Any other type will raise an exception.
        """
        if self.elName() in ['line2', 'line3', 'line4']:
            if connect:
                elems = self.elems.chained()
                if len(elems)!=1:
                    # BV: We should return all connected parts
                    raise ValueError("Can not convert a Mesh to a single continuos curve")
                else:
                    elems=elems[0]
                    closed = elems[-1, -1] == elems[0, 0]
                    # BV: This should be done without conversion to Formex
                    return Mesh(self.coords, elems, eltype=self.elType()).toFormex().toCurve(closed=closed)


            else:
                closed = self.elems[-1, -1] == self.elems[0, 0]
                return self.toFormex().toCurve(closed=closed)

        else:
            raise ValueError("Can not convert a Mesh of type '%s' to a curve" % self.elName())


    def centroids(self):
        """
        Return the centroids of all elements of the Mesh.

        The centroid of an element is the point with coordinates
        equal to the average of those of all nodes of the element.

        Returns
        -------
        Coords
            A Coords object with shape (:meth:`nelems`, 3), holding the
            centroids of all the elements in the Mesh.

        Examples
        --------
        >>> rectangle(L=3,W=2,nl=3,nw=2).centroids()
        Coords([[ 0.5,  0.5,  0. ],
                [ 1.5,  0.5,  0. ],
                [ 2.5,  0.5,  0. ],
                [ 0.5,  1.5,  0. ],
                [ 1.5,  1.5,  0. ],
                [ 2.5,  1.5,  0. ]])

        """
        return self.coords[self.elems].mean(axis=1)


    def bboxes(self):
        """
        Returns the bboxes of all elements in the Mesh.

        Returns
        -------
        float array (nelems,2,3).
            An array with the minimal and maximal values of the
            coordinates of the nodes of each element, stored along
            the 1-axis.
        """
        return self.coords[self.elems].bboxes()


#######################################################################
    ## Entity selection and mesh traversal ##


    def getLowerEntities(self, level=-1, unique=False):
        """
        Get the entities of a lower dimensionality.

        Parameters
        ----------
        level: int
            The :term:`level` of the entities to return. If negative,
            it is a value relative to the level of the caller. If non-negative,
            it specifies the absolute level. Thus, for a Mesh with a 3D
            element type, getLowerEntities(-1) returns the faces, while for a
            2D element type, it returns the edges.
            For both meshes however, getLowerEntities(+1) returns the edges.
        unique: bool, optional
            If True, return only the unique entities.

        Returns
        -------
        :class:`~connectivity.Connectivity`
            A Connectivity defining the lower entities of the specified
            level in terms of the nodes of the Mesh.
            By default, all entities for all elements are returned and
            entities shared by multiple elements will appear multiple times.
            With ``unique=True`` only the unique ones are returned.

            The return value may be an empty table, if the element type does
            not have the requested entities (e.g. 'quad4' Mesh does not
            have entities of level 3).

            If the targeted entity level is outside the range 0..3, the return
            value is None.

        See Also
        --------
        level: return the dimensionality of the Mesh
        :meth:`connectivity.Connectivity.insertLevel`: returns two tables:
            elems vs. lower entities, lower enitites vs. nodes.

        Examples
        --------
        Mesh with one 'quad4' element and 4 nodes.

        >>> M = Mesh(eltype='quad4')

        The element defined in function of the nodes.

        >>> print(M.elems)
        [[0 1 2 3]]

        The edges of the element defined in function of the nodes.

        >>> print(M.getLowerEntities(-1))
        [[0 1]
         [1 2]
         [2 3]
         [3 0]]

        And finally, the nodes themselves: not very useful, but works.

        >>> print(M.getLowerEntities(-2))
        [[0]
         [1]
         [2]
         [3]]
        """
        sel = self.elType().getEntities(level)
        ent = self.elems.selectNodes(sel)
        ent.eltype = sel.eltype
        if unique:
            utils.warn("depr_mesh_getlowerentities_unique")
            ent = ent.removeDuplicate()

        return ent


    @utils.deprecated("The use of Mesh.getElems() is deprecated: use Mesh.elems instead.")
    def getElems(self):
        """Get the elems table.

        Returns
        -------
        :class:`~elements.Elems`
            The element connectivity table (the :attr:`elems` attribute).

        Notes
        -----
        This is deprecated. Use the :attr:`elems` attribute instead.
        """
        return self.elems


    def getNodes(self):
        """Return the set of unique node numbers in the Mesh.

        Returns
        -------
        int array
            The sorted node numbers that are actually used in the connectivity
            table.
            For a compacted Mesh, it is equal to ``arange(self.nelems)``.
        """
        return unique(self.elems)


    def getPoints(self):
        """Return the nodal coordinates of the Mesh.

        Returns
        -------
        :class:`~coords.Coords`
            The coordinates of the nodes that are actually used in
            the connectivity table. For a compacted Mesh, it is equal to
            the coords attribute.
        """
        return self.coords[self.getNodes()]


    def getEdges(self):
        """Return the unique edges of all the elements in the Mesh.

        Returns
        ------
        :class:`~elements.Elems`
            A connectivity table defining the unique element edges in function
            of the nodes.
            This is like ``self.getLowerEntities(1,unique=True)``, but
            the result is stored internally in the Mesh object so that
            it does not need recomputation on a next call.
        """
        if self.edges is None:
            self.edges = self.elems.insertLevel(1)[1]
        return self.edges


    def getFaces(self):
        """Return the unique faces of all the elements in the Mesh.

        Returns
        -------
        :class:`~elements.Elems`
            A connectivity table defining all the element faces in function
            of the nodes.
            This is like ``self.getLowerEntities(2,unique=True)``, but
            the result is stored internally in the Mesh object so that
            it does not need recomputation on a next call.
        """
        if self.faces is None:
            self.faces = self.elems.insertLevel(2)[1]
        return self.faces


    # TODO: Can this anyhow be different from self.elems or None?
    # Is there any purpose then on storing it?
    def getCells(self):
        """Return the cells of the elements.

        This is a convenient function to create a table with the element
        cells. It is equivalent to ``self.getLowerEntities(3,unique=True)``,
        but this also stores the result internally so that future
        requests can return it without the need for computing it again.
        """
        if self.cells is None:
            self.cells = self.elems.insertLevel(3)[1]
        return self.cells


    def edgeMesh(self):
        """Return a Mesh with the unique edges of the elements.

        This can only be used with a Mesh of level >= 1.
        """
        return Mesh(self.coords, self.getEdges())


    def faceMesh(self):
        """Return a Mesh with the unique faces of the elements.

        This can only be used with a Mesh of level >= 2.
        """
        return Mesh(self.coords, self.getFaces())


    def getElemEdges(self):
        """Defines the elements in function of its edges.

        Returns
        -------
        :class:`~elements.Elems`
            A connectivity table with the elements defined in
            function of the edges.

        Notes
        -----
        As a side effect, this also stores the definition of the edges
        and the returned element to edge connectivity in the attributes
        `edges`, resp. `elem_edges`.
        """
        if self.elem_edges is None:
            self.elem_edges, self.edges = self.elems.insertLevel(1)
        return self.elem_edges


    def getFreeEntities(self, level=-1, return_indices=False):
        """Return the free entities of the specified level.

        Parameters
        ----------
        level: int
            The :term:`level` of the entities to return. If negative,
            it is a value relative to the level of the caller. If non-negative,
            it specifies the absolute level.
        return_indices: bool
            If True, also returns an index array (nentities,2) for inverse
            lookup of the higher entity (column 0) and its local
            lower entity number (column 1).

        Returns
        -------
        :class:`~elements.Elems`
            A connectivity table with the free entities of the
            specified level of the Mesh. Free entities are entities
            that are only connected to a single element.

        See Also
        --------
        getFreeEntitiesMesh: return the free entities as a Mesh
        getBorder: return the free entities of the first lower level

        Examples
        --------
        >>> M = Formex('3:.12.34').toMesh()
        >>> print(M.report())
        Mesh: nnodes: 4, nelems: 2, nplex: 3, level: 2, eltype: tri3
          BBox: [ 0.  0.  0.], [ 1.  1.  0.]
          Size: [ 1.  1.  0.]
          Area: 1.0
          Coords: [[ 0.  0.  0.]
                   [ 1.  0.  0.]
                   [ 0.  1.  0.]
                   [ 1.  1.  0.]]
          Elems: [[0 1 3]
                  [3 2 0]]
        >>> M.getFreeEntities(1)
        Elems([[0, 1],
               [1, 3],
               [3, 2],
               [2, 0]], eltype=Line2)
        >>> M.getFreeEntities(1,True)[1]
        array([[0, 0],
               [0, 1],
               [1, 0],
               [1, 1]])

        """
        hi, lo = self.elems.insertLevel(level)
        if hi.size == 0:
            if return_indices:
                return Connectivity(), []
            else:
                return Connectivity()

        hiinv = hi.inverse()
        ncon = (hiinv>=0).sum(axis=1)
        isbrd = (ncon<=1)   # < 1 should not occur
        brd = lo[isbrd]
        if not return_indices:
            return brd

        # also return indices where the border elements come from
        binv = hiinv[isbrd]
        enr = binv[binv >= 0]  # element number
        a = hi[enr]
        b = arange(lo.shape[0])[isbrd].reshape(-1, 1)
        fnr = where(a==b)[1]   # local border part number
        return brd, column_stack([enr, fnr])


    def getFreeEntitiesMesh(self, level=-1, compact=True):
        """Return a Mesh with lower entities.

        Parameters
        ----------
        level: int
            The :term:`level` of the entities to return. If negative,
            it is a value relative to the level of the caller. If non-negative,
            it specifies the absolute level.
        compact: bool
            If True (default), the returned Mesh will be compacted. If False,
            the returned Mesh will contain all the nodes present in the
            input Mesh.

        Returns
        -------
        :class:`Mesh`
            A Mesh containing the lower entities of the specified
            level. If the Mesh has property numbers, the lower entities inherit
            the property of the element to which they belong.

        See Also
        --------
        getFreeEdgesMesh: return a Mesh with the free entities of the level 1
        getBorderMesh: return the free entities Mesh of the first lower level

        """
        if self.prop is None:
            M = Mesh(self.coords, self.getFreeEntities(level=level))

        else:
            brd, indices = self.getFreeEntities(return_indices=True, level=level)
            enr = indices[:, 0]
            M = Mesh(self.coords, brd, prop=self.prop[enr])

        if compact:
            M = M.compact()
        return M


    def getFreeEdgesMesh(self, compact=True):
        """Return a Mesh with the free edges.

        Parameters
        ----------
        compact: bool
            If True (default), the returned Mesh will be compacted. If False,
            the returned Mesh will contain all the nodes present in the
            input Mesh.

        Returns
        -------
        :class:`Mesh`
            A Mesh containing the free edges of the input Mesh.
            If the input Mesh has property numbers, the edge elements inherit
            the property of the element to which they belong.

        See Also
        --------
        getFreeEntitiesMesh: return the free entities  Mesh of any lower level
        getBorderMesh: return the free entities Mesh of level -1
        """
        return self.getFreeEntitiesMesh(level=1, compact=compact)


    def getBorder(self, return_indices=False):
        """Return the border of the Mesh.

        Border entities are the free entities of the first lower level.

        Parameters
        ----------
        return_indices: bool
            If True, also returns an index array (nentities,2) for inverse
            lookup of the higher entity (column 0) and its local
            lower entity number (column 1).

        Returns
        -------
        :class:`~elements.Elems`
            A connectivity table with the border entities of the
            specified level of the Mesh. Free entities are entities
            that are only connected to a single element.

        See Also
        --------
        getFreeEntities: return the free entities of any lower level
        getBorderMesh: return the border as a Mesh

        Notes
        -----
        This is a convenient shorthand for ::

          self.getFreeEntities(level=-1,return_indices=return_indices)
        """
        return self.getFreeEntities(level=-1, return_indices=return_indices)


    def getBorderMesh(self, compact=True):
        """Return a Mesh representing the border.

        Parameters
        ----------
        compact: bool
            If True (default), the returned Mesh will be compacted. If False,
            the returned Mesh will contain all the nodes present in the
            input Mesh.

        Returns
        -------
        :class:`Mesh`
            A Mesh containing the border of the input Mesh. The level of the
            Mesh is one less than that of the input Mesh.
            If the input Mesh has property numbers, the border elements inherit
            the property of the element to which they belong.

        Notes
        -----
        This is a convenient shorthand for ::

          self.getFreeEntitiesMesh(level=-1,compact=compact)
        """
        return self.getFreeEntitiesMesh(level=-1, compact=compact)


    borderMesh = getBorderMesh


    def getBorderElems(self):
        """Find the elements that are touching the border of the Mesh.

        Returns
        -------
        int array
            A list of the numbers of the elements that fully contain at
            least one of the elements of the border Mesh.
            Thus, in a volume Mesh, elements only touching the border
            by a vertex or an edge are not considered border elements.
        """
        brd, ind = self.getBorder(True)
        return unique(ind[:, 0])


    def getBorderNodes(self):
        """Find the nodes that are on the border of the Mesh.

        Returns
        -------
        int array
            A list of the numbers of the nodes that are on the
            border of the Mesh.
        """
        brd = self.getBorder()
        return unique(brd)


    def peel(self, nodal=False):
        """Remove the border elements from a Mesh.

        Parameters
        ----------
        nodal: bool
            If True, all elements connected to a border node are removed.
            The default will only remove the elements returned by
            :meth:`getBorderElems`.

        Returns
        -------
        Mesh
            A Mesh with the border elements removed.

        """
        if nodal:
            brd=self.connectedTo(self.getBorderNodes())
        else:
            brd=self.getBorderElems()
        return self.cselect(brd)


    # TODO: the level parameter here seems useless: how does one know
    # the indices of the lower level entities??
    @utils.warning("mesh_connectedTo")
    def connectedTo(self, entities, level=0):
        """Find the elements connected to specific lower entities.

        Parameters
        ----------
        entities: int or int :term:`array_like`.
            The indices of the lower entities to which connection should
            exist.
        level: int
            The :term:`level` of the entities to which connection should
            exist. If negative, it is a value relative to the level of the
            caller. If non-negative, it specifies the absolute level.
            Default is 0 (nodes).

        Returns
        -------
        int array
            A list of the numbers of the elements that contain at
            least one of the specified lower entities.
        """
        if level == 0:
            elems = self.elems
        else:
            elems, lo = self.elems.insertLevel(level)
        return elems.connectedTo(entities)


    @utils.deprecated("mesh_notConnectedTo")
    def notConnectedTo(self, nodes):
        pass


    def adjacentTo(self, elements, level=0):
        """Find the elements adjacent to the specified elements.

        Adjacent elements are elements that share some lower entity.

        Parameters
        ----------
        elements: int or int :term:`array_like`
            Element numbers to find the adjacent elements for.
        level: int
            The :term:`level` of the entities used to define adjacency.
            If negative, it is a value relative to the level of the
            caller. If non-negative, it specifies the absolute level.
            Default is 0 (nodes).

        Returns
        -------
        int array
            A list of the numbers of all the elements in the Mesh that are
            adjacent to any of the specified elements.

        """
        if level == 0:
            elems = self.elems
        else:
            elems = self.elems.insertLevel(level)[0]
        return np.unique(elems.adjacentElements(elements))


    def reachableFrom(self, elements, level=0):
        """Select the elements reachable from the specified elements.

        Elements are reachable if one can travel from one of the origin
        elements to the target, by only following the specified level
        of connections.

        Parameters
        ----------
        elements: int or int :term:`array_like`
            Element number(s) from where to start the walk.
        level: int
            The :term:`level` of the entities used to define connections.
            If negative, it is a value relative to the level of the
            caller. If non-negative, it specifies the absolute level.
            Default is 0 (nodes).

        Returns
        -------
        int array
            A list of the numbers of all the elements in the Mesh reachable
            from any of the specified elements by walking over entities of the
            specified level. The list will include the original set of elements.
        """
        return where(self.frontWalk(startat=elements, level=level, frontinc=0, partinc=1, maxval=1) == 0)[0]


#############################################################################
    # Adjacency #

    def adjacency(self, level=0, diflevel=-1):
        """Create an element adjacency table.

        Two elements are said to be adjacent if they share a lower
        entity of the specified level.

        Parameters
        ----------
        level: int
            Hierarchic level of the geometric items connecting two elements:
            0 = node, 1 = edge, 2 = face. Only values of a lower hierarchy than
            the level of the Mesh itself make sense. Default is to consider
            nodes as the connection between elements.
        diflevel: int, optional
            If >= level, and smaller than the level of the Mesh itself,
            elements that have a connection of this level are removed.
            Thus, in a Mesh with volume elements, self.adjacency(0,1) gives the
            adjacency of elements by a node but not by an edge.

        Returns
        -------
        adj: :class:`~adjacency.Adjacency`
            An Adjaceny table specifying for each element its neighbours
            connected by the specified geometrical subitems.
        """
        if diflevel > level:
            return self.adjacency(level).symdiff(self.adjacency(diflevel))

        if level == 0:
            elems = self.elems
        else:
            elems = self.elems.insertLevel(level)[0]
        return elems.adjacency()


    def frontWalk(self, level=0, startat=0, frontinc=1, partinc=1, maxval=-1):
        """Visit all elements using a frontal walk.

        In a frontal walk a forward step is executed simultanuously from all
        the elements in the current front. The elements thus reached become
        the new front. An element can be reached from the current element if
        both are connected by a lower entity of the specified level. Default
        level is 'point'.

        Parameters
        ----------
        level: int
            Hierarchy of the geometric items connecting two elements:
            0 = node, 1 = edge, 2 = face. Only values of a lower hierarchy than
            the elements of the Mesh itself make sense. There are no
            connections on the upper level.
        startat: int or list of ints
            Initial element number(s) in the front.
        frontinc: int
            Increment for the front number on each frontal step.
        partinc: int
            Increment for the front number when the front gets empty and
            a new part is started.
        maxval: int
            Maximum frontal value. If negative (default) the walk will
            continue until all elements have been reached. If non-negative,
            walking will stop as soon as the frontal value reaches this
            maximum.

        Returns
        -------
        int array
            An array of ints specifying for each element in which step
            the element was reached by the walker.

        See Also
        --------
        :meth:`adjacency.Adjacency.frontWalk`

        Examples
        --------
        >>> M = Mesh(eltype='quad4').subdivide(5,2)
        >>> print(M.frontWalk())
        [0 1 2 3 4 1 1 2 3 4]
        """
        return self.adjacency(level).frontWalk(startat=startat, frontinc=frontinc, partinc=partinc, maxval=maxval)


    def maskedEdgeFrontWalk(self, mask=None, startat=0, frontinc=1, partinc=1, maxval=-1):
        """Perform a front walk over masked edge connections.

        This is like frontWalk(level=1), but allows to specify a mask to
        select the edges that are used as connectors between elements.

        Parameters:

        - `mask`: Either None or a boolean array or index flagging the nodes
          which are to be considered connectors between elements. If None,
          all nodes are considered connections.

        The remainder of the parameters are like in
        :meth:`adjacency.Adjacency.frontWalk`.
        """
        if self.level() != 1:
            hi, lo = self.elems.insertLevel(1)
        else:
            hi = self.elems
        adj = hi.adjacency(mask=mask)
        return adj.frontWalk(startat=startat, frontinc=frontinc, partinc=partinc, maxval=maxval)


    # BV: DO WE NEED THE nparts ?
    def partitionByConnection(self, level=0, startat=0, sort='number', nparts=-1):
        """
        Detect the connected parts of a Mesh.

        The Mesh is partitioned in parts in which all elements are
        connected. Two elements are connected if it is possible to draw a
        continuous line from a point in one element to a point in
        the other element without leaving the Mesh.


        Parameters:

        - `sort`: str. Weighted sorting method. It can assume values
          'number' (default), 'length', 'area', 'volume'.
        - `nparts`: is the equivalent of parameter `maxval` in
          :meth:`frontWalk`.
          Maximum frontal value. If negative (default) the walk will
          continue until all elements have been reached. If non-negative,
          walking will stop as soon as the frontal value reaches this
          maximum.

        The remainder of the parameters are like in
        :meth:`frontWalk`.

        The partitioning is returned as a integer array having a value
        for each element corresponding to the part number it belongs to.

        By default the parts are sorted in decreasing order of the number
        of elements. If you specify nparts, you may wish to switch off the
        sorting by specifying sort=''.
        """
        p = self.frontWalk(level=level, startat=startat, frontinc=0, partinc=1, maxval=nparts)
        if sort=='number':
            p = sortSubsets(p)
        if sort=='length':
            p = sortSubsets(p, self.lengths())
        if sort=='area':
            p = sortSubsets(p, self.areas())
        if sort=='volume':
            p = sortSubsets(p, self.volumes())
        return p


    def splitByConnection(self, level=0, startat=0, sort='number'):
        """Split the Mesh into connected parts.

        The parameters `level` and `startat` are like in
        :meth:`frontWalk`.
        The parameter `sort` is like in
        :meth:`partitionByConnection`.

        Returns a list of Meshes that each form a connected part.
        By default the parts are sorted in decreasing order of the number
        of elements.
        """
        p = self.partitionByConnection(level=level, startat=startat, sort=sort)
        return self.splitProp(p)


    def largestByConnection(self, level=0):
        """Return the largest connected part of the Mesh.

        This is equivalent with, but more efficient than ::

          self.splitByConnection(level)[0]
        """
        p = self.partitionByConnection(level=level)
        return self.clip(p==0)


    def growSelection(self, sel, mode='node', nsteps=1):
        """Grow a selection of a surface.

        `p` is a single element number or a list of numbers.
        The return value is a list of element numbers obtained by
        growing the front `nsteps` times.
        The `mode` argument specifies how a single frontal step is done:

        - 'node' : include all elements that have a node in common,
        - 'edge' : include all elements that have an edge in common.
        """
        level = {'node': 0, 'edge': 1}[mode]
        p = self.frontWalk(level=level, startat=sel, maxval=nsteps)
        return where(p>=0)[0]


    def partitionByAngle(self, **kargs):
        """Partition a level-2 Mesh by the angle between adjacent elements.

        The Mesh is partitioned in parts bounded by the sharp edges in the
        surface. The arguments and return value are the same as in
        :meth:`trisurface.TriSurface.partitionByAngle`.

        For eltypes other than 'tri3',
        a conversion to 'tri3' is done before computing the partitions.
        """
        if self.elName() == 'tri3':
            p = self.toSurface().partitionByAngle(**kargs)
        else:
            S = self.copy().setProp(arange(self.nelems())).toSurface()
            p = S.partitionByAngle(**kargs)
            j = unique(S.prop, return_index=True)[1]
            p = p[j]
        return p


###########################################################################

    #
    #  IDEA: Should we move these up to Connectivity ?
    #        That would also avoid some possible problems
    #        with storing conn and econn
    #

    def nodeConnections(self):
        """Find and store the elems connected to nodes."""
        if self.conn is None:
            self.conn = self.elems.inverse()
        return self.conn


    def nNodeConnected(self):
        """Find the number of elems connected to nodes."""
        return (self.nodeConnections() >=0).sum(axis=-1)


    def edgeConnections(self):
        """Find and store the elems connected to edges."""
        if self.econn is None:
            self.econn = self.getElemEdges().inverse()
        return self.econn


    def nEdgeConnected(self):
        """Find the number of elems connected to edges."""
        return (self.edgeConnections() >=0).sum(axis=-1)


    #
    # Are these really needed? better use adjacency(level)
    #
    #
    def nodeAdjacency(self):
        """Find the elems adjacent to each elem via one or more nodes."""
        return self.elems.adjacency()


    def nNodeAdjacent(self):
        """Find the number of elems which are adjacent by node to each elem."""
        return (self.nodeAdjacency() >=0).sum(axis=-1)


    def edgeAdjacency(self):
        """Find the elems adjacent to elems via an edge."""
        return self.getElemEdges().adjacency()


    def nEdgeAdjacent(self):
        """Find the number of adjacent elems."""
        return (self.edgeAdjacency() >=0).sum(axis=-1)


    def nonManifoldNodes(self):
        """Return the non-manifold nodes of a Mesh.

        Non-manifold nodes are nodes where subparts of a mesh of level >= 2
        are connected by a node but not by an edge.

        Returns an integer array with a sorted list of non-manifold node
        numbers. Possibly empty (always if the dimensionality of the Mesh
        is lower than 2).
        """
        if self.level() < 2:
            return []

        ML = self.splitByConnection(1, sort='')
        nm = [intersect1d(Mi.elems, Mj.elems) for Mi, Mj in combinations(ML, 2)]
        return unique(concat(nm))


    # TODO: Explain how this is sorted
    def nonManifoldEdges(self):
        """Return the non-manifold edges of a Mesh.

        Non-manifold edges are edges where subparts of a mesh of level 3
        are connected by an edge but not by an face.

        Returns an integer array with a sorted list of non-manifold edge
        numbers. Possibly empty (always if the dimensionality of the Mesh
        is lower than 3).

        As a side effect, this constructs the list of edges in the object.
        The definition of the nonManifold edges in terms of the nodes can
        thus be got from ::

          self.edges[self.nonManifoldEdges()]
        """
        if self.level() < 3:
            return []

        elems = self.getElemEdges()
        p = self.partitionByConnection(2, sort='')
        eL = [elems[p==i] for i in unique(p)]
        nm = [intersect1d(ei, ej) for ei, ej in combinations(eL, 2)]
        return unique(concat(nm))


    def nonManifoldEdgeNodes(self):
        """Return the non-manifold edge nodes of a Mesh.

        Non-manifold edges are edges where subparts of a mesh of level 3
        are connected by an edge but not by an face.

        Returns an integer array with a sorted list of numbers of nodes
        on the non-manifold edges.
        Possibly empty (always if the dimensionality of the Mesh
        is lower than 3).
        """
        if self.level() < 3:
            return []

        ML = self.splitByConnection(2, sort='')
        nm = [intersect1d(Mi.elems, Mj.elems) for Mi, Mj in combinations(ML, 2)]
        return unique(concat(nm))


    def fuse(self, parts=None, nodes=None, **kargs):
        """Fuse the nodes of a Meshes.

        Nodes that are within the tolerance limits of each other
        are merged into a single node.

        Parameters:

        - `parts`: int :term:`array_like` with length equal to number of elements.
          If specified, it will be used to split the Mesh into parts (see
          :func:`splitProp`) and do the fuse operation per part.
          Elements for which the value of `nparts` is negative will not
          be involved in the fuse operations.
        - `nodes`: int :term:: a list of node numbers. If specified,
          only these nodes will be involved in the fuse operation. This
          option can not be used together with the `parts` option.
        - Extra arguments for tuning the fuse operation are passed to the
          :meth:`coords.Coords:fuse` method.

        """
        if parts is None:
            if nodes is None:
                coords, index = self.coords.fuse(**kargs)
            else:
                keep = at.complement(nodes, self.nnodes())
                coords, fusindex = self.coords[nodes].fuse(**kargs)
                coords = Coords.concatenate([self.coords[keep], coords])
                index = -ones(self.nnodes(), dtype=Int)
                index[keep] = arange(len(keep), dtype=Int)
                index[nodes] = len(keep) + fusindex
            return self.__class__(coords, index[self.elems], prop=self.prop, eltype=self.elType())
        else:
            parts = checkArray(parts, (self.nelems(),), 'i')
            ML = self.splitProp(parts)
            if parts.min() >= 0:
                n = (unique(parts) < 0).sum()
            else:
                n = 0
            ML = ML[:n] + [M.fuse(**kargs) for M in ML[n:]]
            return Mesh.concatenate(ML, fuse=False)


    def matchCoords(self, coords, **kargs):
        """Match nodes of coords with nodes of self.

        coords can be a Coords or a Mesh object
        This is a convenience function equivalent to ::

           self.coords.match(mesh.coords,**kargs)

        or ::

           self.coords.match(coords,**kargs)

        See also :meth:`coords.Coords.match`
        """
        if not(isinstance(coords, Coords)):
            coords=coords.coords
        return self.coords.match(coords, **kargs)


    def matchCentroids(self, mesh, **kargs):
        """Match elems of Mesh with elems of self.

        self and Mesh are same eltype meshes
        and are both without duplicates.

        Elems are matched by their centroids.
        """
        c = Mesh(self.centroids(), arange(self.nelems()))
        mc = Mesh(mesh.centroids(), arange(mesh.nelems()))
        return c.matchCoords(mc, **kargs)


    # BV: I'm not sure that we need this. Looks like it can or should
    # be replaced with a method applied on the BorderMesh
    #~ FI It has been tested on quad4-quad4, hex8-quad4, tet4-tri3
    def matchLowerEntitiesMesh(self, mesh, level=-1):
        """_Match lower entity of mesh with the lower entity of self.

        self and Mesh can be same eltype meshes or different eltype but of the
        same hierarchical type (i.e. hex8-quad4 or tet4 - tri3)
        and are both without duplicates.

        Returns the indices array of the elems of self that matches
        the lower entity of mesh, and the matched lower entity number
        """
        if level < 0:
            level = m1.elType().ndim + level

        sel = self.elType().getEntities(level)
        hi, lo = self.elems.insertLevel(sel)
        hiinv = hi.inverse()
        fm = Mesh(self.coords, lo)
        sel1 = mesh.elType().getEntities(level)
        mesh = Mesh(mesh.coords, mesh.elems.insertLevel(sel1)[1])
        c = fm.matchCentroids(mesh)
        hiinv = hiinv[c]
        hpos = findFirst(c, hi).reshape(hi.shape)
        enr =  unique(hiinv[hiinv >= 0])  # element number
        fnr=column_stack(where(hpos!=-1))  # face number
        return enr, fnr

    def matchFaces(self, mesh):
        """_Match faces of mesh with faces of self.

        self and Mesh can be same eltype meshes or different eltype but of the
        same hierarchical type (i.e. hex8-quad4 or tet4 - tri3)
        and are both without duplicates.

        eturns the indices array of the elems of self that matches
        the faces of mesh, and the matched face number
        """
        enr, fnr = self.matchLowerEntitiesMesh(mesh, level=2)
        return enr, fnr


    def compact(self, return_index=False):
        """Remove unconnected nodes and renumber the mesh.

        Returns a mesh where all nodes that are not used in any
        element have been removed, and the nodes are renumbered to
        a compacter scheme.

        If return_index is True, also returns an index specifying the
        index of the new nodes in the old node scheme.

        Examples
        --------
        >>> x = Coords([[i] for i in arange(5)])
        >>> M = Mesh(x,[[0,2],[1,4],[4,2]])
        >>> M,ind = M.compact(True)
        >>> print(M.coords)
        [[ 0. 0. 0.]
         [ 1. 0. 0.]
         [ 2. 0. 0.]
         [ 4. 0. 0.]]
        >>> print(M.elems)
        [[0 2]
         [1 3]
         [3 2]]
        >>> M = Mesh(x,[[0,2],[1,3],[3,2]])
        >>> M = M.compact()
        >>> print(M.coords)
        [[ 0. 0. 0.]
         [ 1. 0. 0.]
         [ 2. 0. 0.]
         [ 3. 0. 0.]]
        >>> print(M.elems)
        [[0 2]
         [1 3]
         [3 2]]
        >>> print(ind)
        [0 1 2 4]
        >>> M = M.cselect([0,1,2])
        >>> M.coords.shape, M.elems.shape
        ((4, 3), (0, 2))
        >>> M = M.compact()
        >>> M.coords.shape, M.elems.shape
        ((0, 3), (0, 2))

        """
        if self.nelems() == 0:
            ret = self.__class__(Coords(), self.elems)
            nodes = array([], dtype=Int)
        else:
            elems, nodes = self.elems.renumber()
            if elems is self.elems:
                # node numbering is compact
                if self.coords.shape[0] > len(nodes):
                    # remove extraneous nodes
                    self.coords = self.coords[:len(nodes)]
                # numbering has not been changed, safe to use same object
                ret = self
            else:
                # numbering has been changed, return new object
                coords = self.coords[nodes]
                ret = self.__class__(coords, elems, prop=self.prop, eltype=self.elType())

        if return_index:
            return ret, nodes
        else:
            return ret


    def _select(self, selected, compact=True):
        """Return a Mesh only holding the selected elements.

        This is the low level select method. The normal user interface
        is via the Geometry.select method.
        """
        selected = checkArray1D(selected)
        M = self.__class__(self.coords, self.elems[selected], eltype=self.elType())
        if self.prop is not None:
            M.setProp(self.prop[selected])
        if compact:
            M = M.compact()
        return M


    def avgNodes(self, nodsel, wts=None):
        """Create average nodes from the existing nodes of a mesh.

        `nodsel` is a local node selector as in :meth:`selectNodes`
        Returns the (weighted) average coordinates of the points in the
        selector as `(nelems*nnod,3)` array of coordinates, where
        nnod is the length of the node selector.
        `wts` is a 1-D array of weights to be attributed to the points.
        Its length should be equal to that of nodsel.
        """
        elems = self.elems.selectNodes(nodsel)
        return self.coords[elems].average(wts=wts, axis=1)


    # The following is equivalent to avgNodes(self,nodsel,wts=None)
    # But is probably more efficient
    def meanNodes(self, nodsel):
        """Create nodes from the existing nodes of a mesh.

        `nodsel` is a local node selector as in :meth:`selectNodes`
        Returns the mean coordinates of the points in the selector as
        `(nelems*nnod,3)` array of coordinates, where nnod is the length
        of the node selector.
        """
        elems = self.elems.selectNodes(nodsel)
        return self.coords[elems].mean(axis=1)


    def addNodes(self, newcoords, eltype=None):
        """Add new nodes to elements.

        `newcoords` is an `(nelems,nnod,3)` or`(nelems*nnod,3)` array of
        coordinates. Each element gets exactly `nnod` extra nodes from this
        array. The result is a Mesh with plexitude `self.nplex() + nnod`.
        """
        newcoords = newcoords.reshape(-1, 3)
        newnodes = arange(newcoords.shape[0]).reshape(self.elems.shape[0], -1) + self.coords.shape[0]
        elems = Connectivity(concatenate([self.elems, newnodes], axis=-1))
        coords = Coords.concatenate([self.coords, newcoords])
        return Mesh(coords, elems, self.prop, eltype)


    def addMeanNodes(self, nodsel, eltype=None):
        """Add new nodes to elements by averaging existing ones.

        `nodsel` is a local node selector as in :meth:`selectNodes`
        Returns a Mesh where the mean coordinates of the points in the
        selector are added to each element, thus increasing the plexitude
        by the length of the items in the selector.
        The new element type should be set to correct value.
        """
        newcoords = self.meanNodes(nodsel)
        return self.addNodes(newcoords, eltype)


    def selectNodes(self, nodsel, eltype=None):
        """Return a mesh with subsets of the original nodes.

        `nodsel` is an object that can be converted to a 1-dim or 2-dim
        array. Examples are a tuple of local node numbers, or a list
        of such tuples all having the same length.
        Each row of `nodsel` holds a list of local node numbers that
        should be retained in the new connectivity table.
        """
        elems = self.elems.selectNodes(nodsel)
        prop = self.prop
        if prop is not None:
            prop = column_stack([prop]*len(nodsel)).reshape(-1)
        return Mesh(self.coords, elems, prop=prop, eltype=eltype)


    @utils.deprecated_by('Mesh.withProp', 'Mesh.selectProp')
    def withProp(self, val):
        return self.selectProp(val, compact=False)


    @utils.deprecated_by('Mesh.withoutProp', 'Mesh.cselectProp')
    def withoutProp(self, val):
        return self.cselectProp(val, compact=False)


    def hits(self, entities, level):
        """Count the lower entities from a list connected to the elements.

        `entities`: a single number or a list/array of entities
        `level`: 0 or 1 or 2 if entities are nodes or edges or faces, respectively.

        The numbering of the entities corresponds to self.insertLevel(level).
        Returns an (nelems,) shaped int array with the number of the
        entities from the list that are contained in each of the elements.
        This method can be used in selector expressions like::

          self.select(self.hits(entities,level) > 0)
        """
        hi = self.elems.insertLevel(level)[0]
        return hi.hits(nodes=entities)


    def splitRandom(self, n, compact=True):
        """Split a Mesh in n parts, distributing the elements randomly.

        Returns a list of n Mesh objects, constituting together the same
        Mesh as the original. The elements are randomly distributed over
        the subMeshes.

        By default, the Meshes are compacted. Compaction may be switched
        off for efficiency reasons.
        """
        sel = random.randint(0, n, (self.nelems()))
        return [self.select(sel==i, compact=compact) for i in range(n) if i in sel]


###########################################################################
    ## simple mesh transformations ##

    def reverse(self, sel=None):
        """Return a Mesh where the elements have been reversed.

        Reversing an element has the following meaning:

        - for 1D elements: reverse the traversal direction,
        - for 2D elements: reverse the direction of the positive normal,
        - for 3D elements: reverse inside and outside directions of the
          element's border surface. This also changes the sign of the
          elementt's volume.

        The :meth:`reflect` method by default calls this method to undo
        the element reversal caused by the reflection operation.

        Parameters:

        -`sel`: a selector (index or True/False array)
        """
        utils.warn('warn_mesh_reverse')
        # TODO: These can be merged
        if sel is None:
            if hasattr(self.elType(), 'reversed'):
                elems = self.elems[:, self.elType().reversed]
            else:
                elems = self.elems[:, ::-1]
        else:
            elems = self.elems.copy()
            elsel = elems[sel]
            if hasattr(self.elType(), 'reversed'):
                elsel = elsel[:, self.elType().reversed]
            else:
                elsel = elsel[:, ::-1]
            elems[sel] = elsel
        return self.__class__(self.coords, elems, prop=self.prop, eltype=self.elType())


    def reflect(self, dir=0, pos=0.0, reverse=True, **kargs):
        """Reflect the coordinates in one of the coordinate directions.

        Parameters:

        - `dir`: int: direction of the reflection (default 0)
        - `pos`: float: offset of the mirror plane from origin (default 0.0)
        - `reverse`: boolean: if True, the :meth:`reverse` method is
          called after the reflection to undo the element reversal caused
          by the reflection of its coordinates. This will in most cases have
          the desired effect. If not however, the user can set this to False
          to skip the element reversal.
        """
        if reverse is None:
            reverse = True
            utils.warn("warn_mesh_reflect")

        M = Geometry.reflect(self, dir=dir, pos=pos)
        if reverse:
            M = M.reverse()
        return M


    def convert(self, totype, fuse=False, verbose=False):
        """Convert a Mesh to another element type.

        Converting a Mesh from one element type to another can only be
        done if both element types are of the same dimensionality.
        Thus, 3D elements can only be converted to 3D elements.

        The conversion is done by splitting the elements in smaller parts
        and/or by adding new nodes to the elements.

        Not all conversions between elements of the same dimensionality
        are possible. The possible conversion strategies are implemented
        in a table. New strategies may be added however.

        The return value is a Mesh of the requested element type, representing
        the same geometry (possibly approximatively) as the original mesh.

        If the requested conversion is not implemented, an error is raised.

        .. warning:: Conversion strategies that add new nodes may produce
          double nodes at the common border of elements. The :meth:`fuse`
          method can be used to merge such coincident nodes. Specifying
          fuse=True will also enforce the fusing. This option become the
          default in future.
        """
        #
        # totype is a string !
        #
        if verbose:
            print("Convert Mesh from %s to %s" % (self.elName(), totype))


        if totype == self.elName():
            return self

        strategy = self.elType().conversions.get(totype, None)

        while not isinstance(strategy, list):
            # This allows for aliases in the conversion database
            strategy = self.elType().conversions.get(strategy, None)

            if strategy is None:
                raise ValueError("Don't know how to convert %s -> %s" % (self.elName(), totype))

        # 'r' and 'v' steps can only be the first and only step
        steptype, stepdata = strategy[0]
        if steptype == 'r':
            # Randomly convert elements to one of the types in list
            return self.convertRandom(stepdata)
        elif steptype == 'v':
            return self.convert(stepdata).convert(totype)

        # Execute a strategy
        mesh = self
        totype = totype.split('-')[0]
        for step in strategy:
            steptype, stepdata = step

            if steptype == 'a':
                mesh = mesh.addMeanNodes(stepdata, totype)

            elif steptype == 's':
                mesh = mesh.selectNodes(stepdata, totype)

            elif steptype == 'x':
                mesh = globals()[stepdata](mesh)

            else:
                raise ValueError("Unknown conversion step type '%s'" % steptype)

        if fuse:
            mesh = mesh.fuse()
        return mesh


    def convertRandom(self, choices):
        """Convert choosing randomly between choices

        Returns a Mesh obtained by converting the current Mesh by a
        randomly selected method from the available conversion type
        for the current element type.
        """
        ml = self.splitRandom(len(choices), compact=False)
        ml = [m.convert(c) for m, c in zip(ml, choices)]
        prop = self.prop
        if prop is not None:
            prop = concatenate([m.prop for m in ml])
        elems = concatenate([m.elems for m in ml], axis=0)
        eltype = {m.elName() for m in ml}
        if len(eltype) > 1:
            raise RuntimeError("Invalid choices for random conversions")
        eltype = eltype.pop()
        return Mesh(self.coords, elems, prop, eltype)


    ## TODO:
    ## - mesh_wts and mesh_els functions should be moved to elements.py
    def subdivide(self, *ndiv, **kargs):
        """Subdivide the elements of a Mesh.

        Note
        ----
        This only works for some element types: 'line2', 'tri3', 'quad4',
        'hex8'.

        Parameters:

        - `ndiv`: specifies the number (and place) of divisions (seeds)
          along the edges of the elements. Accepted type and value depend
          on the element type of the Mesh. Currently implemented:

          - 'tri3': ndiv is a single int value specifying the number of
            divisions (of equal size) for each edge.

          - 'quad4': ndiv is a sequence of two int values nx,ny, specifying
            the number of divisions along the first, resp. second
            parametric direction of the element

          - 'hex8': ndiv is a sequence of three int values nx,ny,nz specifying
            the number of divisions along the first, resp. second and the third
            parametric direction of the element

        - `fuse`: bool, if True (default), the resulting Mesh is completely
          fused. If False, the Mesh is only fused over each individual
          element of the original Mesh.

        Returns a Mesh where each element is replaced by a number of
        smaller elements of the same type.

        .. note:: This is currently only implemented for Meshes of type 'tri3'
          and 'quad4' and 'hex8' and for the derived class 'TriSurface'.


        """
        elname = self.elName()
        try:
            mesh_wts = globals()[elname+'_wts']
            mesh_els = globals()[elname+'_els']
        except:
            raise ValueError("Can not subdivide element of type '%s'" % elname)

        wts = mesh_wts(*ndiv)
        lndiv = [nd if at.isInt(nd) else len(nd)-1 for nd in ndiv]
        els = mesh_els(*lndiv)
        X = self.coords[self.elems]
        U = dot(wts, X).transpose([1, 0, 2]).reshape(-1, 3)
        e = concatenate([els+i*wts.shape[0] for i in range(self.nelems())])
        M = self.__class__(U, e, eltype=self.elType())
        if self.prop is not None:
            M.setProp(at.repeatValues(self.prop, prod(lndiv)))
        if kargs.get('fuse', True):
            M = M.fuse()
        return M


    @utils.warning('mesh_reduce_degenerate')
    def reduceDegenerate(self, *arg, **kargs):
        return self.splitDegenerate(**kargs)


    def splitDegenerate(self, reduce=True, return_indices=False):
        """Split a Mesh in non-degenerate and degenerate elements.

        Splits a Mesh in non-degenerate elements and degenerate elements,
        and tries to reduce degenerate elements to lower plexitude elements.

        Parameters
        ----------
        reduce: bool or :class:`~elements.ElementType` name
            If True, the degenerate elements will be tested against
            known degeneration patterns, and the matching elements will be
            transformed to non-degenerate elements of a lower plexitude.
            If a string, it is an element name and only transforms to this
            element type will be considered.
            If False, no reduction of the degenerate elements will be
            attempted.
        return_indices: bool, optional
            If True, also returns the element indices in the original
            Mesh for all of the elements in the derived Meshes.

        Returns
        -------
        ML: list of Mesh objects
            The list of Meshes resulting from the split operation. The first
            holds the non-degenerate elements of the original Mesh. The last
            holds the remaining degenerate elements.
            The intermediate Meshes, if any, hold elements of a lower plexitude
            than the original.

        Warning
        -------
        The Meshes that hold reduced elements may still contain degenerate
        elements for the new element type

        Examples
        --------
        >>> M = Mesh(np.zeros((4,3)),
        ...     [[0,0,0,0],
        ...      [0,0,0,1],
        ...      [0,0,1,2],
        ...      [0,1,2,3],
        ...      [1,2,3,3],
        ...      [2,3,3,3],
        ...     ],eltype='quad4')
        >>> M.elems.listDegenerate()
        array([0, 1, 2, 4, 5])
        >>> for Mi in M.splitDegenerate(): print(Mi)
        Mesh: nnodes: 4, nelems: 1, nplex: 4, level: 2, eltype: quad4
          BBox: [ 0.  0.  0.], [ 0.  0.  0.]
          Size: [ 0.  0.  0.]
          Area: 0.0
        Mesh: nnodes: 4, nelems: 5, nplex: 3, level: 2, eltype: tri3
          BBox: [ 0.  0.  0.], [ 0.  0.  0.]
          Size: [ 0.  0.  0.]
          Area: 0.0
        >>> conn,ind = M.splitDegenerate(return_indices=True)
        >>> print(ind[0],ind[1])
        [3] [0 1 2 5 4]
        >>> print(conn[1].elems)
        [[0 0 0]
         [0 0 1]
         [0 1 2]
         [2 3 3]
         [1 2 3]]
        """
        if reduce is False:
            deg = self.elems.testDegenerate()
            ML = [self.select(~deg, compact=False),
                   self.select(deg, compact=False)]
            if return_indices:
                ind = [np.where(~deg)[0], np.where(deg)[0]]

        else:
            target = None if reduce is True else reduce
            conn, ind = self.elems.reduceDegenerate(target, return_indices=True)
            ML = [Mesh(self.coords, e) for e in conn]
            if self.prop is not None:
                ML = [M.setProp(self.prop[i]) for M, i in zip(ML, ind)]

        if return_indices:
            return ML, ind
        else:
            return ML


    def removeDegenerate(self):
        """Remove the degenerate elements from a Mesh.

        Returns
        -------
        Mesh
            A Mesh with all degenerate elements removed.
        """
        deg = self.elems.testDegenerate()
        return self.select(~deg, compact=False)


    def removeDuplicate(self, permutations='all'):
        """Remove the duplicate elements from a Mesh.

        Duplicate elements are elements that consist of the same nodes.

        Parameters
        ----------
        permutations: str
            Defines which permutations of the nodes are allowed while still
            considering the elements duplicates. Possible values are:

        - 'none': no permutations are allowed: the node list of the elements
          must have the same value at every position in order to be considered
          duplicates;
        - 'roll': rolling is allowed. Node lists that can be transformed into
          each other by rolling are considered equal;
        - 'all': any permutation of the same set of nodes will be considered
          a duplicate element. This is the default.

        Returns
        -------
        Mesh
            A Mesh with all duplicate elements removed.
        """
        return self.select(at.uniqueRows(self.elems, permutations))


    def renumber(self, order='elems'):
        """Renumber the nodes of a Mesh in the specified order.

        Parameters
        ----------
        order: int :term:`array_like` or str
            If an array, it is an index with length equal to the number of
            nodes. It should be a permutation of ``arange(self.nnodes())``.
            The index specifies the node number that should come at this
            position. Thus, the order values are the old node numbers on
            the new node number positions.

            ``order`` can also be a predefined string that will generate the
            node index automatically:

            - 'elems': the nodes are number in order of their appearance in the
              Mesh connectivity.
            - 'random': the nodes are numbered randomly.
            - 'front': the nodes are numbered in order of their frontwalk.

        Returns
        -------
        Mesh
            A Mesh equivalent with the input, but with the nodes numbered
            differently.

        """
        if order == 'elems':
            order = renumberIndex(self.elems)
        elif order == 'random':
            order = arange(self.nnodes())
            random.shuffle(order)
        elif order == 'front':
            adj = self.elems.adjacency('n')
            p = adj.frontWalk()
            order = p.argsort()
        newnrs = inverseUniqueIndex(order)
        return self.__class__(self.coords[order], newnrs[self.elems], prop=self.prop, eltype=self.elType())


    def reorder(self, order='nodes'):
        """Reorder the elements of a Mesh.

        Parameters
        ----------
        order: :term:`array_like` or str
            If an array, it is a permutation of the numbers in
            ``arange(self.nelems())``, specifying the requested order of
            the elements.

            ``order`` can also be one of the following predefined strings:

            - 'nodes': order the elements in increasing node number order.
            - 'random': number the elements in a random order.
            - 'reverse': number the elements in reverse order.

        Returns
        -------
        Mesh
            A Mesh equivalent with self but with the elements ordered as
            specified.

        """
        order = self.elems.reorder(order)
        if self.prop is None:
            prop = None
        else:
            prop = self.prop[order]
        return self.__class__(self.coords, self.elems[order], prop=prop, eltype=self.elType())

    #
    # Should we create some general 'masked mesh' class?
    #
    def connectedElements(self, startat, mask, level=0):
        """Return the elements reachable from startat.

        Finds the elements which can be reached from startat by walking along
        a mask (a subset of elements). Walking is possible over nodes, edges
        or faces, as specified in level.

        Parameters
        ----------
        startat: int or :term:`array_like`, int.
            The starting element number(s).
        level: int
            Specifies how elements can be reached: via node (0), edge (1)
            or face (2).
        mask: :term:`array_like`, bool or int.
            Flags the elements that are considered walkable. It is an int array
            with the walkable element numbers, or a bool array flagging the
            these elements with a value True.

        """
        startat = asarray(startat).reshape(-1)
        if len(intersect1d(startat, arange(self.nelems()))) < len(startat):
            raise ValueError("wrong elem index found in startat, outside range 0 - %d" % self.nelems())

        mask = asarray(mask)
        if mask.dtype == bool:
            if len(mask)!=self.nelems():
                raise ValueError("if it is an array of boolean mask should have all elements %d, got %d" % (self.nelems(), len(mask)))
            mask = where(mask)[0]
        if len(intersect1d(mask, arange(self.nelems()))) < len(mask):
            raise ValueError("wrong elem index found in mask, outside range 0 - %d" % self.nelems())

        startat = intersect1d(startat, mask)
        if len(startat) == 0:
            return []

        startat = findFirst(mask, startat)
        return mask[self.select(mask).reachableFrom(startat, level=level)]


##############################################################
    #
    # Connection, Extrusion, Sweep, Revolution
    #

    def connect(self, coordslist, div=1, degree=1, loop=False, eltype=None):
        """Connect a sequence of topologically congruent Meshes into a hypermesh.

        Parameters:

        - `coordslist`: either a list of Coords objects, or a list of
          Mesh objects or a single Mesh object.

          If Mesh objects are given, they should (all) have the same element
          type as `self`. Their connectivity tables will not be used though.
          They will only serve to construct a list of Coords objects by
          taking the `coords` attribute of each of the Meshes. If only a single
          Mesh was specified, `self.coords` will be added as the first Coords
          object in the list.

          All Coords objects in the coordslist (either specified or
          constructed from the Mesh objects), should have the exact same
          shape as `self.coords`. The number of Coords items in the list should
          be a multiple of `degree`, plus 1.

          Each of the Coords in the final coordslist is combined with the
          connectivity table, element type and property numbers of `self` to
          produce a list of toplogically congruent meshes.
          The return value is the hypermesh obtained by connecting
          each consecutive slice of (degree+1) of these meshes. The hypermesh
          has a dimensionality that is one higher than the original Mesh (i.e.
          points become lines, lines become surfaces, surfaces become volumes).
          The resulting elements will be of the given `degree` in the
          direction of the connection.

          Notice that unless a single Mesh was specified as coordslist, the
          coords of `self` are not used. In many cases however `self` or
          `self.coords` will be one of the items in the specified `coordslist`.

        - `degree`: degree of the connection. Currently only degree 1 and 2
          are supported.

          - If degree is 1, every Coords from the `coordslist`
            is connected with hyperelements of a linear degree in the
            connection direction.

          - If degree is 2, quadratic hyperelements are
            created from one Coords item and the next two in the list.
            Note that all Coords items should contain the same number of nodes,
            even for higher order elements where the intermediate planes
            contain less nodes.

            Currently, degree=2 is not allowed when `coordslist` is specified
            as a single Mesh.

        - `loop`: if True, the connections with loop around the list and
          connect back to the first. This is accomplished by adding the first
          Coords item back at the end of the list.

        - `div`: This should only be used for degree==1.

          With this parameter the generated connections can be further
          subdivided along the connection direction. `div` is either a
          single input for :func:`~arraytools.smartSeed`,
          or a list thereof.
          In the latter case, the length of the list should be one less
          than the length of the `coordslist`. Each pair of consecutive
          items from the coordinate list will be connected using the
          seeds generated by the corresponding value from `div`, passed to
          :func:`~arraytools.smartSeed`.
          Notice that if seed
          values are specified directly as a list of floats, the list
          should start with a value 0.0 and end with 1.0.

        - `eltype`: the element type of the constructed hypermesh. Normally,
          this is set automatically from the base element type and the
          connection degree. If a different element type is specified,
          a final conversion to the requested element type is attempted.
        """
        if isinstance(coordslist, list):
            if isinstance(coordslist[0], Mesh):
                if sum([c.elType() != self.elType() for c in coordslist]):
                    raise ValueError("All Meshes in the list should have same element type")
                clist = [c.coords for c in coordslist]
            else:
                clist = coordslist
        elif isinstance(coordslist, Mesh):
            clist = [self.coords, coordslist.coords]
            if degree == 2:
                raise ValueError("This only works for linear connection")
            ## BV: Any reason why this would not work??
            ##     xm = 0.5 * (clist[0]+clist[1])
            ##     clist.insert(1, xm)
        else:
            raise ValueError("Invalid coordslist argument")

        if sum([c.shape != self.coords.shape for c in clist]):
            raise ValueError("Incompatible shape  in coordslist")

        # implement loop parameter
        if loop:
            clist.append(clist[0])

        if (len(clist)-1) % degree != 0:
            raise ValueError("Invalid length of coordslist (%s) for degree %s." % (len(clist), degree))

        # set divisions
        if degree > 1:
            div = 1
        if not isinstance(div, list) or isFloat(div[0]):
            div=[div]

        # now we should have list of: ints, tuples or floatlists
        div = [at.smartSeed(divi)[1:] for divi in div]
        # check length
        nsteps = (len(clist)-1) // degree
        if len(div) == 1:
            div = div * nsteps
        elif len(div)!=nsteps:
            raise ValueError("A list of div seeds must have a length equal to (len(clist)-1)//degree) = %s" % nsteps)

        # For higher order non-lagrangian elements the procedure could be
        # optimized by first compacting the coords and elems.
        # Instead we opted for the simpler method of adding the maximum
        # number of nodes, and then selecting the used ones.
        # A final compact() throws out the unused points.

        # Concatenate the coordinates
        if degree == 1:
            # We do not have a 2nd degree interpolation yet
            x = [Coords.interpolate(xi, xj, d).reshape(-1, 3) for xi, xj, d in zip(clist[:-1], clist[1:], div)]
            clist = clist[:1] + x
        x = Coords.concatenate(clist)

        # Create the connectivity table
        nnod = self.ncoords()
        nrep = (x.shape[0]//nnod - 1) // degree
        ## print("NREP %s" % nrep)
        e = self.elems.extrude(nnod, degree).replic(nrep, nnod*degree)

        # Create the Mesh
        M = Mesh(x, e).setProp(self.prop)
        # convert to proper eltype
        if eltype:
            M = M.convert(eltype)
        return M


    def extrude(self, div, dir=0, length=1., degree=1, eltype=None):
        """Extrude a Mesh along a straight line.

        The Mesh is extruded over a given length in the given direction.

        Parameters
        ----------
        div: smartseed
            Specifies how the extruded direction will be subdivided in
            elements. It can be anything that is acceptable as input for
            :func:`~arraytools.smartSeed`.
        dir: int (0,1,2) or float :term:`array_like` (3,)
            The direction of the extrusion: either a global axis
            number or a direction vector.
        length: float
            The length of the extrusion, measured along the direction ``dir``.

        Returns
        -------
        Mesh
            A Mesh obtained by extruding the input Mesh over the
            given ``length`` in direction ``dir``, subdividing this
            length according to the seeds generated
            by ``smartSeed(div)``.

        Examples
        --------
        >>> M = Mesh(Formex(origin())).extrude(3,0,3)
        >>> print(M)
        Mesh: nnodes: 4, nelems: 3, nplex: 2, level: 1, eltype: line2
          BBox: [ 0.  0.  0.], [ 3.  0.  0.]
          Size: [ 3.  0.  0.]
          Length: 3.0

        """
        if isFloat(dir):
            # Probably old style extrude parameters?
            utils.warn("warn_mesh_extrude")
        t = at.smartSeed(div)
        if degree > 1:
            t2 = 0.5 * (t[:-1] + t[1:])
            t = concatenate([t[:1], column_stack([t2, t[1:]]).ravel()])
        x0 = self.coords
        x1 = x0.trl(dir, length)
        dx = x1-x0
        x = [x0 + ti*dx for ti in t]
        return self.connect(x, degree=degree, eltype=eltype)


    def revolve(self, n, axis=0, angle=360., around=None, loop=False, eltype=None):
        """Revolve a Mesh around an axis.

        Returns a new Mesh obtained by revolving the given Mesh
        over an angle around an axis in n steps, while extruding
        the mesh from one step to the next.
        This extrudes points into lines, lines into surfaces and surfaces
        into volumes.
        """
        angles = arange(n+1) * angle / n
        seq = [self.coords.rotate(angle=a, axis=axis, around=around) for a in angles]
        return self.connect(seq, loop=loop, eltype=eltype)


    def sweep(self, path, eltype=None, **kargs):
        """Sweep a mesh along a path, creating an extrusion

        Parameters:

        - `path`: Curve object. The path over which to sweep the Mesh.
        - `eltype`: string. Name of the element type on the
          returned Meshes.
        - `**kargs`: keyword arguments that are passed to
          :meth:`curve.Curve.sweep2`, with the same meaning.
          Usually, you will need to at least set the `normal` parameter.

        Returns a Mesh obtained by sweeping the given Mesh over a path.
        The returned Mesh has double plexitude of the original.
        If `path` is a closed Curve connect back to the first.

        This operation is similar to the extrude() method, but the path
        can be any 3D curve.
        """
        loop = path.closed
        seq = path.sweep2(self.coords, **kargs)
        return self.connect(seq, eltype=eltype, loop=loop)


    def smooth(self, iterations=1, lamb=0.5, k=0.1, edg=True, exclnod=[], exclelem=[], weight=None):
        """Return a smoothed mesh.

        Smoothing algorithm based on lowpass filters.

        If edg is True, the algorithm tries to smooth the
        outer border of the mesh seperately to reduce mesh shrinkage.

        Higher values of k can reduce shrinkage even more
        (up to a point where the mesh expands),
        but will result in less smoothing per iteration.

        - `exclnod`: It contains a list of node indices to exclude from the smoothing.
          If exclnod is 'border', all nodes on the border of the mesh will
          be unchanged, and the smoothing will only act inside.
          If exclnod is 'inner', only the nodes on the border of the mesh will
          take part to the smoothing.

        - `exclelem`: It contains a list of elements to exclude from the smoothing.
          The nodes of these elements will not take part to the smoothing.
          If exclnod and exclelem are used at the same time the union of them
          will be exluded from smoothing.

        -`weight` : it is a string  that can assume 2 values `inversedistance` and
          `distance`. It allows to specify the weight of the adjancent points according
          to their distance to the point
        """
        if self.elType().ndim == 1:
            if edg == True:
                raise ValueError("Cannot use edg=True with a line mesh")

        if iterations < 1:
            return self

        if lamb*k == 1:
            raise ValueError("Cannot assign values of lamb and k which result in lamb*k==1")

        mu = -lamb/(1-k*lamb)
        adj = self.getEdges().adjacency(kind='n')
        incl = resize(True, self.ncoords())

        if exclnod == 'border':
            exclnod = unique(self.getBorder())
            k = 0.  # k can be zero because it cannot shrink
            edg = False  # there is no border edge
        if exclnod == 'inner':
            exclnod = delete(arange(self.ncoords()), unique(self.getBorder()))
        exclelemnod = unique(self.elems[exclelem])
        exclude=array(unique(concatenate([exclnod, exclelemnod])), dtype = int)

        incl[exclude] = False

        if edg:
            externals = resize(False, self.ncoords())
            expoints = unique(self.getFreeEntities())
            if len(expoints) not in [0, self.ncoords()]:
                externals[expoints] = True
                a = adj[externals].ravel()
                inpoints = delete(arange(self.ncoords()), expoints)
                a[findFirst(inpoints, a) != -1] = -2
                adj[externals] = a.reshape(adj[externals].shape)
            else:
                print('Failed to recognize external points.\nShrinkage may be considerable.')
        w = ones(adj.shape, dtype=float)

        if weight == 'inversedistance':
            dist = length(self.coords[adj]-self.coords.reshape(-1, 1, 3))
            w[dist!=0] /= dist[dist!=0]

        if weight == 'distance':
            w = length(self.coords[adj]-self.coords.reshape(-1, 1, 3))

        w[adj<0] = 0.
        w /= w.sum(-1).reshape(-1, 1)
        w = w.reshape(adj.shape[0], adj.shape[1], 1)
        c = self.coords.copy()
        for i in range(iterations):
            c[incl] = (1.-lamb)*c[incl]  + lamb*(w[incl] *c[adj][incl]).sum(1)
            c[incl] = (1.-mu)*c[incl]  + mu*(w[incl] *c[adj][incl]).sum(1)
        return self.__class__(c, self.elems, prop=self.prop, eltype=self.elType())


    def __add__(self, other):
        """Return the sum of two Meshes.

        The sum of the Meshes is simply the concatenation thereof.
        It allows us to write simple expressions as M1+M2 to concatenate
        the Meshes M1 and M2. Both meshes should be of the same plexitude
        and have the same eltype.
        The result will be of the same class as self (either a Mesh or a
        subclass thereof).
        """
        return self.concatenate([self, other])


    @classmethod
    def concatenate(clas, meshes, fuse=True, **kargs):
        """Concatenate a list of meshes of the same plexitude and eltype

        All Meshes in the list should have the same plexitude.
        Meshes with plexitude are ignored though, to allow empty
        Meshes to be added in.

        Merging of the nodes can be tuned by specifying extra arguments
        that will be passed to :meth:`coords.Coords:fuse`.

        If any of the meshes has property numbers, the resulting mesh will
        inherit the properties. In that case, any meshes without properties
        will be assigned property 0.
        If all meshes are without properties, so will be the result.

        This is a class method, and should be invoked as follows::

          Mesh.concatenate([mesh0,mesh1,mesh2])
        """
        def _force_prop(m):
            if m.prop is None:
                return zeros(m.nelems(), dtype=Int)
            else:
                return m.prop

        meshes = [m for m in meshes if m.nplex() > 0]
        nplex = {m.nplex() for m in meshes}
        if len(nplex) > 1:
            raise ValueError("Cannot concatenate meshes with different plexitude: %s" % str(nplex))
        eltype = {m.elType() for m in meshes}
        if len(eltype) > 1:
            raise ValueError("Cannot concatenate meshes with different eltype: %s" % [m.elName() for m in meshes])

        # Keep the available props
        prop = [m.prop for m in meshes if m.prop is not None]
        if len(prop) == 0:
            prop = None
        elif len(prop) < len(meshes):
            prop = concatenate([_force_prop(m) for m in meshes])
        else:
            prop = concatenate(prop)

        coords, elems = mergeMeshes(meshes, fuse=fuse, **kargs)
        elems = concatenate(elems, axis=0)
        return clas(coords, elems, prop=prop, eltype=eltype.pop())


    # Test and clipping functions


    def test(self, nodes='all', dir=0, min=None, max=None, atol=0.):
        """Flag elements having nodal coordinates between min and max.

        This function is very convenient in clipping a Mesh in a specified
        direction. It returns a 1D integer array flagging (with a value 1 or
        True) the elements having nodal coordinates in the required range.
        Use where(result) to get a list of element numbers passing the test.
        Or directly use clip() or cclip() to create the clipped Mesh

        The test plane can be defined in two ways, depending on the value of dir.
        If dir == 0, 1 or 2, it specifies a global axis and min and max are
        the minimum and maximum values for the coordinates along that axis.
        Default is the 0 (or x) direction.

        Else, dir should be compaitble with a (3,) shaped array and specifies
        the direction of the normal on the planes. In this case, min and max
        are points and should also evaluate to (3,) shaped arrays.

        nodes specifies which nodes are taken into account in the comparisons.
        It should be one of the following:

        - a single (integer) point number (< the number of points in the Formex)
        - a list of point numbers
        - one of the special strings: 'all', 'any', 'none'

        The default ('all') will flag all the elements that have all their
        nodes between the planes x=min and x=max, i.e. the elements that
        fall completely between these planes. One of the two clipping planes
        may be left unspecified.
        """
        if min is None and max is None:
            raise ValueError("At least one of min or max have to be specified.")

        if pf.isString(nodes):
            nod = arange(self.nplex())
        else:
            nod = nodes

        # Perform the test on the selected nodes
        X = self.coords[self.elems][:, nod]
        T = X.test(dir=dir, min=min, max=max, atol=atol)

        if len(T.shape) > 1:
            # We have results for more than 1 node per element
            if nodes == 'any':
                T = T.any(axis=1)
            elif nodes == 'none':
                T = ~T.any(axis=1)
            else:
                T = T.all(axis=1)

        return asarray(T)


    def clipAtPlane(self, p, n, nodes='any', side='+'):
        """Return the Mesh clipped at plane (p,n).

        This is a convenience function returning the part of the Mesh
        at one side of the plane (p,n)
        """
        if side == '-':
            n = -n
        return self.clip(self.test(nodes=nodes, dir=n, min=p))


    def intersectionWithLines(self, approximated=True, **kargs):
        """Return the intersections of a level-2 Mesh with lines.

        The Mesh is intersected with lines. The arguments and return values are
        the same as in :meth:`trisurface.TriSurface.intersectionWithLines`,
        except for the `approximated`.

        For a Mesh with eltype 'tri3', the intersections are exact. For other
        eltypes, if `approximated` is True a conversion to 'tri3' is done before
        computing the intersections. This may produce an exact result,
        an approximate result or no result (if the conversion fails).
        Of course the user can create his own approximation to a 'tri3'
        surface first, before calling this method.
        """
        if self.elName() == 'tri3':
            p, i = self.toSurface().intersectionWithLines(**kargs)
        else:
            if approximated:
                S = self.copy().setProp(list(range(self.nelems()))).toSurface()
                p, i = S.intersectionWithLines(**kargs)
                i[:, 2] = S.prop[i[:, 2]]
            else:
                raise ValueError('Exact intersectionWithLines not implemented for %s mesh'%self.elName())
        return p, i


    def levelVolumes(self):
        """Return the level volumes of all elements in a Mesh.

        The level volume of an element is defined as:

        - the length of the element if the Mesh is of level 1,
        - the area of the element if the Mesh is of level 2,
        - the (signed) volume of the element if the Mesh is of level 3.

        The level volumes can be computed directly for Meshes of eltypes
        'line2', 'tri3' and 'tet4' and will produce accurate results.
        All other Mesh types are converted to one of these before computing
        the level volumes. Conversion may result in approximation of the
        results. If conversion can not be performed, None is returned.

        If succesful, returns an (nelems,) float array with the level
        volumes of the elements.
        Returns None if the Mesh level is 0, or the conversion to the
        level's base element was unsuccesful.

        Note that for level-3 Meshes, negative volumes will be returned
        for elements having a reversed node ordering.
        """
        from pyformex.geomtools import levelVolumes

        base_elem = {
            1: 'line2',
            2: 'tri3',
            3: 'tet4'
            }

        try:
            base = base_elem[self.level()]
        except:
            return None

        if self.elName() == base:
             M = self
        else:
            try:
                M = self.shallowCopy(prop=arange(self.nelems())).convert(base)
            except:
                print("CONVERSION TO %s FAILED!" % base)
                return None

        V = levelVolumes(M.coords[M.elems])
        if V is not None and M != self:
            V = array([V[where(M.prop==i)[0]].sum() for i in range(self.nelems())])
        return V


    def lengths(self):
        """Return the length of all elements in a level-1 Mesh.

        For a Mesh with eltype 'line2', the lengths are exact. For other
        eltypes, a conversion to 'line2' is done before computing the lengths.
        This may produce an exact result, an approximated result or no result
        (if the conversion fails).

        If succesful, returns an (nelems,) float array with the lengths.
        Returns None if the Mesh level is not 1, or the conversion to 'line2'
        does not succeed.
        """
        if self.level() == 1:
            return self.levelVolumes()
        else:
            return None


    def areas(self):
        """Return the area of all elements in a level-2 Mesh.

        For a Mesh with eltype 'tri3', the areas are exact. For other
        eltypes, a conversion to 'tri3' is done before computing the areas.
        This may produce an exact result, an approximate result or no result
        (if the conversion fails).

        If succesful, returns an (nelems,) float array with the areas.
        Returns None if the Mesh level is not 2, or the conversion to 'tri3'
        does not succeed.
        """
        if self.level() == 2:
            return self.levelVolumes()
        else:
            return None


    def volumes(self):
        """Return the signed volume of all the mesh elements

        For a 'tet4' tetraeder Mesh, the volume of the elements is calculated
        as 1/3 * surface of base * height.

        For other Mesh types the volumes are calculated by first splitting
        the elements into tetraeder elements.

        The return value is an array of float values with length equal to the
        number of elements.
        If the Mesh conversion to tetraeder does not succeed, the return
        value is None.
        """
        if self.level() == 3:
            return self.levelVolumes()
        else:
            return None


    def length(self):
        """Return the total length of a Mesh.

        Returns the sum of self.lengths(), or 0.0 if the self.lengths()
        returned None.
        """
        try:
            return self.lengths().sum()
        except:
            return 0.0


    def area(self):
        """Return the total area of a Mesh.

        Returns the sum of self.areas(), or 0.0 if the self.areas()
        returned None.
        """
        try:
            return self.areas().sum()
        except:
            return 0.0


    def volume(self):
        """Return the total volume of a Mesh.

        For a Mesh of level < 3, a value 0.0 is returned.
        For a Mesh of level 3, the volume is computed by converting its
        border to a surface and taking the volume inside that surface.
        It is equivalent with ::

            self.toSurface().volume()

        This is far more efficient than `self.volumes().sum()`.
        """
        if self.level() == 3:
            return self.toSurface().volume()
        else:
            return 0.0


    def fixVolumes(self):
        """Reverse the elements with negative volume.

        Elements with negative volume may result from incorrect
        local node numbering. This method will reverse all elements
        in a Mesh of dimensionality 3, provide the volumes of these
        elements can be computed.
        """
        return self.reverse(self.volumes() < 0.)


##########################################
    ## Deprecated ##

    @utils.deprecated_by('Mesh.nodalToElement', 'Field.convert')
    def nodalToElement(self, val):
        return val[self.elems].mean(axis=1)


    @utils.deprecated_by('Mesh.getLowerEntitiesSelector', 'Element.getEntities')
    def getLowerEntitiesSelector(self, level=-1):
        return self.elType().getEntities(level)


    # 20190514
    @utils.deprecated_by('Mesh.setType','Mesh.setEltype or Mesh.eltype assignment')
    def setType(self, eltype):
        return self.setEltype(eltype)


##########################################
    ## Allow drawing ##

    def actor(self, **kargs):

        if self.nelems() == 0:
            return None

        from pyformex.opengl.drawable import GeomActor
        return GeomActor(self, **kargs)


######################## Functions #####################


def mergeNodes(nodes, fuse=True, **kargs):
    """Merge all the nodes of a list of node sets.

    Merging the nodes creates a single Coords object containing all nodes,
    and the indices to find the points of the original node sets in the
    merged set.

    Parameters:

    - `nodes`: a list of Coords objects, all having the same shape, except
      possibly for their first dimension
    - `fuse`: if True (default), coincident (or very close) points will
      be fused to a single point
    - `**kargs`: keyword arguments that are passed to the fuse operation

    Returns:

    - a Coords with the coordinates of all (unique) nodes,
    - a list of indices translating the old node numbers to the new. These
      numbers refer to the serialized Coords.

    The merging operation can be tuned by specifying extra arguments
    that will be passed to :meth:`coords.Coords.fuse`.
    """
    coords = Coords(concatenate([x for x in nodes], axis=0))
    if fuse:
        coords, index = coords.fuse(**kargs)
    else:
        index = arange(coords.shape[0])
    n = array([0] + [x.npoints() for x in nodes]).cumsum()
    ind = [index[f:t] for f, t in zip(n[:-1], n[1:])]
    return coords, ind


def mergeMeshes(meshes, fuse=True, **kargs):
    """Merge all the nodes of a list of Meshes.

    Each item in meshes is a Mesh instance.
    The return value is a tuple with:

    - the coordinates of all unique nodes,
    - a list of elems corresponding to the input list,
      but with numbers referring to the new coordinates.

    The merging operation can be tuned by specifying extra arguments
    that will be passed to :meth:`coords.Coords:fuse`.
    Setting fuse=False will merely concatenate all the mesh.coords, but
    not fuse them.
    """
    coords = [m.coords for m in meshes]
    elems = [m.elems for m in meshes]
    coords, index = mergeNodes(coords, fuse, **kargs)
    return coords, [Connectivity(i[e], eltype=e.eltype) for i, e in zip(index, elems)]


def line2_wts(seed0):
    """Create weights for line2 subdivision.

    Parameters
    ----------
    seed0: int or list of floats
        The subdivisions along the first parametric direction of the element.
        If an int, the subdivisions will be equally spaced between 0 and 1

    Examples
    --------
    >>> line2_wts(4)
    array([[ 0.  ,  1.  ],
           [ 0.25,  0.75],
           [ 0.5 ,  0.5 ],
           [ 0.75,  0.25],
           [ 1.  ,  0.  ]])
    """
    wts = at.gridpoints(seed0)
    return column_stack([wts, 1-wts])


def line2_els(nx):
    #n = nx+1
    els = [array([0, 1]) + i for i in range(nx)]
    return row_stack(els)


def tri3_wts(ndiv):
    n = ndiv+1
    seeds = arange(n)
    pts = concatenate([
        column_stack([seeds[:n-i], [i]*(n-i)])
        for i in range(n)])
    pts = column_stack([ndiv-pts.sum(axis=-1), pts])
    return pts / float(ndiv)


def tri3_els(ndiv):
    n = ndiv+1
    els1 = [row_stack([array([0, 1, n-j]) + i for i in range(ndiv-j)]) + j * n - j*(j-1)//2 for j in range(ndiv)]
    els2 = [row_stack([array([1, 1+n-j, n-j]) + i for i in range(ndiv-j-1)]) + j * n - j*(j-1)//2 for j in range(ndiv-1)]
    elems = row_stack(els1+els2)

    return elems


def quad4_wts(seed0, seed1):
    """ Create weights for quad4 subdivision.

    Parameters:

    - 'seed0' : int or list of floats . It specifies divisions along the
      first parametric direction of the element

    - 'seed1' : int or list of floats . It specifies divisions along
      the second parametric direction of the element

    If these parameters are integer values the divisions will be equally
    spaced between  0 and 1

    This is equivalent with `~arraytools.gridpoints(seed0, seed1)`.
    """
    return at.gridpoints(seed0, seed1)


def quad4_els(nx, ny):
    """Quad4 element connectivity for a regular stack of nx,ny elements.

    The node numbers vary first in the x, then in the y direction.
    """
    n = nx+1
    els = [row_stack([array([0, 1, n+1, n]) + i for i in range(nx)]) + j * n for j in range(ny)]
    return row_stack(els)


# TODO: remove or document in subdivide
quad9_wts=quad4_wts
quad9_els=quad4_els


def quadgrid(seed0, seed1, roll=0):
    """Create a quadrilateral mesh of unit size with the specified seeds.

    Parameters:

    - `seed0`,`seed1`: seeds for the elements along the parametric directions
      0 and 1. Each can be one of the following:

      - an integer number, specifying the number of equally sized elements
        along that direction,
      - a tuple (n,) or (n,e0) or (n,e0,e1), to be used as parameters in the
        :func:`mesh.seed` function,
      - a list of monotonously increasing float values in the range 0.0 to 1.0,
        specifying the relative positions of the nodes. Normally, the first and
        last values of the seeds are 0. and 1., leading to a unit square grid.

    The node and element numbers vary first in the x, then in the y direction.
    """
    from pyformex.elements import Quad4
    seed0 = at.smartSeed(seed0)
    seed1 = at.smartSeed(seed1)
    wts = at.gridpoints(seed0, seed1)
    n0 = len(seed0)-1
    n1 = len(seed1)-1
    E = Quad4.toMesh()
    if roll:
        E = E.rollAxes(roll)
    X = E.coords.reshape(-1, 4, 3)
    U = dot(wts, X).transpose([1, 0, 2]).reshape(-1, 3)
    els = quad4_els(n0, n1)
    e = concatenate([els+i*wts.shape[0] for i in range(E.nelems())])
    M = Mesh(U, e, eltype=E.elType())
    return M.fuse()


def hex8_wts(seed0, seed1, seed2):
    """ Create weights for hex8 subdivision.

    Parameters:

    - 'seed0' : int or list of floats . It specifies divisions along the
      first parametric direction of the element

    - 'seed1' : int or list of floats . It specifies divisions along
      the second parametric direction of the element

    - 'seed2' : int or list of floats . It specifies divisions along
      the t parametric direction of the element


    If these parametes are integer values the divisions will be equally
    spaced between  0 and 1
    """
    return at.gridpoints(seed0, seed1, seed2)



def hex8_els(nx, ny, nz):
    """ Create connectivity table for hex8 subdivision.

    """
    n = nz+1
    els = [row_stack([row_stack([asarray([array([0, 1, n+1, n]) + k for k in range(nz)])+i * n for i  in range(nx)])]) +j * (n*(nx+1)) for j in range(ny+1)]
    els = concatenate([row_stack(els[:-1]), row_stack(els[1:])], axis=1)
    return els


def rectangle(L=1., W=1., nl=1, nw=1):
    """Create a plane rectangular mesh of quad4 elements.

    Parameters:
    - `L`,`W`: length,width of the rectangle.

    """
    return quadgrid(nl, nw).resized([L, W, 1.0])


def rectangleWithHole(L, W, r, nr, nl, nw=None, e0=0.0, eltype='quad4'):
    """Create a quarter of rectangle with a central circular hole.

    Parameters:

    - `L`: float. Length of the (quarter) rectangle
    - `W`: float. Width of the (quarter) rectangle
    - `r`: float. Radius of the hole
    - `nr`: integer. Number of elements over radial direction
    - `nl`: integer. Number of elements over tangential direction along L
    - `nw`: integer. Number of elements over tangential direction along W.
        If None (default), it will be set equal to nl.
    -  `e0`: float. Concentration factor for elements in the radial direction

    Returns a Mesh
    """
 #   L = W
    from pyformex import elements
    if nw is None:
        nw = nl
    base = elements.Quad9.vertices.scale([L, W, 1.])
    F0 = Formex([[[r, 0., 0.]]]).rosette(5, 90./4)
    F2 = Formex([[[L, 0.]], [[L, W/2.]], [[L, W]], [[L/2., W]], [[0., W]]])
    F1 = F0.interpolate(F2, div=[0.5])
    FL = [F0, F1, F2]
    X0, X1, X2 = [F.coords.reshape(-1, 3) for F in FL]
    trf0 = Coords([X0[0], X2[0], X2[2], X0[2], X1[0], X2[1], X1[2], X0[1], X1[1]])
    trf1 = Coords([X0[2], X2[2], X2[4], X0[4], X1[2], X2[3], X1[4], X0[3], X1[3]])

    seedr = at.smartSeed((nr, e0))
    seedl = at.smartSeed(nl)
    seedw = at.smartSeed(nw)
    gridl = quadgrid(seedr, seedl).resized([L, W, 1.0])
    gridw = quadgrid(seedr, seedw).resized([L, W, 1.0])

    gridl = gridl.isopar('quad9', trf0, base)
    gridw = gridw.isopar('quad9', trf1, base)
    return (gridl+gridw).fuse()


def quadrilateral(x, n1, n2):
    """Create a quadrilateral mesh

    Parameters:

    - `x`: Coords(4,3): Four corners of the mesh, in anti-clockwise order.
    - `n1`: number of elements along sides x0-x1 and x2-x3
    - `n2`: number of elements along sides x1-x2 and x3-x4

    Returns a Mesh of quads filling the quadrilateral defined  by the four
    points `x`.
    """
    from pyformex.elements import Quad4
    x = checkArray(x, (4, 3), 'f')
    M = rectangle(1., 1., nl, nw).isopar('quad4', x, Quad4.vertices)
    return M


def continuousCurves(c0, c1):
    """Make sure the two curves are continuous.

    Ensures that the end point of curve c0 and the start point of curve c1
    are coincident.
    This is done by replacing these two points with their mean value.
    """
    c0.coords[-1] = c1.coords[0] = 0.5 * (c0.coords[-1] + c1.coords[0])


def triangleQuadMesh(P0, C0, n, P0wgt=1.0):
    """Create a quad Mesh over a triangular region

    The triangle can have a single non-straight edge. The domain is
    described by a curve and a point. The straight lines between the
    curve ends and the point are the other two sides.

    Parameters:

    - `P0`: a point
    - `C`: a curve
    - `ndiv`: a tuple of 3 int's. The quad kernel near the point will
      have n0*n1 elements (n0 to the start of the curve, n1 to the end.
      The zone near the curve has n0+n1 elements along the curve, and
      n2 elements perpendicular to the curve.
    """
    from pyformex.plugins.curve import PolyLine
    # Make sure we have PolyLines
    C = [
        PolyLine([P0, C0.coords[0]]),
        C0.approx(),
        PolyLine([C0.coords[-1], P0]),
        ]
    # Make sure the end points are connected
    (continuousCurves(C[i], C[(i+1)%3]) for i in range(3))
    # Number of divisions along each side
    n = array(n)
    nt = array([n[0]+n[2], n[1]+n[0], n[2]+n[1]])
    # Divide the three sides
    C = [C[i].approx(nseg=nt[i], equidistant=True) for i in range(3)]
    # Split the curves in two parts
    C = [C[i].split(n[i]) for i in range(3)]
    # Create the center point
    P1 = Coords.concatenate([Ci[1].coords[0] for Ci in C]).mean(axis=0)
    if P0wgt > 0.0:
        P1 = (3.*P1 + P0wgt*P0) / (3.+P0wgt)
    # Create the central PolyLines
    D = [PolyLine([C[i][1].coords[0], P1]).approx(nseg=n[(i+1)%3], equidistant=True)  for i in range(3)]
    # Create the submeshes
    M = [Ci.toMesh().connect(Di.toMesh(), div=ni) for Ci, Di, ni in [
        (C[0][0], D[2], n[1]),
        (C[1][0], D[0], n[2]),
        (C[1][1], D[2].reverse(), n[2]),
        ]]
    # Concatenate Meshes
    return Mesh.concatenate(M)


def quarterCircle(n1, n2):
    """
    Create a mesh of quadrilaterals filling a quarter circle.

    Parameters
    ----------
    n1: int
        Number of elements along the edges 01 and 23
    n2: int
        Number of elements along the edges 12 and 30

    Returns
    -------
    Mesh
       A 'quad4' Mesh filling a quarter circle with radius 1 and
       center at the origin, in the first quadrant of the axes.

    Notes
    -----
    The quarter circle mesh has a kernel of n1*n1 cells, and two
    border meshes of n1*n2 cells. The resulting mesh has n1+n2 cells
    in radial direction and 2*n1 cells in tangential direction (in the
    border mesh).

    """
    from pyformex.plugins.curve import Arc, PolyLine
    r = float(n1)/(n1+n2)  # radius of the kernel
    P0, P1 = Coords([[0., 0., 0.], [r, 0., 0.]])
    P2 = P1.rot(45.)
    P3 = P1.rot(90.)
    # Kernel is a quadrilateral
    C0 = PolyLine([P0, P1]).approx(ndiv=n1).toMesh()
    C1 = PolyLine([P3, P2]).approx(ndiv=n1).toMesh()
    M0 = C0.connect(C1, div=n1)
    # Border meshes
    C0 = Arc(center=P0, radius=1., angles=(0., 45.)).approx(ndiv=n1).toMesh()
    C1 = PolyLine([P1, P2]).approx(ndiv=n1).toMesh()
    M1 = C0.connect(C1, div=n2)
    C0 = Arc(center=P0, radius=1., angles=(45., 90.)).approx(ndiv=n1).toMesh()
    C1 = PolyLine([P2, P3]).approx(ndiv=n1).toMesh()
    M2 = C0.connect(C1, div=n2)
    return M0+M1+M2


def wedge6_roll(elems):
    """Roll wedge6 elems to make the lowest node of bottom plane the first

    This is a helper function for the :meth:`wedge6_tet4` conversion.
    """
    elems = elems.reshape(-1, 2, 3)
    r = elems[:, 0, :].argmin(axis=-1)
    elemsout = []
    for i in range(3):
        w = where(r==i)[0]
        if len(w) > 0:
            els = roll(elems[w], -i, axis=-1)
            elemsout.append(els)
    elems = np.concatenate(elemsout)
    return Connectivity(elems.reshape(-1, 6), eltype='wedge6')


def wedge6_tet4(M):
    """Convert a Mesh from wedge6 to tet4

    This converts a 'wedge6' Mesh to 'tet4', by replacing each wedge
    element with three tets. The conversion ensures that the subdivision
    of the wedge elements are compatible in the common quad faces of
    any two wedge elements.

    Parameters
    ----------
    M: Mesh
        A Mesh of eltype 'wedge6'.

    Returns
    -------
    Mesh
        A Mesh of eltype 'tet4' representing the same domain as the
        input Mesh. The nodes are the same as those of the input Mesh.
        The number of elements is three times that of the input Mesh.
        The order of numbering of the elements is dependent on the
        conversion algorithm.
    """
    from pyformex.elements import Wedge6

    # First roll to put lowest node first
    elems = wedge6_roll(M.elems)
    M = Mesh(M.coords, elems, eltype='wedge6', prop=M.prop)

    # Now convert to tet4
    sl, sr = [Wedge6.conversions[c][0][1] for c in ['tet4-3l', 'tet4-3r']]
    elems = M.elems.reshape(-1, 2, 3)
    wl = where(elems[:, 0, 1] < elems[:, 0, 2])[0]
    wr = at.complement(wl, elems.shape[0])
    Ml = M.select(wl).selectNodes(sl, 'tet4')
    Mr = M.select(wr).selectNodes(sr, 'tet4')
    return Ml+Mr


# End
