#
##
##  This file is part of pyFormex 1.0.7  (Mon Jun 17 12:20:39 CEST 2019)
##  pyFormex is a tool for generating, manipulating and transforming 3D
##  geometrical models by sequences of mathematical operations.
##  Home page: http://pyformex.org
##  Project page:  http://savannah.nongnu.org/projects/pyformex/
##  Copyright 2004-2019 (C) Benedict Verhegghe (benedict.verhegghe@ugent.be)
##  Distributed under the GNU General Public License version 3 or later.
##
##  This program is free software: you can redistribute it and/or modify
##  it under the terms of the GNU General Public License as published by
##  the Free Software Foundation, either version 3 of the License, or
##  (at your option) any later version.
##
##  This program is distributed in the hope that it will be useful,
##  but WITHOUT ANY WARRANTY; without even the implied warranty of
##  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##  GNU General Public License for more details.
##
##  You should have received a copy of the GNU General Public License
##  along with this program.  If not, see http://www.gnu.org/licenses/.
##

# This is the only pyFormex module that is imported by the main script,
# so this is the place to put startup code

"""pyFormex main module

This module contains the main function of pyFormex, which is run by the
startup script.
"""
from __future__ import absolute_import, division, print_function

import sys, os

import pyformex as pf
from pyformex import Path
from pyformex import utils
from pyformex import software

# Note: none of the above import numpy

###########################  main  ################################


def printcfg(key):
    try:
        print("!! refcfg[%s] = %s" % (key, pf.refcfg[key]))
    except KeyError:
        pass
    print("!! cfg[%s] = %s" % (key, pf.cfg[key]))


def remove_pyFormex(pyformexdir, executable):
    """Remove the pyFormex installation."""
    if not executable.stem == 'pyformex':
        print(
            """
############ ERROR!
The --remove option can only be executed from the pyformex command.
Use the command: pyformex --remove.
"""
        )
        return

    if pf.installtype == 'D':
        print(
            """
############ ERROR!
It looks like this version of pyFormex was installed from a distribution
.deb package. You should use your distribution's package tools to remove
this pyFormex installation.
"""
        )
        return

    if pf.installtype in 'SG':
        print(
            """
############ ERROR!
It looks like you are running pyFormex directly from a source tree at
%s. I will not remove it.
If you have enough privileges, you can just remove the whole source tree
from the file system.
""" % pyformexdir
        )
        return

    if not pyformexdir.is_absolute():
        print(
            """
############ ERROR!
The pyFormex installation path is not an absolute path. Probably you are
executing this command from inside a pyFormex source directory.
The 'pyformex --remove' command should be executed from outside the source.
"""
        )
        return

    # If we get here, this is probably an install from tarball release

    # Remove the installation
    import glob
    prefixdir = pyformexdir.commonpath('/usr/local')
    #print(pyformexdir, prefixdir)
    bindir = prefixdir / 'bin'
    egginfo = pyformexdir.with_name(
        "pyformex-%s.egg-info" % (pf.__version__.replace('-', '_'))
    )
    print("egginfo = %s" % egginfo)
    script = bindir / 'pyformex'
    print("script = %s" % script)

    #
    # TEST FOR OTHER PYTHON VERSION
    if pf.PY2:
        this = '2'
        other = '3'
    if pf.PY3:
        this = '3'
        other = '2'
    cmd = '%s -%s --whereami' % (script, other)
    P = utils.system(cmd)
    if P.sta == 0:
        out = P.out.split('\n')
        if out[0] == pf.fullVersion() and '(R)' in out[2]:
            # Also remove other?
            pass
        else:
            other = ''
    else:
        other = ''

    scripts = [script] + glob.glob(script + '-*')
    gtsexe = ['gtscoarsen', 'gtsinside', 'gtsrefine', 'gtsset', 'gtssmooth']
    scripts += [bindir / g for g in gtsexe]
    scripts = [p for p in scripts if p.exists()]
    datadir = prefixdir / 'share'
    #print("datadir = %s" % datadir)
    data = [
        datadir / 'man/man1' / f for f in [
            'pyformex.1', 'pyformex-dxfparser.1', 'pyformex-postabq.1',
            'gtscoarsen.1', 'gtsinside.1', 'gtsrefine.1', 'gtsset.1',
            'gtssmooth.1'
        ]
    ]
    data += [
        datadir / f for f in [
            'applications/pyformex.desktop',
            'pixmaps/pyformex-64x64.png',
            'pixmaps/pyformex.xpm',
        ]
    ]
    data = [p for p in data if p.exists()]
    other_files = scripts + data
    #print(other_files)

    if other:
        print(
            """
############ NOTE!
I found a Python2 and a Python3 installation of %s
If you continue the removal, I will only remove the Python%s version.
Afterwards, your Python%s version will still run. You can remove that
with the command 'pyformex -%s --remove'.
""" % (pf.fullVersion(), this, other, other)
        )
        do_other = 'KEEP'
    else:
        do_other = 'REMOVE'

    print(
        """
############ BEWARE!
This procedure will:
- %s the complete Python%s pyFormex installation in %s,
- %s the pyFormex executable(s) (%s) in %s,
- %s some pyFormex data files (%s) in %s.

You should only use this on a pyFormex installed from a source .tar.gz release
with the command 'python(2/3) setup.py install'.
You need proper permissions to actually delete the files.
After succesful removal, you will not be able to run this pyFormex again,
unless you re-install it.
""" % (
            'REMOVE', this, pyformexdir, do_other, ', '.join(
                [p.name for p in scripts]
            ), bindir, do_other, ', '.join([p.name for p in data]), datadir
        )
    )

    s = pf.userInput("Are you sure you want to remove pyFormex? yes/NO: ")
    if s == 'yes':
        print("Removing pyformex tree: %s" % pyformexdir)
        pyformexdir.removeTree()
        if egginfo.exists():
            egginfo.unlink()

        if other == '':
            print("Removing other files: %s" % other_files)
            for f in other_files:
                if f.exists():
                    print("Removing %s" % f)
                    try:
                        Path(f).unlink()
                    except:
                        print("Could not remove %s" % f)
                else:
                    print("No such file: %s" % f)

        print("\nBye, bye! I won't be back until you reinstall me!")
    elif s.startswith('y') or s.startswith('Y'):
        print("You need to type exactly 'yes' to remove me.")
    else:
        print("Thanks for letting me stay this time.")
    sys.exit()


def savePreferences():
    """Save the preferences.

    The name of the preferences file is determined at startup from
    the configuration files, and saved in ``pyformex.preffile``.
    If a local preferences file was read, it will be saved there.
    Otherwise, it will be saved as the user preferences, possibly
    creating that file.
    If ``pyformex.preffile`` is None, preferences are not saved.
    """
    pf.debug("savePreferences to: %s" % pf.preffile, pf.DEBUG.CONFIG)
    if pf.preffile is None:
        return

    # Create the user conf dir
    #try:
    pf.preffile.parent.mkdir(parents=True, exist_ok=True)
    #except:
    #    print("The path where your user preferences should be stored can not be created!\nPreferences are not saved!")
    #    return

    #     print("""
    # ############################
    # ## Currently I can not save the preferences in the new format!
    # ## After all, this is a development version ;)
    # ###############################################################
    # """)
    #     return

    # Currently erroroneously processed, therefore not saved
    del pf.prefcfg['render/light0']
    del pf.prefcfg['render/light1']
    del pf.prefcfg['render/light2']
    del pf.prefcfg['render/light3']

    pf.debug("=" * 60, pf.DEBUG.CONFIG)
    pf.debug("!!!Saving config:\n%s" % pf.prefcfg, pf.DEBUG.CONFIG)

    try:
        pf.debug("Saving preferences to file %s" % pf.preffile, pf.DEBUG.CONFIG)
        pf.prefcfg.write(pf.preffile)
        res = "Saved"
    except:
        res = "Could not save"
        raise
    pf.debug("%s preferences to file %s" % (res, pf.preffile), pf.DEBUG.CONFIG)
    return res == "Saved"


def apply_config_changes(cfg):
    """Apply incompatible changes in the configuration

    cfg is the user configuration that is to be saved.
    """
    # Safety checks
    if not isinstance(cfg['warnings/filters'], list):
        cfg['warnings/filters'] = []

    for filt in cfg['warnings/filters']:
        if len(filt) > 2 and filt[2] == 'D':
            cfg['warnings/filters'].remove(filt)

    # Adhoc changes
    if isinstance(cfg['gui/dynazoom'], str):
        cfg['gui/dynazoom'] = [cfg['gui/dynazoom'], '']

    for i in range(8):
        t = "render/light%s" % i
        try:
            cfg[t] = dict(cfg[t])
        except:
            pass

    for d in ['scriptdirs', 'appdirs']:
        if d in cfg:
            scriptdirs = []
            for i in cfg[d]:
                if i[1] == '' or Path(i[1]).is_dir():
                    scriptdirs.append(tuple(i))
                elif i[0] == '' or Path(i[0]).is_dir():
                    scriptdirs.append((i[1], i[0]))
            cfg[d] = scriptdirs

    # Rename settings
    for old, new in [
        ('history', 'gui/scripthistory'),
        ('gui/history', 'gui/scripthistory'),
        ('raiseapploadexc', 'showapploaderrors'),
        ('webgl/xtkscript', 'webgl/script'),
    ]:
        if old in cfg:
            if new not in cfg:
                cfg[new] = cfg[old]
            del cfg[old]

    # Delete settings
    for key in [
        'input/timeout',
        'filterwarnings',
        'render/ambient',
        'render/diffuse',
        'render/specular',
        'render/emission',
        'render/material',
        'canvas/propcolors',
        'Save changes',
        'canvas/bgmode',
        'canvas/bgcolor2',
        '_save_',
    ]:
        if key in cfg:
            print("DELETING CONFIG VARIABLE %s" % key)
            del cfg[key]


def run_pytest(modules):
    """Run the pytests for the specified pyFormex modules.

    Parameters:

    - `modules`: a list of pyFormex modules in Python notation,
      relative to the pyFormex package. If an empty list is supplied,
      all available pytests will be run.

    Test modules are stored under the path `pf.cfg['testdir']`, with the
    same hierarchy as the pyFormex source modules, and are named
    `test_MODULE.py`, where MODULE is the corresponding source module.

    """
    try:
        import pytest
    except:
        print("Can not import pytest")
        pass
    testpath = pf.cfg['testdir']
    print("TESTPATH={}".format(testpath))
    args = ['--maxfail', '10']
    if not modules:
        pytest.main(args + [testpath])
    else:
        print("Running pytests for modules %s" % modules)
        for m in modules:
            path = Path(*m.split('.'))
            path = testpath / path.with_name("test_{}.py".format(path.name))
            if path.exists():
                pytest.main(args + [path])
            else:
                print("No such test module: %s" % path)


def list_modules(pkgs=['all']):
    """Return the list of pure Python modules in a pyFormex subpackage.

    Parameters:

    - `pkgs`: a list of pyFormex subpackage names. The subpackage name is a
      subdirectory of the main pyformex package directory.
      Two special package names are recognized:

      - 'core': returns the modules in the top level pyFormex package
      - 'all': returns all pyFormex modules

      An empty list is interpreted as ['all'].
    """
    modules = []
    if pkgs == []:
        pkgs = ['all']
    for subpkg in pkgs:
        modules.extend(utils.moduleList(subpkg))
    return modules


def doctest_module(module):
    """Run the doctests in a module's docstrings.

    module is a pyFormex module dotted path. The leading 'pyformex.'
    may be omitted.
    """
    import doctest
    import warnings
    # Note that a non-empty fromlist is needed to make the
    # __import__ function always return the imported module
    # even if a dotted path is specified
    if not module.startswith('pyformex.'):
        module = 'pyformex.' + module
    mod = __import__(module, fromlist=[None])

    with warnings.catch_warnings():
        warnings.simplefilter("ignore")
        return doctest.testmod(mod, optionflags=doctest.NORMALIZE_WHITESPACE)


def run_doctest(modules):
    """Run the doctests for the specified pyFormex modules.

    Doctests are tests embedded in the docstrings of the Python source.

    Parameters:

    - `modules`: a list of pyFormex modules in Python notation,
      relative to the pyFormex package. If an empty list is supplied,
      all doctests in all pyFormex modules will be run.

    To allow consistent output of floats independent of machine precision,
    numpy's floating point print precision is set to two decimals.
    """
    import numpy
    if utils.checkVersion('numpy', '1.14') < 0:
        # does not have the legacy argument
        numpy.set_printoptions(precision=2, suppress=True)
    else:
        numpy.set_printoptions(precision=2, suppress=True, legacy='1.13')
    from pyformex import arraytools
    numpy.set_string_function(arraytools.array2str)
    if not modules:
        modules = ['all']
    for mod in modules:
        if mod in ['core', 'all'] or mod.endswith('.'):
            todo = utils.moduleList(mod.replace('.', ''))
        else:
            todo = [mod]
        for m in todo:
            try:
                result = doctest_module(m)
            except Exception as e:
                result = "FAIL"
                pf.debug(e, pf.DEBUG.DOCTEST)

            print("Module {}: {}".format(m, result))


def migrateUserConfig():
    """Migrate the user preferences in $HOME/.config

    Conversion of old style has been abandoned.
    Currently this doesn't do anything.
    """
    pass


def filterWarnings():
    pf.debug(
        "Current warning filters: %s" % pf.cfg['warnings/filters'],
        pf.DEBUG.WARNING
    )
    try:
        for w in pf.cfg['warnings/filters']:
            utils.filterWarning(*w)
    except:
        pf.debug(
            "Error while processing warning filters: %s" %
            pf.cfg['warnings/filters'], pf.DEBUG.WARNING
        )


def activateWarningFilters():
    """Activate the warning filters

    """
    from pyformex import messages
    filterWarnings()

    def _format_warning(message, category, filename, lineno, line=None):
        """Replace the default warnings.formatwarning

        This allows the warnings being called using a simple mnemonic
        string. The full message is then found from the message module.
        """
        message = messages.getMessage(message)
        message = """..

pyFormex Warning
================
%s

`%s called from:` %s `line:` %s
""" % (message, category.__name__, filename, lineno)
        if line:
            message += "%s\n" % line
        return message

    import warnings
    #if pf.cfg['warnings/deprec']:
    # activate DeprecationWarning (since 2.7 default is ignore)
    #warnings.simplefilter('default', DeprecationWarning)
    #TODO: DeprecationWarning should be replaced with FutureWarning
    #pass

    if pf.cfg['warnings/nice']:
        warnings.formatwarning = _format_warning


# End
