import gtk
import gtk.glade
import gobject
import datetime
import time

from gui import *
from newtask import NewTask
from about import About

# import glade path from config module
from config import GLADE_FILE_PATH


# The main window's class.
class MainWin:
    def __init__ (self):        
        # The GtkTreePath of the last highlighted inactive task.
        # It's used to switch between the inactive/highlighted colors
        # in the task list.
        self.lastPath = None

        # We load the widgets and the callbacks from the glade file.
        self.widgetTree = gtk.glade.XML (GLADE_FILE_PATH + "/mainwin.glade")

        # It's the base widget.
        self.win = self.widgetTree.get_widget ("mainwin")

        # Get normal and insensitive state color from Gtk theme
        self.normalColor = self.getThemeColor(gtk.STATE_NORMAL)
        self.insensitiveColor = self.getThemeColor(gtk.STATE_INSENSITIVE)
        self.selectedColor = self.getThemeColor(gtk.STATE_SELECTED)

        # Initialize a ScheduleListStore object
        self.listStore=SchedulerListStore(self.normalColor,
                                         self.insensitiveColor)
        
        # and set it as model of our main Gtk::TreeView
        self.taskList = self.widgetTree.get_widget("tasklist")
        self.taskList.set_model(self.listStore)
        
        # Disable multiselect on TreeView
        self.taskList.get_selection().set_mode(gtk.SELECTION_SINGLE)
        
        # initialize a cell renderer
        renderer = CellRendererText()
        
        # and finally set the visible column
        column = gtk.TreeViewColumn("Task", renderer, text=1, foreground=3)
        self.taskList.append_column(column)
        column = gtk.TreeViewColumn("Next run", renderer, text=2, foreground=3)
        self.taskList.append_column(column)

        # autoconnect signals to callbacks
    	dict = {}
    	for key in dir(self.__class__):
    	    dict[key] = getattr(self, key)
        self.widgetTree.signal_autoconnect (dict)

        # connect some specific signals
        selection = self.taskList.get_selection()
        selection.connect("changed",self.on_tasklist_selection_changed)
        
        # update the GUI items
        self.updateItemsSensitivity()
    
    ###############################################
    #                                             #
    #            SIGNALS CALLBACKS                #
    #                                             #
    ###############################################
    def on_mainwin_delete_event (self, obj, event):
        return self.quit ()
    
    def on_mainwin_destroy_event (self, obj, event):
        return self.quit ()
    
    def on_tasklist_button_press_event (self, obj, event):
        # If the right-mouse button is pressed, show popup menu
        if (event.button == 3):
            self.showPopupMenu(event)

    def on_tasklist_selection_changed(self,obj):
        self.updateItemsSensitivity()
    
    def on_mainmenu_save_activate (self, obj):
        self.save ()
    
    def on_mainmenu_quit_activate (self, obj):
        return self.quit ()
    
    def on_mainmenu_new_activate (self, obj):
        self.createTask ()
    
    def on_mainmenu_properties_activate (self, obj):
        self.editTask ()
    
    def on_mainmenu_activate_activate (self, obj):
        self.setSchedulerActiveStatus(True)
    
    def on_mainmenu_deactivate_activate (self, obj):
        self.setSchedulerActiveStatus(False)
    
    def on_mainmenu_delete_activate (self, obj):
        self.deleteScheduler ()
    
    def on_mainmenu_about_activate (self, obj):
        self.about ()

    def on_popup_properties_activate (self, obj):
        self.editTask ()
    
    def on_popup_activate_activate (self, obj):
        self.setSchedulerActiveStatus(True)
    
    def on_popup_deactivate_activate (self, obj):
        self.setSchedulerActiveStatus(False)
    
    def on_popup_delete_activate (self, obj):
        self.deleteScheduler ()
    
    def on_toolbar_new_clicked (self, obj):
        self.createTask ()
    
    def on_toolbar_properties_clicked (self, obj):
        self.editTask ()
    
    def on_toolbar_activate_clicked (self, obj):
        self.setSchedulerActiveStatus(True)
    
    def on_toolbar_deactivate_clicked (self, obj):
        self.setSchedulerActiveStatus(False)
    
    def on_toolbar_delete_clicked (self, obj):
        self.deleteScheduler()
    
    def on_tasklist_row_activated (self, obj, path, column):
        self.editTask ()

    def on_confirm_delete_response (self, obj, response):
        # If deletion is confirmed, delete the task.
        if (response == -8):
            self.deleteScheduler(True)
        # Free the dialog.
        obj.destroy ()

    def on_newtask_validate (self,obj,scheduler):
        # store scheduler sent as parameters in ourself
        self.listStore.addScheduler(scheduler)

    ###############################################
    #                                             #
    #             CLASS METHODS                   #
    #                                             #
    ###############################################

    # Gets the colors from the GTK theme
    def getThemeColor (self, num):
        if (num < 5):
            redhex = hex (self.win.rc_get_style ().fg [num].red)
            greenhex = hex (self.win.rc_get_style ().fg [num].green)
            bluehex = hex (self.win.rc_get_style ().fg [num].blue)
        else:
            redhex = hex (self.win.rc_get_style ().bg [num - 5].red)
            greenhex = hex (self.win.rc_get_style ().bg [num - 5].green)
            bluehex = hex (self.win.rc_get_style ().bg [num - 5].blue)
        redstr = redhex.split ('0x')[1]
        greenstr = greenhex.split ('0x')[1]
        bluestr = bluehex.split ('0x') [1]
        
        return "#" + redstr + greenstr + bluestr

    # Show a confirmation dialog before deleting the task.
    def deleteScheduler (self,confirm = False):
        if (confirm):
            # deletion is confirmed, delete Scheduler
            model,iter = self.taskList.get_selection().get_selected()
            self.listStore.delScheduler (iter)
        else:
            # Load a delete confirmation dialog window
            xml = gtk.glade.XML (GLADE_FILE_PATH + "/confirm_delete.glade")
            # autoconnect signals
            dict = {}
            for key in dir(self.__class__):
                dict[key] = getattr(self, key)
            xml.signal_autoconnect (dict)

            # get the dialog window widget
            confirm = xml.get_widget ("confirm_delete")
   
            # Center the dialog on the main window.
            confirm.set_transient_for (self.win)
            confirm.set_position (gtk.WIN_POS_CENTER_ON_PARENT)
    
            # Display the dialog.
            confirm.show ()
        return

    def showPopupMenu(self,event):
        self.widgetTree.get_widget ("popup").popup (None, None, None,
                                                    event.button, event.time)

    def updateItemsSensitivity(self):
        itemDict = {
            "mainmenu_properties": 1,
            "mainmenu_activate"  : 1,
            "mainmenu_deactivate": 1,
            "mainmenu_delete"    : 1,
            "toolbar_new"        : 1,
            "toolbar_properties" : 1,
            "toolbar_activate"   : 1,
            "toolbar_deactivate" : 1,
            "toolbar_delete"     : 1,
            "popup_properties"   : 1,
            "popup_activate"     : 1,
            "popup_deactivate"   : 1,
            "popup_delete"       : 1
            }
        # get iter of selected row
        model,iter = self.taskList.get_selection().get_selected()
        # check if there is a selected row
        if(not iter):
            itemDict["mainmenu_properties"] = 0
            itemDict["mainmenu_activate"]   = 0
            itemDict["mainmenu_deactivate"] = 0
            itemDict["mainmenu_delete"]     = 0
            itemDict["toolbar_properties"] = 0
            itemDict["toolbar_activate"]   = 0
            itemDict["toolbar_deactivate"] = 0
            itemDict["toolbar_delete"]     = 0

        else:
            # check if the selected Scheduler is active
            # NOTE: Not implemented at the moment, active is alwas True
            active = model.get_value(iter,0).isActive()
            itemDict["mainmenu_activate"]   = not active
            itemDict["mainmenu_deactivate"] = active
            itemDict["toolbar_activate"]    = not active
            itemDict["toolbar_deactivate"]  = active
            itemDict["popup_activate"]      = not active
            itemDict["popup_deactivate"]    = active

        # set sensitivity on items
        for key in itemDict:
            self.widgetTree.get_widget(key).set_sensitive(itemDict[key])


    # Set the active status of the selected Scheduler
    def setSchedulerActiveStatus(self,status):
        # get selected row
        model,iter = self.taskList.get_selection().get_selected()
        # do something only if there is a selected row
        if(iter):
            # set status to scheduler
            model.get_value(iter,0).setActive(status)
            # change color of text
            if(status):
                model.set_value(iter,3,self.normalColor)
            else:
                model.set_value(iter,3,self.insensitiveColor)
        # update sensitivity of menu items
        self.updateItemsSensitivity()
        # and set the modified status to True

    # Writes the current configuration.
    def save (self):
        self.listStore.saveSchedulers()
    
    # Unreferences the main window's widgets, and exits.
    def quit (self):
        # is there some modified entries
        if (self.taskList.get_model().isModified()):
            # yes, ask for saving before exiting
            dialog = gtk.Dialog("Quit without saving?",
                     self.win,
                     gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT,
                     (gtk.STOCK_NO, gtk.RESPONSE_NO,
                      gtk.STOCK_YES, gtk.RESPONSE_YES))
            label = gtk.Label('You have some unsaved data\nDo you really want to quit?')
            dialog.vbox.pack_start(label)
            dialog.show_all()
            if(dialog.run() == gtk.RESPONSE_NO):
                dialog.destroy()
                return True
        gtk.main_quit()
        
    # Launches the "about" dialog.
    def about (self):
        aboutObject = About (self.win)



#######################################
#     TO BE REWRITTEN  OR DELETED     #
#######################################

    
    # Launches the new task assistant and refreshes the task list
    # if a new task was created.
    def createTask (self):
        assistantObject = NewTask (self)
        assistantObject.connect("newtask-validate",self.on_newtask_validate)


    # Launches the properties dialog
    def editTask (self):
        model,iter = self.taskList.get_selection().get_selected()
        currentScheduler = model.get_value (iter, 0)
        self.editor = SchedulerEditor(currentScheduler)
        self.editor.show(self.win)
    
    
